// --------------------------------------------------------------------
// CFurPush.cpp
// Whatis:	Class for sending file to fur-pop service
// Authors:	Esko 'Varpu' Ilola	EIL
// History:	EIL	20-APR-2003		Created	this source
// --------------------------------------------------------------------
#include	"CFurPush.hxx"
#include	"CError.hxx"

// --------------------------------------------------------------------
// public:		Constructor
// --------------------------------------------------------------------
CFurPush::CFurPush	(	const char *	aFurPopAddr,		// IP address
						word_t			aFurPopPort,		// Port
						const char *	aFile ) {			// File to send
	CFurPush::Cleanup();

	itsAddr		= ::my_private_strdup( aFurPopAddr );
	itsPort		= aFurPopPort;
	itsSock		= new CTcpSocket( (socket_type_t)socket_type_stream );

	// Grab file details
	CFurPush::SetFile( aFile );
}

// --------------------------------------------------------------------
// public:		Destructor
// --------------------------------------------------------------------
CFurPush::~CFurPush	( ) {
	CFurPush::Free();
}

// --------------------------------------------------------------------
// public:		Protocol driver - request the state
// --------------------------------------------------------------------
fur_push_state_t	CFurPush::State		( void ) {
	switch	( itsState ) {
		case	fur_push_state_connect:		CFurPush::Connect();	break;
		case	fur_push_state_request:		CFurPush::Request();	break;
		case	fur_push_state_response:	CFurPush::Response( fur_push_state_transini );	break;
		case	fur_push_state_transini:	CFurPush::Transini();	break;
		case	fur_push_state_transfer:	CFurPush::Transfer();	break;
		case	fur_push_state_transack:	CFurPush::Response( fur_push_state_finalize );	break;
		case	fur_push_state_finalize:	CFurPush::Finalize();	break;
		case	fur_push_state_serclose:	CFurPush::Serclose();	break;
		case	fur_push_state_ready:		break;
		case	fur_push_state_error:		break;
	}
	return	itsState;
}

// --------------------------------------------------------------------
// public:		Requesting instance data
// --------------------------------------------------------------------
dword_t			CFurPush::SendStart	( void ) const { return itsSendStart; }
dword_t			CFurPush::SendOffset( void ) const { return itsSendOffset; }
dword_t			CFurPush::SendEnd	( void ) const { return itsSendEnd; }
const char *	CFurPush::Message	( void ) const { return itsMessage  ? itsMessage  : ""; }
const char *	CFurPush::Addr		( void ) const { return itsAddr     ? itsAddr     : ""; }
word_t			CFurPush::Port		( void ) const { return itsPort; }
const char *	CFurPush::LocaFile	( void ) const { return itsLocaFile ? itsLocaFile : ""; }
const char *	CFurPush::RemoFile	( void ) const { return itsRemoFile ? itsRemoFile : ""; }

// --------------------------------------------------------------------
// private:		Calculate checksum from the file
// --------------------------------------------------------------------
dword_t			CFurPush::Checksum	( dword_t aLen )	const {
	dword_t	csum = 0;
    dword_t	offs;
	int		rchr;
	int		i;
	byte_t	buff	[2048];

	::fseek( itsStream, 0, SEEK_SET );

	offs = 0;
	while	( offs < aLen ) {
		if	( aLen - offs < 2048 ) {
	       	rchr = ::fgetc( itsStream );
    	    csum = csum + (dword_t)rchr;
			offs++;
		}
		else {
			::fread( buff, 1, 2048, itsStream );
			for	( i = 0; i < 2048; i++ ) {
	    	    csum = csum + (dword_t)buff[i];
			}
			offs = offs + 2048;
		}
	}

	::fseek( itsStream, 0, SEEK_SET );

	return	csum;
}

// --------------------------------------------------------------------
// private:		
// --------------------------------------------------------------------
void			CFurPush::SetFile	( const char * aFile ) {
	const char *	p;
	try	{
		// Set up local file and open my stream
		itsLocaFile	= ::my_strfix( ::my_private_strdup( aFile ) );
		if	( ::stat( LocaFile(), &itsStat ) ) {
			throw	CError( LocaFile(), ::strerror( errno ) );
		}
		itsStream	= ::fopen( LocaFile(), "rb" );
		if	( ! itsStream ) {
			throw	CError( LocaFile(), ::strerror( errno ) );
		}

		// Check out for valid file name without any quirks
#ifdef	WIN32
		p = ::strrchr( LocaFile(), '\\' );
#else
		p = ::strrchr( LocaFile(), '/' );
#endif
		itsRemoFile = ::my_strlower( ::my_private_strdup( p ? p + 1 : LocaFile() ) );
		for	( p = RemoFile(); *p; p++ ) {
        	if	( strchr( "\":$%&/=+?\\@<>|", *p ) ) {
				throw	CError( LocaFile(), "Invalid characters in file name" );
            }
		}
		p = ::strstr( RemoFile(), ".zip" );
		if	( ! p ) {
			throw	CError( LocaFile(), "Propably not a zip file" );
		}
		if	( ::strlen( p ) != 4 ) {
			throw	CError( LocaFile(), "Propably not a zip file" );
		}
		itsTimeout	= ::time( NULL ) + 10;
        itsSendEnd	= (dword_t)itsStat.st_size;
	}

	catch	( CError e ) {
		itsMessage	= ::my_private_strdup( e.Error() );
		itsState	= fur_push_state_error;
	}

	catch	( ... ) {
		itsMessage	= ::my_private_strdup( "Unknown error" );
		itsState	= fur_push_state_error;
	}
}

// --------------------------------------------------------------------
// private:		Try toconnect to the service - 10 seconds timeout
// --------------------------------------------------------------------
void			CFurPush::Connect	( void ) {
	try	{
		if	( itsTimeout < ::time( NULL ) ) {
			throw	CError( itsAddr, "No response" );
		}
		if	( itsSock->Connect( itsAddr, itsPort ) ) {
			itsState = fur_push_state_request;
		}
	}

	catch	( CError e ) {
		itsMessage	= ::my_private_strdup( e.Error() );
		itsState	= fur_push_state_error;
	}

	catch	( ... ) {
		itsMessage	= ::my_private_strdup( "Unknown error" );
		itsState	= fur_push_state_error;
	}
}

// --------------------------------------------------------------------
// private:		Send the request to the service
// --------------------------------------------------------------------
void			CFurPush::Request	( void ) {
	try	{
		// Sending the negotiation package to server
	    // The format of the package is as follows:
	    // filename,filesize,y,m,d,H,M,S<cr>(0x0d)
		struct tm *	mytm = ::gmtime( &(itsStat.st_mtime) );

	    itsSock->Write( RemoFile() );
	    itsSock->Write( "," );
	    itsSock->Write( (long)itsStat.st_size );
		itsSock->Write( "," );
	    itsSock->Write( (long)(mytm->tm_year + 1900) );
		itsSock->Write( "," );
	    itsSock->Write( (long)(mytm->tm_mon + 1) );
		itsSock->Write( "," );
	    itsSock->Write( (long)(mytm->tm_mday) );
		itsSock->Write( "," );
	    itsSock->Write( (long)(mytm->tm_hour) );
		itsSock->Write( "," );
	    itsSock->Write( (long)(mytm->tm_min) );
		itsSock->Write( "," );
	    itsSock->Write( (long)(mytm->tm_sec) );
		itsSock->Write( "\x0d" );
	    itsSock->Flush();

	    itsAckOffs	= 0;
		itsState	= fur_push_state_response;
		itsTimeout	= 10 + ::time( NULL );
	}

	catch	( CError e ) {
		itsMessage	= ::my_private_strdup( e.Error() );
		itsState	= fur_push_state_error;
	}

	catch	( ... ) {
		itsMessage	= ::my_private_strdup( "Unknown error" );
		itsState	= fur_push_state_error;
	}
}

// --------------------------------------------------------------------
// private:		Read the response from the service
// --------------------------------------------------------------------
void			CFurPush::Response	( fur_push_state_t aNextState ) {
	int		chr;
	try	{
		if	( itsTimeout < ::time( NULL ) ) {
			throw	CError( itsAddr, "No response" );
		}

		do	{
	    	chr = itsSock->Read();

		    switch	( chr ) {
        		case    SOCKET_READ_IDLE:
				return;

		    	case	SOCKET_READ_ERROR:
				throw	CError( itsAddr, "Read error" );

	        	case    SOCKET_READ_DISCONNECT:	// We got early disconnect
            	throw	CError( itsAddr, "Disconnected" );
			}

	        itsAck[itsAckOffs] = (char)chr;
    	    if	( chr != 0x0d )	itsAckOffs++;
        	else				itsAck[itsAckOffs] = 0;

	        if	( itsAckOffs > sizeof( itsAck ) - 2 ) {
	        	throw	CError( "Invalid response" );
	        }
	    }	while	( chr != 0x0d );

		itsState	= aNextState;
	}

	catch	( CError e ) {
		itsMessage	= ::my_private_strdup( e.Error() );
		itsState	= fur_push_state_error;
	}

	catch	( ... ) {
		itsMessage	= ::my_private_strdup( "Unknown error" );
		itsState	= fur_push_state_error;
	}
}

// --------------------------------------------------------------------
// private:		Initialize the transfer using the response we got
// --------------------------------------------------------------------
void			CFurPush::Transini	( void ) {
    char *	p;
	dword_t	hf, hc;

	try	{
	    switch	( itsAck[0] ) {
	    	case	0:		// File does not exist - start from scratch
				::fseek( itsStream, 0L, SEEK_SET );
				itsSendStart	= 0;
				itsSendOffset	= 0;
				itsSendEnd		= (dword_t)itsStat.st_size;
				itsState		= fur_push_state_transfer;
		        break;

	        case	'F':    // File exists - length and checksum returned
							// F<len>,<csum><cr>
				p = ::strchr( itsAck, ',' );
		        if	( ! p ) {
					throw	CError( itsAddr, "Invalid response" );
	    	    }
	
		        // Get the checksum and the file length
		        *p = 0;
		        hf = (dword_t)::atol( itsAck + 1 );
		        hc = (dword_t)::atol( p + 1 );

		        // Is it already longer than my version ?
		        if	( hf > (dword_t)(itsStat.st_size) ) {
			        throw	CError( LocaFile(), "Different file" );
		        }

		        // Is the checksum same so far ?
		        if	( hc != CFurPush::Checksum( hf ) ) {
			        throw	CError( LocaFile(), "Different file" );
		        }

		        // Start transferring
				::fseek( itsStream, hf, SEEK_SET );
				itsSendOffset	= hf;
				itsState		= fur_push_state_transfer;
		        break;

	        case	'X':	// Another transfer for this file is under way
    		    throw	CError( itsAddr, itsAck + 1 );

	        default:		// Unknown response
    		    throw	CError( itsAddr, "Invalid response" );
		}
	}

	catch	( CError e ) {
		itsMessage	= ::my_private_strdup( e.Error() );
		itsState	= fur_push_state_error;
	}

	catch	( ... ) {
		itsMessage	= ::my_private_strdup( "Unknown error" );
		itsState	= fur_push_state_error;
	}
}

// --------------------------------------------------------------------
// private:		Transfer the file data to the service
// --------------------------------------------------------------------
void			CFurPush::Transfer	( void ) {
	try {
		char	sendbuf	[1024];
	    dword_t	sendlen = (dword_t)itsStat.st_size - itsSendOffset;

	    // Maximum send length is 1024 bytes
	    if	( sendlen > 1024 )	sendlen = 1024;

	    // Read data into the buffer and send it out
	    if	( sendlen > 0 ) {
			::fread( sendbuf, sendlen, 1, itsStream );
			itsSock->Write( sendbuf, (int)sendlen );
			itsSock->Flush();
		}

		// Update current offset
		itsSendOffset = itsSendOffset + sendlen;

		// Last check if we have done it all
		if	( sendlen == 0 ) {
			itsAckOffs	= 0;
			itsState	= fur_push_state_transack;
			itsTimeout	= 20 + time( NULL );
		}
	}

	catch	( CError e ) {
		itsMessage	= ::my_private_strdup( e.Error() );
		itsState	= fur_push_state_error;
	}

	catch	( ... ) {
		itsMessage	= ::my_private_strdup( "Unknown error" );
		itsState	= fur_push_state_error;
	}
}

// --------------------------------------------------------------------
// private:		Checksum from the service
// --------------------------------------------------------------------
void			CFurPush::Finalize	( void ) {
	try {
		if	( (dword_t)::atol( itsAck ) != CFurPush::Checksum( itsSendOffset ) ) {
			itsSock->Write( "0\x0d" );
            itsSock->Flush();
	        throw	CError( LocaFile(), "Data was corrupt" );
        }
		itsSock->Write( "1\x0d" );
		itsSock->Flush();
		itsState	= fur_push_state_serclose;
		itsTimeout	= 5 + time( NULL );
	}

	catch	( CError e ) {
		itsMessage	= ::my_private_strdup( e.Error() );
		itsState	= fur_push_state_error;
	}

	catch	( ... ) {
		itsMessage	= ::my_private_strdup( "Unknown error" );
		itsState	= fur_push_state_error;
	}
}

// --------------------------------------------------------------------
// private:		Wait for the service to close the connection
// --------------------------------------------------------------------
void			CFurPush::Serclose	( void ) {

	try	{
		if	( itsTimeout < ::time( NULL ) ) {
			throw	CError( itsAddr, "No response" );
		}

		if	( itsSock->Read() == SOCKET_READ_DISCONNECT ) {
			delete	itsSock;		itsSock		= NULL;
			::fclose( itsStream );	itsStream	= NULL;

			itsState	= fur_push_state_ready;
		}
	}

	catch	( CError e ) {
		itsMessage	= ::my_private_strdup( e.Error() );
		itsState	= fur_push_state_error;
	}

	catch	( ... ) {
		itsMessage	= ::my_private_strdup( "Unknown error" );
		itsState	= fur_push_state_error;
	}
}

// --------------------------------------------------------------------
// private:		
// --------------------------------------------------------------------
void			CFurPush::Free		( void ) {
	if	( itsAddr )		delete [] itsAddr;
	if	( itsLocaFile )	delete [] itsLocaFile;
	if	( itsRemoFile )	delete [] itsRemoFile;
	if	( itsMessage )	delete [] itsMessage;
	if	( itsSock )		delete itsSock;
	if	( itsStream )	::fclose( itsStream );
	CFurPush::Cleanup();
}

// --------------------------------------------------------------------
// private:		
// --------------------------------------------------------------------
void			CFurPush::Cleanup	( void ) {
	itsState		= fur_push_state_connect;
	itsAddr			= NULL;
	itsPort			= 0;
	itsLocaFile		= NULL;
	itsRemoFile		= NULL;
	itsMessage		= NULL;
	itsSendStart	= 0;
	itsSendOffset	= 0;
	itsSendEnd		= 0;
	itsSock			= NULL;
	::memset( &itsStat, 0, sizeof( itsStat ) );
	itsStream		= NULL;
	itsTimeout		= ::time( NULL );
	::memset( itsAck, 0, sizeof( itsAck ) );
	itsAckOffs		= 0;
}

// --------------------------------------------------------------------
// EOF:	CFurPush.cxx
// --------------------------------------------------------------------
