// --------------------------------------------------------------------
// CIdentifyFile.cxx
// Whatis:	Class for identifying a file
// Authors:	Esko 'Varpu' Ilola	EIL
// History:	EIL	02-JUL-2002		Created	this source
// --------------------------------------------------------------------
#include	"CError.hxx"
#include	"CTableRule.hxx"
#include	"CUnLevelInfo.hxx"
#include	"CUnUmod.hxx"
#include	"CUnUtils.hxx"
#include	"CIdentifyFile.hxx"
#include	"CAuthor.hxx"

// --------------------------------------------------------------------
// local:		The rulelist is loaded when needed the first time
// --------------------------------------------------------------------
static	data_rule_tl	localrules;

// --------------------------------------------------------------------
// public:		Constructor
// --------------------------------------------------------------------
CIdentifyFile::CIdentifyFile ( const char * aFileName, const char * aFileCsum ) {
	CIdentifyFile::Cleanup();

	// ----------------------------------------------------------------
	// It must be a regular file
	// ----------------------------------------------------------------
	if	( ::stat( aFileName, &itsStat ) ) {
		throw CError( aFileName, ::strerror( errno ) );
	}
#ifdef	WIN32
	if	( ( itsStat.st_mode & _S_IFREG ) == 0 ) {
#else
	if	( ( itsStat.st_mode & __S_IFMT ) != __S_IFREG ) {
#endif
		throw	CError( aFileName, "Not a regular file" );
	}

	// ----------------------------------------------------------------
	// Set up file name etc...
	// ----------------------------------------------------------------
	::my_strfit( itsFileName, sizeof( itsFileName ), aFileName );
	::my_strfit( itsPath, sizeof( itsPath ), aFileName );
	if	( ::strrchr( itsPath, '/' ) ) {
		::strcpy( itsName, ::strrchr( itsPath, '/' ) + 1 );
		*(::strrchr( itsPath, '/' ) + 1) = 0;
	}
	else {
		::memset( itsPath, 0, sizeof( itsPath ) );
		::strcpy( itsName, itsFileName );
	}

	if	( ::strrchr( itsName, '.' ) ) {
		::strcpy( itsSuff, ::strrchr( itsName, '.' ) );
		*::strrchr( itsName, '.' ) = 0;
	}
	else {
		::memset( itsSuff, 0, sizeof( itsSuff ) );
	}

	::my_strfix( ::my_strlower( itsSuff ) );

	// ----------------------------------------------------------------
	// Try out loading the UMOD and UNR types
	// ----------------------------------------------------------------
	CIdentifyFile::LoadUMOD();
	CIdentifyFile::LoadUNR();
	itsExe	= CIdentifyFile::TryEXE();
	itsU1	= CIdentifyFile::TryU1();
	itsUT	= CIdentifyFile::TryUT();
	itsAAO	= CIdentifyFile::TryAAO();
	itsUT2k3= CIdentifyFile::TryUT2k3();

	// ----------------------------------------------------------------
	// Last but not least, sacan the rules
	// ----------------------------------------------------------------
	CIdentifyFile::ScanRules();

	// ----------------------------------------------------------------
	// After identification, destroy out cache
	// ----------------------------------------------------------------
	if	( itsUnUtils )	delete itsUnUtils;
	itsUnUtils	= NULL;

	// ----------------------------------------------------------------
	// Last, before actually leaving, we try to solve the author
	// ----------------------------------------------------------------
	CAuthor		myauthor( aFileName, aFileCsum, true );
	if	( myauthor.LongName() != NULL ) {
		::my_strfit( itsAuthor, sizeof( itsAuthor ), myauthor.LongName() );
		itsAuthorId = myauthor.AuthorId();
	}
}

// --------------------------------------------------------------------
// public:		Destructor
// --------------------------------------------------------------------
CIdentifyFile::~CIdentifyFile	() {
	CIdentifyFile::Free();
}

// --------------------------------------------------------------------
// private:		Try to load the UMOD file type
// --------------------------------------------------------------------
void	CIdentifyFile::LoadUMOD	( void ) {
	try {
		CUnUmod	umod( itsFileName );
		itsUmod = true;
	}
	catch ( ... ) {
	}
}

// --------------------------------------------------------------------
// private:		Try to load the UNR file type
// --------------------------------------------------------------------
void	CIdentifyFile::LoadUNR	( void ) {
	try {
		itsUnUtils = new CUnUtils( itsFileName );
		itsPackageVersion	= itsUnUtils->UnFile().PackageVersion();
		itsLicenseeMode		= itsUnUtils->UnFile().LicenseeMode();
		if	( itsUnUtils->ActorCount( "LevelInfo" ) > 0 ) {
			try	{
				CUnLevelInfo	levelinfo(	itsUnUtils->UnFile(),
											itsUnUtils->UnNameTable(),
											itsUnUtils->UnExportTable(),
											itsUnUtils->UnImportTable() );
				if	( *(levelinfo.Title()) != 0 ) {
					::my_strfit( itsTitle, sizeof( itsTitle ), levelinfo.Title() );
				}
			}
			catch ( ... ) {
			}
		}
	}

	catch ( ... ) {
		itsUnUtils = NULL;
	}
}

// --------------------------------------------------------------------
// private:		Scan all the rules, stop if a match was found
// --------------------------------------------------------------------
void	CIdentifyFile::ScanRules( void ) {
	CMySqlConnect	db( "quest", "", "UTCMS" );
	CTableRule		myrule;
	CTableType		mytype;
	CMySqlWhere		w;
	data_rule_tl	myrulelist2;
	data_type_tl	mytypelist;
	data_rule_tli	myruleloop;

	if	( localrules.size() < 1 ) {
		w << "rule_name >= 'ify00000' and rule_name <= 'ify99999' order by rule_name";
		localrules = myrule.Select( db, w );
	}

	for	( myruleloop = localrules.begin(); myruleloop != localrules.end(); myruleloop++ ) {
		if	( CIdentifyFile::TryRule( (*myruleloop).rule_rule ) ) {
			w = "";
			w << "type_idnt=" << (*myruleloop).rule_type;
			mytypelist = mytype.Select( db, w );

			if	( mytypelist.size() < 1 ) {
				throw CError( "Type not found", w.Where() );
			}

			// --------------------------------------------------------
			// This is the type
			// --------------------------------------------------------
			itsType = *(mytypelist.begin());

			// --------------------------------------------------------
			// Category
			// --------------------------------------------------------
			::my_strfit( itsCtgr, sizeof( itsCtgr ), itsType.type_ctgr );

			// --------------------------------------------------------
			// Unpacking rule
			// --------------------------------------------------------
			w = "";
			w << "rule_type=" << (*myruleloop).rule_type << " and rule_name='unpacking'";
			myrulelist2 = myrule.Select( db, w );
			if	( myrulelist2.size() < 1 )	::strcpy( itsUnpack, "none" );
			else							::strcpy( itsUnpack, (*(myrulelist2.begin())).rule_rule );

			// --------------------------------------------------------
			// Primary and secondary type
			// --------------------------------------------------------
			w = "";
			w << "rule_type=" << (*myruleloop).rule_type << " and rule_name='primary'";
			myrulelist2 = myrule.Select( db, w );
			if	( myrulelist2.size() > 0 )	::my_strfit( itsTyp1, sizeof( itsTyp1 ), (*(myrulelist2.begin())).rule_rule );

			w = "";
			w << "rule_type=" << (*myruleloop).rule_type << " and rule_name='secondary'";
			myrulelist2 = myrule.Select( db, w );
			if	( myrulelist2.size() > 0 )	::my_strfit( itsTyp2, sizeof( itsTyp2 ), (*(myrulelist2.begin())).rule_rule );

			break;
		}
	}
}

// --------------------------------------------------------------------
// private:		Try one rule
// --------------------------------------------------------------------
bool	CIdentifyFile::TryRule	( const char * aRule ) {
	char	token	[4][256];
	int		ti, ii;
	bool	result;

	result = true;
	while	( ( *aRule ) && ( result ) ) {

		for	( ti = 0; ti < 4; ti++ ) token[ti][0] = 0;
		
		ti = 0;

		while	( ( *aRule ) && ( *aRule != '+' ) ) {
			ii = 0;

			if	( ti > 3 ) throw CError( "Invalid rule argument (too many)" );

			while	( ( *aRule ) && ( *aRule != ' ' ) && ( *aRule != '+' ) ) {
				token[ti][ii++] = *(aRule++);
				if	( ii > 254 ) throw CError( "Invalid rule argument (too long)" );
			}

			token[ti++][ii] = 0;

			while ( *aRule == ' ' )	aRule++;
		}
		
		if		( ! ::strcmp( token[0], "suff" ) )			result =   CIdentifyFile::TrySuff( token[1] );
		else if	( ! ::strcmp( token[0], "notsuff" ) )		result = ! CIdentifyFile::TrySuff( token[1] );
		else if	( ! ::strcmp( token[0], "pref" ) )			result =   CIdentifyFile::TryPref( token[1] );
		else if	( ! ::strcmp( token[0], "notpref" ) )		result = ! CIdentifyFile::TryPref( token[1] );
		else if	( ! ::strcmp( token[0], "file" ) )			result =   CIdentifyFile::TryFile( token[1] );
		else if	( ! ::strcmp( token[0], "notfile" ) )		result = ! CIdentifyFile::TryFile( token[1] );
		else if	( ! ::strcmp( token[0], "unreal" ) )		result = itsUnUtils != NULL;
		else if	( ! ::strcmp( token[0], "notunreal" ) )		result = itsUnUtils == NULL;
		else if	( ! ::strcmp( token[0], "umod" ) )			result =   itsUmod;
		else if	( ! ::strcmp( token[0], "notumod" ) )		result = ! itsUmod;
		else if	( ! ::strcmp( token[0], "actor" ) )			result = CIdentifyFile::TryActor( token[1], token[2], token[3] );
		else if	( ! ::strcmp( token[0], "sign" ) )			result = CIdentifyFile::TrySign( token[1] );
		else if	( ! ::strcmp( token[0], "texture" ) )		result = CIdentifyFile::TryTexture( token[1] );
		else if	( ! ::strcmp( token[0], "function" ) )		result =   CIdentifyFile::TryFunction( token[1] );
		else if	( ! ::strcmp( token[0], "nofunction" ) )	result = ! CIdentifyFile::TryFunction( token[1] );
		else if	( ! ::strcmp( token[0], "needs" ) )			result = CIdentifyFile::TryNeeds( token[1] );
		else if	( ! ::strcmp( token[0], "hasnot" ) )		result = CIdentifyFile::TryHasnot( token[1] );
		else if	( ! ::strcmp( token[0], "isu1" ) )			result =   itsU1;
		else if	( ! ::strcmp( token[0], "notu1" ) )			result = ! itsU1;
		else if	( ! ::strcmp( token[0], "isut" ) )			result =   itsUT;
		else if	( ! ::strcmp( token[0], "notut" ) )			result = ! itsUT;
		else if	( ! ::strcmp( token[0], "isut2k3" ) )		result =   itsUT2k3;
		else if	( ! ::strcmp( token[0], "notut2k3" ) )		result = ! itsUT2k3;
		else if	( ! ::strcmp( token[0], "isaao" ) )			result =   itsAAO;
		else if	( ! ::strcmp( token[0], "notaao" ) )		result = ! itsAAO;
		else if	( ! ::strcmp( token[0], "isexe" ) )			result =   itsExe;
		else if	( ! ::strcmp( token[0], "notexe" ) )		result = ! itsExe;
		else if	( ! ::strcmp( token[0], "any" ) )		break;
		else	throw CError( "Invalid token in rule", token[0] );

		if	( *aRule == '+' ) {
			aRule++;
			while ( *aRule == ' ' )	aRule++;
		}
	}
	return result;
}

static	const char *	utitems[] = {
	"ut_", "UT_", "to_", "TO_",
	NULL
};

// --------------------------------------------------------------------
// private:		Try for UT2k3 package
// --------------------------------------------------------------------
bool	CIdentifyFile::TryUT2k3	( void ) {
	if	( ! itsUnUtils )			return false;
	if	( itsPackageVersion < 119 )	return false;
	return true;
}

// --------------------------------------------------------------------
// private:		Try for AAO package
// --------------------------------------------------------------------
bool	CIdentifyFile::TryAAO	( void ) {
	if	( ! itsUnUtils )				return false;
	if	( itsPackageVersion >= 119 )	return false;
	if	( itsPackageVersion <  100 )	return false;
	return true;
}

// --------------------------------------------------------------------
// private:		Try for UT game level / package
// --------------------------------------------------------------------
bool	CIdentifyFile::TryUT	( void ) {
	if	( ! itsUnUtils )				return false;
	if	( itsPackageVersion >= 100 )	return false;
	if	( itsPackageVersion >= 69 )		return true;
	for ( dword_t n = 0; n < itsUnUtils->UnNameTable().Count(); n++ ) {
		for	( int i = 0; utitems[i]; i++ ) {
			if	( ! ::strncmp( itsUnUtils->UnNameTable().Name(n), utitems[i], ::strlen( utitems[i] ) ) ) {
				return true;
			}
		}
	}
	return CIdentifyFile::TryNeeds( "botpack" );
}

// --------------------------------------------------------------------
// private:		Try for Unreal game level
// --------------------------------------------------------------------
bool	CIdentifyFile::TryU1	( void ) {
	if	( ! itsUnUtils )			return false;
	if	( itsPackageVersion >= 69 )	return false;
	for ( dword_t n = 0; n < itsUnUtils->UnNameTable().Count(); n++ ) {
		for	( int i = 0; utitems[i]; i++ ) {
			if	( ! ::strncmp( itsUnUtils->UnNameTable().Name(n), utitems[i], ::strlen( utitems[i] ) ) ) {
				return false;
			}
		}
	}
	if	( CIdentifyFile::TryNeeds( "botpack" ) )	return false;
	return true;
}

// --------------------------------------------------------------------
// private:		Try for executable file
// --------------------------------------------------------------------
bool	CIdentifyFile::TryEXE	( void ) {
	bool	result= false;
	FILE *	mystm = ::fopen( itsFileName, "rb" );

	if	( mystm ) {
		char	mzbuf[2];

		if	( ::fread( mzbuf, 2, 1, mystm ) ) {
			if	( ( mzbuf[0] == 'M' ) && ( mzbuf[1] == 'Z' ) ) {
				result = true;
			}
		}

		::fclose( mystm );
	}
	return	result;
}

// --------------------------------------------------------------------
// private:		Try suffix matching
// --------------------------------------------------------------------
bool	CIdentifyFile::TrySuff	( const char * aSuff ) {
	return	::strcmp( aSuff, itsSuff ) == 0;
}

// --------------------------------------------------------------------
// private:		Try file name prefix matching
// --------------------------------------------------------------------
bool	CIdentifyFile::TryPref	( const char * aPref ) {
	char	myfile[101];
	
	::strcpy( myfile, itsName );
	myfile[ ::strlen( aPref ) ] = 0;
	::my_strlower( myfile );

	return	::strcmp( aPref, myfile ) == 0;
}

// --------------------------------------------------------------------
// private:		Try file name matching
// --------------------------------------------------------------------
bool	CIdentifyFile::TryFile	( const char * aFile ) {
	char	myfile[101];

	::strcpy( myfile, itsName );
	::strcat( myfile, itsSuff );
	::my_strlower( myfile );
	return	::strcmp( aFile, myfile ) == 0;
}

// --------------------------------------------------------------------
// private:		Try for Unreal actor presence
// --------------------------------------------------------------------
bool	CIdentifyFile::TryActor	( 	const char *	aActor,
									const char *	aA1,
									const char *	aA2 ) {
	dword_t		c	= 0;
	dword_t		v	= ::atol( aA2 );

	if	( itsUnUtils ) {
		if		( ! ::strcmp( aActor, "#v" ) )	c = itsPackageVersion;
		else if	( ! ::strcmp( aActor, "#l" ) )	c = itsLicenseeMode;
		else									c = itsUnUtils->ActorCount( aActor );
	}

	if		( ! ::strcmp( aA1, "==" ) )		return	c == v;
	else if	( ! ::strcmp( aA1, "<=" ) )		return	c <= v;
	else if	( ! ::strcmp( aA1, "<"  ) )		return	c <  v;
	else if	( ! ::strcmp( aA1, ">"  ) )		return	c >  v;
	else if	( ! ::strcmp( aA1, ">=" ) )		return	c >= v;
	else if	( ! ::strcmp( aA1, "!=" ) )		return	c != v;
	else	throw CError( "Boolean operator expected instead of", aA1 );

	return	false;
}

// --------------------------------------------------------------------
// private:		Try for presence of a texture
// --------------------------------------------------------------------
bool	CIdentifyFile::TryTexture	(	const char * aTexture ) {
	if	( itsUnUtils ) {
		if	( itsUnUtils->UnNameTable().Exist( aTexture ) ) {
			dword_t	exclass = itsUnUtils->UnNameTable().Find( "Texture" );
			dword_t	textnam = itsUnUtils->UnNameTable().Find( aTexture );
			dword_t	exloop;

			for ( exloop = 0; exloop < itsUnUtils->UnExportTable().Count(); exloop++ ) {
				CUnExport	exprt = itsUnUtils->UnExportTable().Export( exloop );
				CUnImport	imprt = itsUnUtils->UnImportTable().Import( NAMEIMPORT( exprt.Class() ) );

		        // ----------------------------------------------------
		        // Skip over things that are not textures
		        // ----------------------------------------------------
		        if  ( exprt.Super() != 0 )				continue;
		        if  ( exclass != imprt.ObjectName() )	continue;
				if	( textnam == exprt.ObjectName() )	return true;
			}
		}
	}
	return false;
}

// --------------------------------------------------------------------
// private:		Try for presence of a function
// --------------------------------------------------------------------
bool	CIdentifyFile::TryFunction	(	const char * aFunction ) {
	if	( itsUnUtils ) {
		if	( itsUnUtils->UnNameTable().Exist( aFunction ) ) {
			dword_t	exclass = itsUnUtils->UnNameTable().Find( "Function" );
			dword_t	textnam = itsUnUtils->UnNameTable().Find( aFunction );
			dword_t	exloop;

			for ( exloop = 0; exloop < itsUnUtils->UnExportTable().Count(); exloop++ ) {
				CUnExport	exprt = itsUnUtils->UnExportTable().Export( exloop );
				CUnImport	imprt = itsUnUtils->UnImportTable().Import( NAMEIMPORT( exprt.Class() ) );

		        // ----------------------------------------------------
		        // Skip over things that are not functions
		        // ----------------------------------------------------
//		        if  ( exprt.Super() != 0 )				continue;
		        if  ( exclass != imprt.ObjectName() )	continue;
				if	( textnam == exprt.ObjectName() )	return true;
			}
		}
	}
	return false;
}

// --------------------------------------------------------------------
// private:		Try if needs a package
// --------------------------------------------------------------------
bool	CIdentifyFile::TryNeeds	(	const char * aPackage ) {
	if	( itsUnUtils ) {
		dword_t	findindx[20];
		char	findname[64];
		word_t	ix, w;

		// Extract the comma separated names and put them into index array
		ix = 0;
		while	( ix < 20 ) {
			findindx[ix] = 0;
			if	( *aPackage ) {
				w = 0;
				while	( w < sizeof( findname ) - 1 ) {
					if	( *aPackage == 0 )		break;
					if	( *aPackage == ',' )	break;
					findname[w++] = *(aPackage++);
				}
				findname[w] = 0;
				if		( *aPackage == ',' )	aPackage++;
				else if	( *aPackage )			throw CError( "Package name too long" );

				if	( itsUnUtils->UnNameTable().Exist( findname ) ) {
			    	findindx[ix] = itsUnUtils->UnNameTable().Find( findname );
					break;
				}
			}
			else {
				ix++;
			}
		}

	    dword_t i_core  = itsUnUtils->UnNameTable().Find( "core" );
    	dword_t	i_pack  = itsUnUtils->UnNameTable().Find( "package" );
		for	( ix = 0; ix < 20; ix++ ) {
			if	( findindx[ix] ) {
				for	( dword_t i = 0; i < itsUnUtils->UnImportTable().Count(); i++ ) {
			        CUnImport_t imp = itsUnUtils->UnImportTable().Import( i );
	
			        if  ( imp.Package() != 0 )              	continue;
	        		if  ( imp.ClassPackage() != i_core )    	continue;
			        if  ( imp.ClassName() != i_pack )       	continue;
					if	( imp.ObjectName() == findindx[ix] )	return true;
				}
			}
		}
	}
	return false;
}

// --------------------------------------------------------------------
// private:		Try if has not these packages
// --------------------------------------------------------------------
bool	CIdentifyFile::TryHasnot	(	const char * aActor ) {
	if	( itsUnUtils ) {
		char	findname[64];
		word_t	w;
		while	( *aActor ) {
			w = 0;
			while	( w < sizeof( findname ) - 1 ) {
				if	( *aActor == 0 )		break;
				if	( *aActor == ',' )		break;
				findname[w++] = *(aActor++);
			}
			findname[w] = 0;
			if		( *aActor == ',' )	aActor++;
			else if	( *aActor )			throw CError( "Package name too long" );
			if	( itsUnUtils->UnNameTable().Exist( findname ) ) {
				return	false;
			}
		}
		return true;
	}
	return false;
}

// --------------------------------------------------------------------
// private:		Try sign - first few bytes of the file
// --------------------------------------------------------------------
bool	CIdentifyFile::TrySign	( const char * aSign ) {
	bool	result = false;
	try	{
		FILE *	stm = ::fopen( itsFileName, "rb" );
		char	buf[100];

		if	( stm ) {
			::memset( buf, 0, sizeof( buf ) );
			::fread( buf, ::strlen( aSign ), 1, stm );
			buf[ ::strlen( aSign ) ] = 0;
			::fclose( stm );
			result = ::strcmp( aSign, buf ) == 0;
		}
	}
	catch ( ... ) {
	}
	return result;
}

// --------------------------------------------------------------------
// private:		Clean up the mess
// --------------------------------------------------------------------
void	CIdentifyFile::Cleanup	( void ) {
	itsUnUtils = NULL;
	itsUmod	= false;
	itsExe	= false;
	itsU1	= false;
	itsUT	= false;
	itsAAO	= false;
	itsUT2k3= false;
	::memset( &itsStat, 0, sizeof( itsStat ) );
	::memset( &itsType, 0, sizeof( itsType ) );
	::memset( itsTyp1,  0, sizeof( itsTyp1 ) );
	::memset( itsCtgr,  0, sizeof( itsCtgr ) );
	::memset( itsTyp2,  0, sizeof( itsTyp2 ) );
	::memset( itsFileName, 0, sizeof( itsFileName ) );
	::memset( itsPath, 0, sizeof( itsPath ) );
	::memset( itsName, 0, sizeof( itsName ) );
	::memset( itsSuff, 0, sizeof( itsSuff ) );
	::memset( itsAuthor, 0, sizeof( itsAuthor ) );
	::memset( itsTitle, 0, sizeof( itsTitle ) );
	itsPackageVersion	= 0;
	itsLicenseeMode		= 0;
	itsAuthorId			= 0;
}

// --------------------------------------------------------------------
// private:		Free data
// --------------------------------------------------------------------
void	CIdentifyFile::Free	( void ) {
	if	( itsUnUtils )	delete itsUnUtils;
	CIdentifyFile::Cleanup();
}

// --------------------------------------------------------------------
// EOF:	CIdentifyFile.cxx
// --------------------------------------------------------------------
