// --------------------------------------------------------------------
// CWinIniItem.cpp
// Whatis:  Windows style INI file item
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 10-FEB-2003     Created this source
// --------------------------------------------------------------------
#include	"CError.hxx"
#include	"CWinIniItem.hxx"

// --------------------------------------------------------------------
// public:	Constructors
// --------------------------------------------------------------------
CWinIniItem::CWinIniItem	() {
	CWinIniItem::Cleanup();
}

// --------------------------------------------------------------------
CWinIniItem::CWinIniItem	( const CWinIniItem & aC ) {
	CWinIniItem::Cleanup();
	CWinIniItem::Name	( aC.Name() );
	CWinIniItem::Value	( aC.Value() );
}

// --------------------------------------------------------------------
CWinIniItem CWinIniItem::operator =	( const CWinIniItem & aC ) {
	CWinIniItem::Free();
	CWinIniItem::Name	( aC.Name() );
	CWinIniItem::Value	( aC.Value() );
	return	*this;
}

// --------------------------------------------------------------------
// public:	Destructor
// --------------------------------------------------------------------
CWinIniItem::~CWinIniItem	() {
	CWinIniItem::Free();
}

// --------------------------------------------------------------------
// public:	Comparison operators
// --------------------------------------------------------------------
bool	CWinIniItem::operator == ( const CWinIniItem & aC ) const {
	return	::strcmp( CWinIniItem::Name(), aC.Name() ) == 0;
}

// --------------------------------------------------------------------
bool	CWinIniItem::operator != ( const CWinIniItem & aC ) const {
	return	::strcmp( CWinIniItem::Name(), aC.Name() ) != 0;
}

// --------------------------------------------------------------------
bool	CWinIniItem::operator ==	( const char * aS ) const {
	return	::strcmp( CWinIniItem::Name(), aS ? aS : "" ) == 0;
}

// --------------------------------------------------------------------
bool	CWinIniItem::operator !=	( const char * aS ) const {
	return	::strcmp( CWinIniItem::Name(), aS ? aS : "" ) != 0;
}

// --------------------------------------------------------------------
// public:	Streaming
// --------------------------------------------------------------------
void	CWinIniItem::Write	( FILE * aS ) const {
	if	( itsName ) {
		::fprintf( aS, "%s=", itsName );
		if	( ferror( aS ) )	throw CError( ::strerror( errno ) );
		if	( itsValue ) {
			::fprintf( aS, "%s", itsValue );
			if	( ferror( aS ) )	throw CError( ::strerror( errno ) );
		}
		::fprintf( aS, "\n" );
		if	( ferror( aS ) )	throw CError( ::strerror( errno ) );
	}
}

// --------------------------------------------------------------------
void	CWinIniItem::Read	( FILE * aS ) {
	char		myline [2048];
    long int    mypos;

	// Get rid of carbage
	CWinIniItem::Name	( NULL );
	CWinIniItem::Value	( (const char *)NULL );

    // Read the file line at a time - stop at section limit though
	*myline = 0;
    while	( *myline == 0 ) {
    	mypos = ::ftell( aS );
    	if		( fgets( myline, sizeof( myline ) - 1, aS ) ) {
			::my_strfix( myline );
			if	( *myline == '[' ) {
            	::fseek( aS, mypos, SEEK_SET );
                return;
            }
            if	( *myline == 0 )					continue;
            if	( ! ::strncmp( myline, "//", 2 ) )	continue;
        }

        // Error
        else if	( ferror( aS ) ) {
        	throw CError( ::strerror( errno ) );
        }

        // EOF
        else {
			return;
        }
    }

    // Now we have the line - take the name and the value apart
    char *	value = ::strchr( myline, '=' );
	if	( ! value )	value = ::strchr( myline, ':' );
	if	( ! value )	throw	CError( "Invalid format" );

    *value = 0;	value++;
	::my_strfix( myline );
    ::my_strfix( value );

	if	( *myline == 0 )	throw	CError( "Invalid format" );

	CWinIniItem::Name	( myline );
	CWinIniItem::Value	( *value ? value : NULL );
}

// --------------------------------------------------------------------
// public:	Accessing instance data
// --------------------------------------------------------------------
void	CWinIniItem::Name	( const char * aName ) {
	if	( itsName )		delete [] itsName;
	itsName = ::my_private_strdup( aName );
}

// --------------------------------------------------------------------
void	CWinIniItem::Value	( const char * aValue ) {
	if	( itsValue )	delete [] itsValue;
	itsValue = ::my_private_strdup( aValue );
}

// --------------------------------------------------------------------
void	CWinIniItem::Value	( dword_t aValue ) {
	char	intbuf[16];
	::sprintf( intbuf, "%d", aValue );
	CWinIniItem::Value( intbuf );
}

// --------------------------------------------------------------------
void	CWinIniItem::Value	( bool aValue ) {
	CWinIniItem::Value( aValue ? "true" : "false" );
}

// --------------------------------------------------------------------
const char *	CWinIniItem::Name	( void ) const {
	return itsName ? itsName : "";
}

// --------------------------------------------------------------------
const char *	CWinIniItem::Value	( void ) const {
	return itsValue ? itsValue : "";
}

// --------------------------------------------------------------------
dword_t			CWinIniItem::IntValue( void ) const {
	return (dword_t)::atol( CWinIniItem::Value() );
}

// --------------------------------------------------------------------
bool			CWinIniItem::BoolValue( void ) const {
#ifdef	WIN32
	return	::stricmp( CWinIniItem::Value(), "true" ) == 0;
#else
	return	::my_stricmp( CWinIniItem::Value(), "true" ) == 0;
#endif
}

// --------------------------------------------------------------------
// private:	Private helpers
// --------------------------------------------------------------------
void			CWinIniItem::Free	( void ) {
	if	( itsName )		delete [] itsName;
    if	( itsValue )    delete [] itsValue;
    CWinIniItem::Cleanup();
}

// --------------------------------------------------------------------
void			CWinIniItem::Cleanup	( void ) {
	itsName		= NULL;
    itsValue	= NULL;
}

// --------------------------------------------------------------------
// EOF:	CWinIniItem.cpp
// --------------------------------------------------------------------
