// --------------------------------------------------------------------
// CWinIniSection.cpp
// Whatis:  Windows style INI file item
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 10-FEB-2003     Created this source
// --------------------------------------------------------------------
#include	"CWinIniSection.hxx"
#include	"CError.hxx"

// --------------------------------------------------------------------
// public:	Constructors
// --------------------------------------------------------------------
CWinIniSection::CWinIniSection	() {
	CWinIniSection::Cleanup();
}

// --------------------------------------------------------------------
CWinIniSection::CWinIniSection	( const CWinIniSection & aC ) {
	CWinIniSection::Cleanup();
	CWinIniSection::Name	( aC.Name() );
	CWinIniSection::ItemList( aC.ItemList() );
}

// --------------------------------------------------------------------
CWinIniSection	CWinIniSection::operator = ( const CWinIniSection & aC ) {
	CWinIniSection::Free();
	CWinIniSection::Name	( aC.Name() );
	CWinIniSection::ItemList( aC.ItemList() );
	return *this;
}

// --------------------------------------------------------------------
// public:	Destructor
// --------------------------------------------------------------------
CWinIniSection::~CWinIniSection	() {
	CWinIniSection::Free();
}

// --------------------------------------------------------------------
// public:	Comparison operators
// --------------------------------------------------------------------
bool	CWinIniSection::operator ==	( const CWinIniSection & aC ) const {
	return	::strcmp( CWinIniSection::Name(), aC.Name() ) == 0;
}

// --------------------------------------------------------------------
bool	CWinIniSection::operator !=	( const CWinIniSection & aC ) const {
	return	::strcmp( CWinIniSection::Name(), aC.Name() ) != 0;
}

// --------------------------------------------------------------------
bool	CWinIniSection::operator ==	( const char * aS ) const {
	return	::strcmp( CWinIniSection::Name(), aS ? aS : "" ) == 0;
}

// --------------------------------------------------------------------
bool	CWinIniSection::operator !=	( const char * aS ) const {
	return	::strcmp( CWinIniSection::Name(), aS ? aS : ""  ) != 0;
}

// --------------------------------------------------------------------
// public:	Streaming
// --------------------------------------------------------------------
void	CWinIniSection::Write	( FILE * aS ) const {
	if	( itsName ) {
		::fprintf( aS, "[%s]\n", itsName );
		if	( ferror( aS ) )	throw CError( ::strerror( errno ) );

		CWinIniItem_lci	loop;
		for	( loop = itsItemList.begin(); loop != itsItemList.end(); loop++ ) {
			(*loop).Write( aS );
		}

		::fprintf( aS, "\n" );
		if	( ferror( aS ) )	throw CError( ::strerror( errno ) );
	}
}

// --------------------------------------------------------------------
void	CWinIniSection::Read	( FILE * aS ) {
	char		myline [2048];

	// Dispose carbage (my name and list)
	CWinIniSection::Cleanup();

	// Find line that is a tag
	*myline = 0;
    while	( *myline == 0 ) {
    	if	( ! ::fgets( myline, sizeof( myline ) - 1, aS ) ) {
        	if	( ferror( aS ) ) {
	        	throw CError( ::strerror( errno ) );
            }
            return;
        }

		::my_strfix( myline );
		if	( *myline == 0 )	continue;
		if	( *myline == '[' )	break;
		if	( *myline == ';' )	*myline = 0;
    }

	// This should be a tag of form [tagname]
	::strcpy( myline, myline + 1 );	// Get rid of the first character

    char *	term = ::strchr( myline, ']' );
	if	( ! term ) {
    	throw CError( "Invalid format" );
    }

	*term = 0;
	::my_strfix( myline );
	if	( *myline == 0 )	throw CError( "Invalid format" );

	CWinIniSection::Name( myline );

	// Now, read in the items one by one
	CWinIniItem	myitem;
	do	{
		myitem.Read( aS );
		if	( myitem.Name()[0] ) {
			itsItemList.push_back( myitem );
		}
	}	while	( myitem.Name()[0] );
}

// --------------------------------------------------------------------
// public:	Get some data
// --------------------------------------------------------------------
CWinIniItem_l	CWinIniSection::ItemList( const char * aName ) const {
	CWinIniItem_l	list;
    CWinIniItem_lci	loop;

    for	( loop  = itsItemList.begin(); loop != itsItemList.end(); loop++ ) {
    	if	( (*loop) == aName )	list.push_back( *loop );
    }
	return	list;
}

// --------------------------------------------------------------------
// public:	Assign some data
// --------------------------------------------------------------------
void	CWinIniSection::ItemList( const CWinIniItem_l & aL ) {
	itsItemList.clear();
	itsItemList = aL;
}

// --------------------------------------------------------------------
void	CWinIniSection::Name	( const char * aName ) {
	if	( itsName )		delete [] itsName;
	itsName = ::my_private_strdup( aName );
}

// --------------------------------------------------------------------
void	CWinIniSection::Insert	( const CWinIniItem & aI ) {
	itsItemList.push_back( aI );
}

// --------------------------------------------------------------------
void	CWinIniSection::Insert	( const CWinIniItem_l & aL ) {
	CWinIniItem_lci	loop;
	for	( loop = aL.begin(); loop != aL.end(); loop++ ) {
		itsItemList.push_back( *loop );
	}
}

// --------------------------------------------------------------------
void	CWinIniSection::Delete	( const CWinIniItem & aI ) {
	itsItemList.remove( aI );
}

// --------------------------------------------------------------------
void	CWinIniSection::Delete	( const CWinIniItem_l & aL ) {
	CWinIniItem_lci	loop;
	for	( loop = aL.begin(); loop != aL.end(); loop++ ) {
		itsItemList.remove( *loop );
	}
}

// --------------------------------------------------------------------
const char *	CWinIniSection::Value	( const char * aName ) const {
	CWinIniItem_lci	loop;
	for	( loop = itsItemList.begin(); loop != itsItemList.end(); loop++ ) {
    	if	( (*loop) == aName )	return (*loop).Value();
	}
	return	"";
}

// --------------------------------------------------------------------
// public:	Special inserts to easen the coders life
// --------------------------------------------------------------------
void	CWinIniSection::Insert	( const char * aName, const char * aV ) {
	CWinIniItem	itm;
    itm.Name( aName );
    itm.Value( aV );
	CWinIniSection::Insert( itm );
}

// --------------------------------------------------------------------
void	CWinIniSection::Insert	( const char * aName, bool aV ) {
	CWinIniSection::Insert( aName, aV ? "true" : "false" );
}

// --------------------------------------------------------------------
void	CWinIniSection::Insert	( const char * aName, int aV ) {
	char	buf[32];
    ::sprintf( buf, "%d", aV );
	CWinIniSection::Insert( aName, buf );
}

// --------------------------------------------------------------------
const char *			CWinIniSection::Name	( void ) const { return itsName ? itsName : ""; }

// --------------------------------------------------------------------
const CWinIniItem_l &	CWinIniSection::ItemList( void ) const { return itsItemList; }

// --------------------------------------------------------------------
void					CWinIniSection::Clear	( void ) { itsItemList.clear(); }

// --------------------------------------------------------------------
// private:	Private helpers
// --------------------------------------------------------------------
void	CWinIniSection::Free	( void ) {
	if	( itsName )		delete [] itsName;
	CWinIniSection::Cleanup();
}

// --------------------------------------------------------------------
void	CWinIniSection::Cleanup	( void ) {
	itsName = NULL;
	itsItemList.clear();
}

// --------------------------------------------------------------------
// EOF:	CWinIniSection.cpp
// --------------------------------------------------------------------
