// --------------------------------------------------------------------
// CZip.hxx
// Whatis:	Class for reading/writing zip files
// Authors:	Esko 'Varpu' Ilola	EIL
// History:	EIL	24-FEB-2003		Created	this source
// --------------------------------------------------------------------
// You should include this file to your code in order to use the CZip
// class library. Do not include the other files unless you know what
// you are doing.
// For information on how to manipulate the IZipStream, see
// the header file IZipStream.hxx
// --------------------------------------------------------------------
#ifndef	CZip_hxx
#define	CZip_hxx

#include	"CGz.hxx"
#include	"CZipFileBlock.hxx"
#include	"CZipCentralDir.hxx"
#include	"CZipFlags.hxx"
#include	"CZipNotification.hxx"
#include	"IZipStream.hxx"

// --------------------------------------------------------------------
// Declaration of the CZip class
// --------------------------------------------------------------------
class	CZip {

	// ----------------------------------------------------------------
	public:		// Constructor and destructor
	// ----------------------------------------------------------------
	CZip	();
	~CZip	();

	// ----------------------------------------------------------------
	public:		// Copy constructor and assignment operator
	// ----------------------------------------------------------------
	CZip				( const CZip & );
    CZip & operator =	( const CZip & );

	// ----------------------------------------------------------------
	public:		// Interface
	// ----------------------------------------------------------------
	// Read the zip
	// ----------------------------------------------------------------
	void		Read		( const char * aZipFile );
	void		Read		( IZipStream & aReader );

	// ----------------------------------------------------------------
	// Save the zip
	// ----------------------------------------------------------------
	void		Write		( const char * aZipFile, CZipNotification * aNotify = NULL );
	void		Write		( IZipStream & aReader, CZipNotification * aNotify = NULL );

	// ----------------------------------------------------------------
    dword_t		ZippedSize	( void ) const;

	// ----------------------------------------------------------------
	// If you intent to create a zip you should start the process by
	// clearing the directory. After this you can add/delete items to/from it.
	// You can also modify existing zip by reading it's directory and then
	// manipulating it.
	// ----------------------------------------------------------------
	void		Cleanup				( void );

	// ----------------------------------------------------------------
	// A special method to add a gzip member - used for gz -> zip conversions
	// ----------------------------------------------------------------
    void		AddDirectoryEntry	(	const CGz &		aGzMember,
                                        const char *	aDirectoryName,
										time_t			aFileTime );

	// ----------------------------------------------------------------
	// With these methods you can add and delete files from the zip.
	// If the Filter is omitted, all entries will be erased.
	// ----------------------------------------------------------------
	void		AddDirectoryEntry	( const CZipFlags &	aFlags,
									  const char *		aLocalFileName,
									  const char *		aDirectoryName );
	void		AddDirectoryEntry	( const CZipFlags &	aFlags,
									  const char *		aData,
                                      dword_t			aDataLen,
									  const char *		aDirectoryName );
	void		AddDirectoryEntry	( const CZipFlags &	aFlags,
									  IZipStream *		aStream,
                                      dword_t			aSize,
									  const char *		aDirectoryName );
	void		DelDirectoryEntry	( const char *		aFilter = NULL );

	// ----------------------------------------------------------------
	// You can also attach comments to the zip. You can remove comments
	// by omitting the comment from the call or setting it manually to NULL
	// If the filter is NULL then all files will be affected.
	// ----------------------------------------------------------------
	void		AddZipComment		( const char * aComment = NULL );
	void		AddFileComment		( const char * aFilter  = NULL,
									  const char * aComment = NULL );

	// ----------------------------------------------------------------
	// Call Decompress method to extract files from memory.
	// ----------------------------------------------------------------
	void		Decompress			( const CZipFlags &	aFlags,
									  const char * 		aTargetPath,
									  const char * 		aFilter = NULL );

	// ----------------------------------------------------------------
	// Call the Compress method after modifying the directory entries.
	// ----------------------------------------------------------------
    void		Compress			( CZipNotification * aNotify = NULL );

	// ----------------------------------------------------------------
	public:		// Accessing instance data.
	// ----------------------------------------------------------------
	// These are read-only for a good reason. You can circumvent this
	// by using type-cast but it should not be nessecary to manipulate
	// these directly.
	// ----------------------------------------------------------------
	const CZipFileBlock_l &	FileBlocks		( void ) const;
	const CZipCentralDir &	CentralDir		( void ) const;

	// ----------------------------------------------------------------
	private:	// Private helpers
	// ----------------------------------------------------------------
    void		Free				( void );
	void		Clear				( void );

	// ----------------------------------------------------------------
	private:	// Instance data
	// ----------------------------------------------------------------
	CZipFileBlock_l		itsFileBlocks;
	CZipCentralDir		itsCentralDir;

};

#endif	// #ifdef  CZip_hxx
// --------------------------------------------------------------------
// EOF:	CZip.hxx
// --------------------------------------------------------------------
