// --------------------------------------------------------------------
// CGzData.cpp
// Whatis:	gz members
// Authors:	Esko 'Varpu' Ilola	EIL
// History:	EIL	07-MAR-2003		Created	this source
// --------------------------------------------------------------------
#include	"CGzData.hxx"
#include	"CZipStreamFile.hxx"
#include	"CError.hxx"

// --------------------------------------------------------------------
// public:	Constructors and destructor
// --------------------------------------------------------------------
CGzData::CGzData	() {
	CGzData::Clear();
}

// --------------------------------------------------------------------
CGzData::~CGzData	() {
	CGzData::Free();
}

// --------------------------------------------------------------------
CGzData::CGzData	( const CGzData & aC ) {
	CGzData::Clear();
    itsCrc32				= aC.Crc32();
    itsIsize				= aC.Isize();
	itsCompressedFileOffset = aC.CompressedFileOffset();
	itsCompressedDataSize	= aC.CompressedDataSize();

    if	( aC.Data() ) {
    	itsData = new byte_t [itsCompressedDataSize];
        ::memcpy( itsData, aC.Data(), itsCompressedDataSize );
    }
    if	( aC.CompressedFileName() ) {
    	itsCompressedFileName = new char [ ::strlen(aC.CompressedFileName()) + 1];
        ::strcpy( itsCompressedFileName, aC.CompressedFileName() );
    }
}

// --------------------------------------------------------------------
CGzData & CGzData::operator =	( const CGzData & aC ) {
	CGzData::Free();
    itsCrc32				= aC.Crc32();
    itsIsize				= aC.Isize();
	itsCompressedFileOffset = aC.CompressedFileOffset();
	itsCompressedDataSize	= aC.CompressedDataSize();

    if	( aC.Data() ) {
    	itsData = new byte_t [itsCompressedDataSize];
        ::memcpy( itsData, aC.Data(), itsCompressedDataSize );
    }
    if	( aC.CompressedFileName() ) {
    	itsCompressedFileName = new char [ ::strlen(aC.CompressedFileName()) + 1];
        ::strcpy( itsCompressedFileName, aC.CompressedFileName() );
    }
	return *this;
}

// --------------------------------------------------------------------
// public:	Interfaces
// --------------------------------------------------------------------
void			CGzData::Read	( IZipStream & aReader ) {
	CGzData::Free();

    // If this is a seekable stream - try out if the file end matches our assumptions
    if	( aReader.CanSeek() ) {
		dword_t	s_off	= aReader.Tell();

		// Seek the file 8 bytes away from the end
        aReader.Seek( -8, zip_stream_seek_end );
        dword_t	e_off	= aReader.Tell();

        // Do we have enough data in between ?
        if	( s_off >= e_off )	throw CError( aReader.Name(), "Not enough data" );

        // Get the CRC and the size
        itsCrc32.Read( aReader );
        itsIsize.Read( aReader );

        // Rest is just setting up the data
		itsCompressedDataSize	= e_off - s_off;
		itsCompressedFileOffset	= s_off;
		itsCompressedFileName	= new char [ ::strlen( aReader.Name() ) + 1 ];
        ::strcpy( itsCompressedFileName, aReader.Name() );
    }

    // Unable to seek - this is really bad
    else {
    	throw	CError( aReader.Name(), "Cannot seek" );
    }
}

// --------------------------------------------------------------------
void			CGzData::Write	( IZipStream & aWriter ) {
    if		( itsCompressedFileName ) {
		CZipStreamFile	myreader( itsCompressedFileName );
        myreader.OpenRO();
        myreader.Seek( itsCompressedFileOffset, zip_stream_seek_set );
        aWriter.Copy( myreader, itsCompressedDataSize );
    }
    else if	( itsData ) {
		aWriter.Write( itsData, itsCompressedDataSize );
    }
    itsCrc32.Write( aWriter );
    itsIsize.Write( aWriter );
}

// --------------------------------------------------------------------
void			CGzData::Cleanup	( void ) {
	CGzData::Free();
}

// --------------------------------------------------------------------
CZipCrc32		CGzData::Crc32					( void ) const {
	return itsCrc32;
}

// --------------------------------------------------------------------
CZipDword		CGzData::Isize					( void ) const {
	return itsIsize;
}

// --------------------------------------------------------------------
const byte_t *	CGzData::Data					( void ) const {
	return itsData;
}

// --------------------------------------------------------------------
const char *	CGzData::CompressedFileName		( void ) const {
	return itsCompressedFileName;
}

// --------------------------------------------------------------------
dword_t			CGzData::CompressedFileOffset	( void ) const {
	return itsCompressedFileOffset;
}

// --------------------------------------------------------------------
dword_t			CGzData::CompressedDataSize		( void ) const {
	return itsCompressedDataSize;
}

// --------------------------------------------------------------------
// private:	Private helpers
// --------------------------------------------------------------------
void	CGzData::Free		( void ) {
	if	( itsData )					delete [] itsData;
    if	( itsCompressedFileName )	delete [] itsCompressedFileName;
	CGzData::Clear();
}

// --------------------------------------------------------------------
void	CGzData::Clear		( void ) {
    itsCrc32.Cleanup();
    itsIsize.Cleanup();
	itsData					= NULL;
    itsCompressedFileName	= NULL;
	itsCompressedFileOffset	= 0;
	itsCompressedDataSize	= 0;
}

// --------------------------------------------------------------------
// EOF:	CGzData.cpp
// --------------------------------------------------------------------
