// --------------------------------------------------------------------
// CZipDeflate.cpp
// Whatis:	Class for compressing using the deflate method
// Authors:	Esko 'Varpu' Ilola	EIL
// History:	EIL	20-JUN-2003		Created	this source
// --------------------------------------------------------------------
#include	"CZipDeflate.hxx"
#include	"CDeflate.hxx"
#include	"CError.hxx"

// --------------------------------------------------------------------
// local:	A local file buffer to avoid clobbering memory
// --------------------------------------------------------------------
static	byte_t	_my_read_buffer [16384]	= { 0 };
static	byte_t	_my_writ_buffer [16384]	= { 0 };

// --------------------------------------------------------------------
// public:	Constructors
// --------------------------------------------------------------------
CZipDeflate::CZipDeflate() {
	CZipDeflate::Clear();
}

// --------------------------------------------------------------------
// public:	Destructor
// --------------------------------------------------------------------
CZipDeflate::~CZipDeflate	() {
	CZipDeflate::Free();
}

// --------------------------------------------------------------------
// public:	Compress into a stream
// --------------------------------------------------------------------
void		CZipDeflate::Compress	( 	CZipCrc32 & 		aCrc32,
										dword_t & 			aLen,
                                        const CZipFlags		aFg,
                                        IZipStream & 		aIn,
                                        IZipStream & 		aOut,
                                        CZipNotification *	aNotify ) {
    dword_t			readct;
    bool			aborting	= false;
    CDeflate		mydeflate;
    codec_stream_t	mystream;

	aLen	= 0L;
    aCrc32	= (dword_t)0L;
	CZipDeflate::Free();

    try {
	    // Update the notification
    	if	( aNotify != NULL ) {
    		aNotify->Task( "Deflating" );
	        aNotify->FileName( aIn.Name() );
            aNotify->FileBytes( aIn.Size() );
            aNotify->FileBytesDone( 0L );
            aborting = aNotify->Aborting();
    	}

		::memset( &mystream, 0, sizeof( mystream ) );
		mystream.total_in	= 0;
       	mystream.total_out	= 0;
		mystream.checksum	= NULL;
        mystream.complevel	= aFg.CompressionLevel();

		while	( ( ! aborting ) && ( ! aIn.Eof() ) ) {
			// Read in the data in 16k blocks - only when needed
            if	( mystream.avail_in == 0 ) {
				readct = aIn.Read( _my_read_buffer, sizeof( _my_read_buffer ) );

	            // Calculate the checksum
    	        aCrc32.Calculate( _my_read_buffer, readct );

	            // Update progress bars
		    	if	( aNotify != NULL ) {
    		        aNotify->FileBytesDone( aNotify->FileBytesDone() + readct );
    	    	    aNotify->TotalBytesDone( aNotify->TotalBytesDone() + readct );
		            aborting = aNotify->Aborting();
		    	}
	            mystream.next_in	= _my_read_buffer;
    	        mystream.avail_in	= readct;
            }

            // Set up the codec stream for each iteration
	        mystream.next_out	= _my_writ_buffer;
    	    mystream.avail_out	= sizeof( _my_writ_buffer );

            // Compress and calculate checksum accordingly
            mydeflate.Compress( mystream );

            // Output anything that was put into the output buffer
			if	( mystream.avail_out < sizeof( _my_writ_buffer ) ) {
            	aOut.Write( _my_writ_buffer, sizeof( _my_writ_buffer ) - mystream.avail_out );
            }
        }

        // In any case terminate the compressor - flush any data to out as well
        mystream.next_out	= _my_writ_buffer;
   	    mystream.avail_out	= sizeof( _my_writ_buffer );
        while	( ! mydeflate.End( mystream ) ) {
			if	( mystream.avail_out < sizeof( _my_writ_buffer ) ) {
            	aOut.Write( _my_writ_buffer, sizeof( _my_writ_buffer ) - mystream.avail_out );
            }
	        mystream.next_out	= _my_writ_buffer;
   		    mystream.avail_out	= sizeof( _my_writ_buffer );
		}
		if	( mystream.avail_out < sizeof( _my_writ_buffer ) ) {
           	aOut.Write( _my_writ_buffer, sizeof( _my_writ_buffer ) - mystream.avail_out );
        }

        // If we didn't get out of there because of abort request - do some checks
		if	( ! aborting ) {
		    // Update the notification
    		if	( aNotify != NULL ) {
	    		aNotify->Task( "" );
		        aNotify->FileName( "" );
	    	}

            // Set size from arguments
            aLen = mystream.total_out;

       	}	// ! aborting

        else {	// Aborting !!!
    	    aLen	= 0;
    	    aCrc32	= 0;
        }
	}

    catch	( CError e ) {
   	    aLen	= 0;
   	    aCrc32	= 0;
    	throw	CError( aIn.Name(), "CZipDeflate::Compress", e.Error() );
    }
    catch	( ... ) {
   	    aLen	= 0;
   	    aCrc32	= 0;
    	throw	CError( aIn.Name(), "CZipDeflate::Compress", "Unknown error" );
    }
}

// --------------------------------------------------------------------
// public:	Compress into a static buffer
// --------------------------------------------------------------------
byte_t *	CZipDeflate::Compress	( 	CZipCrc32 & 		aCrc32,
										dword_t & 			aLen,
                                        const CZipFlags		aFg,
                                        IZipStream & 		aIn,
                                        CZipNotification *	aNotify ) {
	struct stat		mystat;
    dword_t			readct;
    bool			aborting	= false;
    CDeflate		mydeflate;
    codec_stream_t	mystream;
    byte_t *		compdata	= NULL;
    dword_t			compsize;

	aLen	= 0L;
    aCrc32	= (dword_t)0L;
	CZipDeflate::Free();

    try {
		// The stream must be able to seek (in order to get the size)
		if	( ! aIn.CanSeek() ) {
			throw CError( "Can not seek" );
	    }

		// Get statistics of the file - size for one
    	if	( ::stat( aIn.Name(), &mystat ) != 0 ) {
			throw CError( ::strerror( errno ) );
	    }

	    // Update the notification
    	if	( aNotify != NULL ) {
    		aNotify->Task( "Deflating" );
	        aNotify->FileName( aIn.Name() );
            aNotify->FileBytes( mystat.st_size );
            aNotify->FileBytesDone( 0L );
            aborting = aNotify->Aborting();
    	}

        // Allocate enough buffer to hold the compressed data for sure !
        compsize = mystat.st_size + ( mystat.st_size / 9000 ) + 1024;
		compdata = new byte_t [compsize];

		mystream.total_in	= 0;
       	mystream.total_out	= 0;
		mystream.checksum	= NULL;
        mystream.complevel	= aFg.CompressionLevel();
        mystream.next_out	= compdata;
        mystream.avail_out	= compsize;

		while	( ( ! aborting ) && ( ! aIn.Eof() ) ) {
			// Read in the data in 16k blocks
			readct = aIn.Read( _my_read_buffer, sizeof( _my_read_buffer ) );

            // Update progress bars
	    	if	( aNotify != NULL ) {
    	        aNotify->FileBytesDone( aNotify->FileBytesDone() + readct );
    	        aNotify->TotalBytesDone( aNotify->TotalBytesDone() + readct );
	            aborting = aNotify->Aborting();
	    	}

            // Calculate the checksum
            aCrc32.Calculate( _my_read_buffer, readct );

            // Set up the codec stream for each iteration
            mystream.next_in	= _my_read_buffer;
            mystream.avail_in	= readct;

            // Compress and calculate checksum accordingly
            mydeflate.Compress( mystream );

			// The target buffer should NEVER get exhausted !
            if	( mystream.avail_out == 0 ) {
				throw CError( "Target buffer overflow" );
            }
        }

        // In any case terminate the compressor
        while	( ! mydeflate.End( mystream ) );

        // If we didn't get out of there because of abort request - do some checks
		if	( ! aborting ) {
		    // Update the notification
    		if	( aNotify != NULL ) {
	    		aNotify->Task( "" );
		        aNotify->FileName( "" );
	    	}

            // Set size from arguments
            aLen = mystream.total_out;

            // We might want to get a smaller memory part fot the data
            if	( ( aLen ) && ( aLen < compsize ) ) {
				byte_t *	tmp = new byte_t [aLen];
                ::memcpy( tmp, compdata, aLen );
                delete [] compdata;
                compdata = tmp;
           	}
        }	// ! aborting

        else {	// Aborting !!!
    	    aLen	= 0;
    	    aCrc32	= 0;
			if	( compdata ) delete [] compdata;
            compdata = NULL;
        }
	}

    catch	( CError e ) {
   	    aLen	= 0;
   	    aCrc32	= 0;
		if	( compdata ) delete [] compdata;
        compdata = NULL;
    	throw	CError( aIn.Name(), "CZipDeflate::Compress", e.Error() );
    }
    catch	( ... ) {
   	    aLen	= 0;
   	    aCrc32	= 0;
		if	( compdata ) delete [] compdata;
        compdata = NULL;
    	throw	CError( aIn.Name(), "CZipDeflate::Compress", "Unknown error" );
    }

	return	compdata;
}

// --------------------------------------------------------------------
// private:	Private helpers
// --------------------------------------------------------------------
void	CZipDeflate::Free	( void ) {
	CZipDeflate::Clear();
}

// --------------------------------------------------------------------
void	CZipDeflate::Clear	( void ) {
}

// --------------------------------------------------------------------
// EOF:	CZipDeflate.cpp
// --------------------------------------------------------------------
