// --------------------------------------------------------------------
// CZipFileBlock.cpp
// Whatis:	File block of a zip file
// Authors:	Esko 'Varpu' Ilola	EIL
// History:	EIL	24-FEB-2003		Created	this source
// --------------------------------------------------------------------
#include	"CZipFileBlock.hxx"
#include	"CZipStreamFile.hxx"
#include	"CZipStreamTar.hxx"
#include	"CError.hxx"

// --------------------------------------------------------------------
// public:	Constructors
// --------------------------------------------------------------------
CZipFileBlock::CZipFileBlock	() {
	CZipFileBlock::Clear();
}

// --------------------------------------------------------------------
CZipFileBlock::CZipFileBlock	(   const CZipFlags &	aFlags,
									IZipStream *		aStream,
                                    dword_t				aSize,
								  	const char * 		aDirectoryName ) {
	CZipFileBlock::Clear();

  	CZipLocalFileHeader	myheader	( aFlags, aStream, aSize, aDirectoryName );
	itsZipStream		= aStream;
    itsLocalFileHeader	= myheader;
    itsFlags			= aFlags;
}

// --------------------------------------------------------------------
CZipFileBlock::CZipFileBlock	(   const CZipFlags &	aFlags,
									const char * 		aLocalFile,
								  	const char * 		aDirectoryName ) {
	CZipFileBlock::Clear();

    if	( aLocalFile ) {
		itsLocalFileName	= new char [ ::strlen( aLocalFile ) + 1];
        ::strcpy( itsLocalFileName, aLocalFile );
    }

  	CZipLocalFileHeader	myheader	( aFlags, aLocalFile, aDirectoryName );

    itsLocalFileHeader	= myheader;
    itsFlags			= aFlags;
}

// --------------------------------------------------------------------
CZipFileBlock::CZipFileBlock	(   const CZipFlags &	aFlags,
									const char * 		aDataBlock,
									dword_t				aDataLen,
								  	const char * 		aDirectoryName ) {
	CZipFileBlock::Clear();

  	CZipLocalFileHeader	myheader	( aFlags, aDataBlock, aDataLen, aDirectoryName );
    CZipFileData		mydata		( aDataBlock, aDataLen );
    myheader.CompressedSize( mydata.CompressedData().Size() );

    itsLocalFileHeader	= myheader;
    itsFileData			= mydata;
    itsFlags			= aFlags;
}

// --------------------------------------------------------------------
CZipFileBlock::CZipFileBlock	( const CGz &	aGz,
								  const char *	aDirectoryName,
								  time_t		aFileTime ) {
	CZipFileBlock::Clear();

    CZipLocalFileHeader	myheader	( aGz, aDirectoryName, aFileTime );
    CZipFileData		mydata		( aGz );

    itsLocalFileHeader	= myheader;
    itsFileData			= mydata;
}

// --------------------------------------------------------------------
// public:	Destructor
// --------------------------------------------------------------------
CZipFileBlock::~CZipFileBlock	() {
	CZipFileBlock::Free();
}

// --------------------------------------------------------------------
// public:	Copy constructor and assignment operator
// --------------------------------------------------------------------
CZipFileBlock::CZipFileBlock( const CZipFileBlock & aC ) {
	CZipFileBlock::Clear();
    if	( aC.LocalFileName() ) {
		itsLocalFileName	= new char [ ::strlen( aC.LocalFileName() ) + 1];
        ::strcpy( itsLocalFileName, aC.LocalFileName() );
    }
	itsLocalFileHeader	= aC.LocalFileHeader();
	itsFileData			= aC.FileData();
	itsDataDescriptor	= aC.DataDescriptor();
    itsFlags			= aC.Flags();
    itsZipStream		= aC.ZipStream();
}

// --------------------------------------------------------------------
CZipFileBlock & CZipFileBlock::operator = ( const CZipFileBlock & aC ) {
	CZipFileBlock::Free();
    if	( aC.LocalFileName() ) {
		itsLocalFileName	= new char [ ::strlen( aC.LocalFileName() ) + 1];
        ::strcpy( itsLocalFileName, aC.LocalFileName() );
    }
	itsLocalFileHeader	= aC.LocalFileHeader();
	itsFileData			= aC.FileData();
	itsDataDescriptor	= aC.DataDescriptor();
    itsFlags			= aC.Flags();
    itsZipStream		= aC.ZipStream();
	return *this;
}

// --------------------------------------------------------------------
// public:	Comparison operators
// --------------------------------------------------------------------
bool CZipFileBlock::operator == ( const CZipFileBlock & aC ) const {
	return	::my_stricmp( itsLocalFileName, aC.LocalFileName() ) == 0;
}

// --------------------------------------------------------------------
bool CZipFileBlock::operator != ( const CZipFileBlock & aC ) const {
	return	::my_stricmp( itsLocalFileName, aC.LocalFileName() ) != 0;
}

// --------------------------------------------------------------------
bool CZipFileBlock::operator == ( const char * aS ) const {
	return	::my_stricmp( itsLocalFileName, aS ) == 0;
}

// --------------------------------------------------------------------
bool CZipFileBlock::operator != ( const char * aS ) const {
	return	::my_stricmp( itsLocalFileName, aS ) != 0;
}

// --------------------------------------------------------------------
// public:	Read this from stream
// --------------------------------------------------------------------
void	CZipFileBlock::Read	( const CZipHeaderSignature & aSig, IZipStream & aReader ) {
	CZipFileBlock::Free();
	itsLocalFileHeader.Read( aSig, aReader );
	itsFileData.Read( itsLocalFileHeader, aReader );
    itsDataDescriptor.Read( itsLocalFileHeader, aReader );
}

// --------------------------------------------------------------------
// public:	Write this into stream
// --------------------------------------------------------------------
void	CZipFileBlock::Write	( IZipStream & aWriter ) {
	// Do we have to compress the data on the fly ?
	itsLocalFileHeader.Write( aWriter );
	itsFileData.Write( itsLocalFileHeader, aWriter );
    itsDataDescriptor.Write( itsLocalFileHeader, aWriter );
}

// --------------------------------------------------------------------
// public:	Check for directory name match
// --------------------------------------------------------------------
bool	CZipFileBlock::Filter ( const char * aFilter ) {
	return	itsLocalFileHeader.FileName() == aFilter;
}

// --------------------------------------------------------------------
// public:	Compress file data into memory
// --------------------------------------------------------------------
void	CZipFileBlock::Compress( CZipNotification * aNotify ) {
	CZipStreamFile	reader( itsLocalFileName );
    reader.OpenRO();
	itsFileData.Cleanup();
    itsFileData.Compress( itsFlags, reader, aNotify );
	itsLocalFileHeader.Crc32( itsFileData.CompressedData().Crc32() );
	itsLocalFileHeader.CompressedSize( itsFileData.CompressedData().Size() );
}

// --------------------------------------------------------------------
// public:	Compress on the fly - no memory in between
// --------------------------------------------------------------------
void	CZipFileBlock::CompressWrite ( IZipStream & aOut, CZipNotification * aNotify ) {
	// The output stream MUST be seekable to do it this way
    if	( aOut.CanSeek() ) {

	    // Remember the offset of current position of output file
    	int4_t	lhoffset = aOut.Tell();

	    // Write the header in there although it does not have the CRC or zipped size
		itsLocalFileHeader.Write( aOut );

		// Create a reader and compress directly into the output
		itsFileData.Cleanup();
		if	( itsZipStream ) {
	    	itsFileData.CompressWrite( itsFlags, *itsZipStream, aOut, aNotify );
        }
        else {
			CZipStreamFile	reader( itsLocalFileName );
   			reader.OpenRO();
	    	itsFileData.CompressWrite( itsFlags, reader, aOut, aNotify );
        }

        // Update local data
		itsLocalFileHeader.Crc32( itsFileData.CompressedData().Crc32() );
		itsLocalFileHeader.CompressedSize( itsFileData.CompressedData().Size() );

        // Remember the current offset so we can come back
        int4_t	zeoffset = aOut.Tell();

        // Seek to the place where our header should go and put it there
        aOut.Seek( lhoffset, zip_stream_seek_set );
		itsLocalFileHeader.Write( aOut );

        // Skip the saved zip data and we should be done here
        aOut.Seek( zeoffset, zip_stream_seek_set );
	    itsDataDescriptor.Write( itsLocalFileHeader, aOut );
	}

    // As the stream was not seekable we have to use the "late details" option
    else {
    	itsLocalFileHeader.SetLateDetails();
		itsLocalFileHeader.Write( aOut );

		// Create a reader and compress directly into the output
		itsFileData.Cleanup();
		if	( itsZipStream ) {
	    	itsFileData.CompressWrite( itsFlags, *itsZipStream, aOut, aNotify );
        }
        else {
			CZipStreamFile	reader( itsLocalFileName );
   			reader.OpenRO();
	    	itsFileData.CompressWrite( itsFlags, reader, aOut, aNotify );
        }

       	// Update local data
		itsLocalFileHeader.Crc32( itsFileData.CompressedData().Crc32() );
		itsLocalFileHeader.CompressedSize( itsFileData.CompressedData().Size() );

		itsDataDescriptor.Crc32( itsFileData.CompressedData().Crc32().Crc32() );
		itsDataDescriptor.CompressedSize( itsFileData.CompressedData().Size() );
		itsDataDescriptor.UncompressedSize( itsLocalFileHeader.UncompressedSize().Value() );

	    itsDataDescriptor.Write( itsLocalFileHeader, aOut );
    }
}

// --------------------------------------------------------------------
// public:	Decompress memory data into local file
// --------------------------------------------------------------------
void	CZipFileBlock::Decompress(  const CZipFlags & aFlags,
                                    const char * aTargetPath ) {

    // Set up target file name
	if	( itsLocalFileName != NULL ) delete [] itsLocalFileName;
	itsLocalFileName	= NULL;
	itsLocalFileName 	= itsLocalFileHeader.TargetPath( aTargetPath, aFlags.ForgetStoredPath() );

	// Create the file
	CZipStreamFile	writer( itsLocalFileName );
    writer.Create();

    // Decompress and read the (possible) trailing file info
    itsFileData.Decompress( itsLocalFileHeader, aFlags, writer );
}

// --------------------------------------------------------------------
// public:	Cleanning it all up
// --------------------------------------------------------------------
void	CZipFileBlock::Free(	void ) {
	if	( itsLocalFileName != NULL ) delete [] itsLocalFileName;
	CZipFileBlock::Clear();
}

// --------------------------------------------------------------------
void	CZipFileBlock::Clear(	void ) {
    itsLocalFileName	= NULL;
    itsZipStream		= NULL;
	itsLocalFileHeader.Cleanup();
	itsFileData.Cleanup();
	itsDataDescriptor.Cleanup();
}

// --------------------------------------------------------------------
void						CZipFileBlock::Cleanup			( void ) { CZipFileBlock::Free(); }
const char *				CZipFileBlock::LocalFileName	( void ) const { return itsLocalFileName; }
const CZipLocalFileHeader &	CZipFileBlock::LocalFileHeader	( void ) const { return itsLocalFileHeader; }
const CZipFileData &		CZipFileBlock::FileData			( void ) const { return itsFileData; }
const CZipDataDescriptor &	CZipFileBlock::DataDescriptor	( void ) const { return itsDataDescriptor; }
const CZipFlags &			CZipFileBlock::Flags			( void ) const { return itsFlags; }
IZipStream *				CZipFileBlock::ZipStream		( void ) const { return itsZipStream; }

// --------------------------------------------------------------------
dword_t	CZipFileBlock::ZippedSize	( void ) const {
	dword_t	zize	= 0;
	zize = zize + itsLocalFileHeader.ZippedSize();
	zize = zize + itsFileData.ZippedSize();
	zize = itsLocalFileHeader.BitFlag().LateDetails() ?  zize + itsDataDescriptor.ZippedSize() : zize;
    return zize;
}

// --------------------------------------------------------------------
// EOF:	CZipFileBlock.cpp
// --------------------------------------------------------------------
