// --------------------------------------------------------------------
// CZipFileData.cpp
// Whatis:	File data
// Authors:	Esko 'Varpu' Ilola	EIL
// History:	EIL	03-MAR-2003		Created	this source
// --------------------------------------------------------------------
#include	"CZipStreamFile.hxx"
#include	"CZipFileData.hxx"
#include	"CDeflate.hxx"
#include	"CError.hxx"

// --------------------------------------------------------------------
// public:	Constructor and destructor
// --------------------------------------------------------------------
CZipFileData::CZipFileData () {
	CZipFileData::Clear();
}

// --------------------------------------------------------------------
CZipFileData::CZipFileData ( const char * aDataBlock, dword_t aDataSize ) {

	// We have to compress it on the fly - shouldn't take too long
    CDeflate		deflator;
	codec_stream_t	args;
    dword_t			cprsize	= aDataSize + ( aDataSize / 999 ) + 24;
	byte_t *		cprdata	= new byte_t [ cprsize ];

	// Set up the compression parameters
	args.next_in	= (byte_t *)aDataBlock;
	args.avail_in	= aDataSize;
	args.total_in	= 0;
	args.next_out	= cprdata;
	args.avail_out	= cprsize;
	args.total_out	= 0;
	args.checksum	= NULL;
	args.complevel	= 8;

	// Then just compress
	deflator.Compress( args );
	deflator.End( args );

	// Do we want to adjust the buffer size ?
    if	( ( cprsize > args.total_out ) && ( args.total_out > 0 ) ) {
    	byte_t * tmp = new byte_t [args.total_out];
        ::memcpy( tmp, cprdata, args.total_out );
        delete [] cprdata;
        cprdata = tmp;
    }

	CZipFileData::Clear();
	itsCompressedData.RawData( cprdata, args.total_out );
}

// --------------------------------------------------------------------
CZipFileData::CZipFileData ( const CGz & aGz ) {
	CZipFileData::Clear();

    if	( aGz.Members().size() < 1 ) {
    	throw CError( "GZ Reference", "Not enough GZ data" );
    }

    CGzMember_lci	member = aGz.Members().begin();

	itsCompressedFile.Data	( (*member).Data().CompressedFileName() );
    itsCompressedFileOffset	= (*member).Data().CompressedFileOffset();
    itsCompressedFileSize	= (*member).Data().CompressedDataSize();
}

// --------------------------------------------------------------------
CZipFileData::~CZipFileData () {
	CZipFileData::Free();
}

// --------------------------------------------------------------------
// public:	Copy constructor and assignment operator
// --------------------------------------------------------------------
CZipFileData::CZipFileData				( const CZipFileData & aC ) {
	CZipFileData::Clear();
	itsCompressedData		= aC.CompressedData();
	itsCompressedFile		= aC.CompressedFile();
	itsCompressedFileOffset	= aC.CompressedFileOffset();
	itsCompressedFileSize	= aC.CompressedFileSize();
}

// --------------------------------------------------------------------
CZipFileData & CZipFileData::operator =	( const CZipFileData & aC ) {
	CZipFileData::Free();
	itsCompressedData		= aC.CompressedData();
	itsCompressedFile		= aC.CompressedFile();
	itsCompressedFileOffset	= aC.CompressedFileOffset();
	itsCompressedFileSize	= aC.CompressedFileSize();
	return *this;
}

// --------------------------------------------------------------------
// public:	Interfaces
// --------------------------------------------------------------------
void	CZipFileData::Read		(	const CZipLocalFileHeader &	aHd,
									IZipStream &				aIn ) {
	CZipFileData::Free();

    // If this is a seekable stream we can just mark the file and place
    // for later retrieval of the data
    if	( aIn.CanSeek() ) {
		itsCompressedFile.Data( aIn.Name() );
		itsCompressedFileOffset = aIn.Tell();
        itsCompressedFileSize	= aHd.CompressedSize().Value();
		itsCompressedData.Cleanup();
        aIn.Seek( itsCompressedFileSize,  zip_stream_seek_offset );
	}

    else {
		itsCompressedData.Read( aIn, aHd.CompressedSize().Value() );
    }
}

// --------------------------------------------------------------------
void	CZipFileData::Write		(	const CZipLocalFileHeader &	aHd,
									IZipStream &				aOut ) {
    // If this was a seekable stream we just copy the data to the other stream
    if		( itsCompressedFileSize > 0 ) {
    	CZipStreamFile	myfile( (const char *)itsCompressedFile.Data() );
        myfile.OpenRO();
        myfile.Seek( itsCompressedFileOffset, zip_stream_seek_set );
        aOut.Copy( myfile, itsCompressedFileSize );
    }

    // We have the data in the buffer - just write it out
    else if	 ( itsCompressedData.Size() > 0 ) {
		itsCompressedData.Write( aOut );
    }
}

// --------------------------------------------------------------------
// public:	Compress the data coming from in-stream into out-stream
// --------------------------------------------------------------------
void	CZipFileData::CompressWrite(const CZipFlags &			aFg,
                                    IZipStream & 				aIn,
                                    IZipStream & 				aOut,
                                    CZipNotification *			aNotify ) {
    itsCompressedData.Cleanup();
    itsCompressedData.Compress( aFg, aIn, aOut, aNotify );
}

// --------------------------------------------------------------------
// public:	Compress the data coming from in-stream into local buffer
// --------------------------------------------------------------------
void	CZipFileData::Compress	(	const CZipFlags &			aFg,
                                    IZipStream & 				aIn,
                                    CZipNotification *			aNotify ) {
    itsCompressedData.Cleanup();
    itsCompressedData.Compress( aFg, aIn, aNotify );
}

// --------------------------------------------------------------------
void	CZipFileData::Decompress(	const CZipLocalFileHeader &	aHd,
									const CZipFlags &			aFg,
                                    IZipStream & 				aOut ) {
}

// --------------------------------------------------------------------
void	CZipFileData::Free( void ) {
	CZipFileData::Clear();
}

// --------------------------------------------------------------------
void	CZipFileData::Clear( void ) {
	itsCompressedData.Cleanup();
	itsCompressedFile.Cleanup();
	itsCompressedFileOffset = 0;
	itsCompressedFileSize	= 0;
}

// --------------------------------------------------------------------
void	CZipFileData::Cleanup ( void ) {
	CZipFileData::Free();
}

// --------------------------------------------------------------------
const CZipDataField &	CZipFileData::CompressedData( void ) const {
	return itsCompressedData;
}

// --------------------------------------------------------------------
const CZipDataField	&	CZipFileData::CompressedFile( void ) const {
	return itsCompressedFile;
}

// --------------------------------------------------------------------
dword_t					CZipFileData::CompressedFileOffset( void ) const {
	return itsCompressedFileOffset;
}

// --------------------------------------------------------------------
dword_t					CZipFileData::CompressedFileSize( void ) const {
	return itsCompressedFileSize;
}

// --------------------------------------------------------------------
dword_t	CZipFileData::ZippedSize	( void ) const {
	return	  itsCompressedFileSize > itsCompressedData.Size()
    		? itsCompressedFileSize
            : itsCompressedData.Size();
}

// --------------------------------------------------------------------
// EOF:	CZipFileData.cpp
// --------------------------------------------------------------------
