// --------------------------------------------------------------------
// CZipFileHeader.cpp
// Whatis:	File header in central directory
// Authors:	Esko 'Varpu' Ilola	EIL
// History:	EIL	02-MAR-2003		Created	this source
// --------------------------------------------------------------------
#include	"CZipFileHeader.hxx"
#include	"CError.hxx"

// --------------------------------------------------------------------
// public:	Constructor and destructor
// --------------------------------------------------------------------
CZipFileHeader::CZipFileHeader () {
	CZipFileHeader::Clear();
}

// --------------------------------------------------------------------
CZipFileHeader::CZipFileHeader ( const CZipFlags & aFg, const char * aLocalFile, const char * aDirName ) {
	CZipFileHeader::Clear();
    struct stat	mystat;
    if	( ::stat( aLocalFile, &mystat ) ) {
    	throw CError( aLocalFile, ::strerror( errno ) );
    }

	itsSignature = zip_signature_centralfile;
	itsCreatorVersion.SetVersion			( 2, 0, zip_data_compat_msdos );
	itsExtractVersion.SetVersion			( 2, 0, zip_data_compat_msdos );
	itsBitFlag.DeflateType					( aFg.DeflateType() );
	itsCompressionMethod.CompressionMethod	( zip_compression_method_deflated );
	itsLastMdificationTime					= mystat.st_mtime;
	itsCrc32								= (dword_t)0;
	itsCompressedSize						= (dword_t)0;
	itsUncompressedSize						= mystat.st_size;
	itsFilenameLength						= (word_t)::strlen( aDirName );
	itsDiskNumberStart						= 0;
	itsInternalFileAttributes				= 0; // TODO
	itsExternalFileAttributes	   			= 0; // TODO
	itsLocalHeaderOffset					= 0; // TODO
	itsFileName.Data						( aDirName );
}

// --------------------------------------------------------------------
CZipFileHeader::CZipFileHeader ( const CZipFlags & aFg, const CZipFileBlock & aFb, const char * aDirName ) {
	CZipFileHeader::Clear();

	itsSignature = zip_signature_centralfile;
	itsCreatorVersion.SetVersion			( 2, 0, zip_data_compat_msdos );
	itsExtractVersion.SetVersion			( 2, 0, zip_data_compat_msdos );
	itsBitFlag.DeflateType					( aFg.DeflateType() );
	itsCompressionMethod.CompressionMethod	( zip_compression_method_deflated );
	itsLastMdificationTime					= aFb.LocalFileHeader().LastModificationTime();
	itsCrc32								= aFb.LocalFileHeader().Crc32();
	itsCompressedSize						= aFb.LocalFileHeader().CompressedSize();
	itsUncompressedSize						= aFb.LocalFileHeader().UncompressedSize();
	itsFilenameLength						= (word_t)::strlen( aDirName );
	itsDiskNumberStart						= 0;
	itsInternalFileAttributes				= 0; // TODO
	itsExternalFileAttributes	   			= 0; // TODO
	itsLocalHeaderOffset					= 0; // TODO
	itsFileName.Data						( aDirName );
}

// --------------------------------------------------------------------
CZipFileHeader::CZipFileHeader ( const CGz & aGz, const char * aDirName, time_t aTime ) {
	CZipFileHeader::Clear();

	if	( aGz.Members().size() < 1 ) {
    	throw CError( "GZ Reference", "Not enough GZ data" );
    }

    CGzMember_lci	member = aGz.Members().begin();

	itsSignature = zip_signature_centralfile;
	itsCreatorVersion.SetVersion			( 2, 0, zip_data_compat_msdos );
	itsExtractVersion.SetVersion			( 2, 0, zip_data_compat_msdos );
	itsBitFlag.DeflateType					( zip_deflate_type_maximum );
	itsCompressionMethod.CompressionMethod	( zip_compression_method_deflated );
	itsLastMdificationTime					= aTime;
	itsCrc32								= (*member).Data().Crc32();
	itsCompressedSize						= (*member).Data().CompressedDataSize();
	itsUncompressedSize						= (*member).Data().Isize();
	itsFilenameLength						= (word_t)::strlen( aDirName );
	itsDiskNumberStart						= 0;
	itsInternalFileAttributes				= 0; // TODO
	itsExternalFileAttributes	   			= 0; // TODO
	itsLocalHeaderOffset					= 0; // TODO
	itsFileName.Data						( aDirName );
}

// --------------------------------------------------------------------
CZipFileHeader::~CZipFileHeader () {
	CZipFileHeader::Free();
}

// --------------------------------------------------------------------
// public:	Copy constructor and assignment operator
// --------------------------------------------------------------------
CZipFileHeader::CZipFileHeader				( const CZipFileHeader & aC ) {
	CZipFileHeader::Clear();
	itsSignature				= aC.Signature();
	itsCreatorVersion			= aC.CreatorVersion();
	itsExtractVersion			= aC.ExtractVersion();
	itsBitFlag					= aC.BitFlag();
	itsCompressionMethod	 	= aC.CompressionMethod();
	itsLastMdificationTime		= aC.LastMdificationTime();
	itsCrc32					= aC.Crc32();
	itsCompressedSize			= aC.CompressedSize();
	itsUncompressedSize			= aC.UncompressedSize();
	itsFilenameLength			= aC.FilenameLength();
	itsExtraFieldLength			= aC.ExtraFieldLength();
	itsCommentLength			= aC.CommentLength();
	itsDiskNumberStart			= aC.DiskNumberStart();
	itsInternalFileAttributes	= aC.InternalFileAttributes();
	itsExternalFileAttributes	= aC.ExternalFileAttributes();
	itsLocalHeaderOffset		= aC.LocalHeaderOffset();
	itsFileName					= aC.FileName();
	itsExtraField				= aC.ExtraField();
	itsComment					= aC.Comment();
}

// --------------------------------------------------------------------
CZipFileHeader & CZipFileHeader::operator =	( const CZipFileHeader & aC ) {
	CZipFileHeader::Free();
	itsSignature				= aC.Signature();
	itsCreatorVersion			= aC.CreatorVersion();
	itsExtractVersion			= aC.ExtractVersion();
	itsBitFlag					= aC.BitFlag();
	itsCompressionMethod	 	= aC.CompressionMethod();
	itsLastMdificationTime		= aC.LastMdificationTime();
	itsCrc32					= aC.Crc32();
	itsCompressedSize			= aC.CompressedSize();
	itsUncompressedSize			= aC.UncompressedSize();
	itsFilenameLength			= aC.FilenameLength();
	itsExtraFieldLength			= aC.ExtraFieldLength();
	itsCommentLength			= aC.CommentLength();
	itsDiskNumberStart			= aC.DiskNumberStart();
	itsInternalFileAttributes	= aC.InternalFileAttributes();
	itsExternalFileAttributes	= aC.ExternalFileAttributes();
	itsLocalHeaderOffset		= aC.LocalHeaderOffset();
	itsFileName					= aC.FileName();
	itsExtraField				= aC.ExtraField();
	itsComment					= aC.Comment();
	return *this;
}

// --------------------------------------------------------------------
// public:	Interfaces
// --------------------------------------------------------------------
void	CZipFileHeader::Read 	(	const CZipHeaderSignature & aSig,
                                    IZipStream & 				aReader ) {
	CZipFileHeader::Free();
	itsSignature = aSig;
	itsCreatorVersion.Read			( aReader );
	itsExtractVersion.Read			( aReader );
	itsBitFlag.Read					( aReader );
	itsCompressionMethod.Read		( aReader );
	itsLastMdificationTime.Read		( aReader );
	itsCrc32.Read					( aReader );
	itsCompressedSize.Read			( aReader );
	itsUncompressedSize.Read		( aReader );
	itsFilenameLength.Read			( aReader );
	itsExtraFieldLength.Read		( aReader );
	itsCommentLength.Read			( aReader );
	itsDiskNumberStart.Read			( aReader );
	itsInternalFileAttributes.Read	( aReader );
	itsExternalFileAttributes.Read	( aReader );
	itsLocalHeaderOffset.Read		( aReader );
	itsFileName.Read				( aReader, itsFilenameLength.Value() );
	itsExtraField.Read				( aReader, itsExtraFieldLength.Value() );
	itsComment.Read					( aReader, itsCommentLength.Value() );
}

// --------------------------------------------------------------------
void	CZipFileHeader::Write	( CZipFileBlock & aBlock, IZipStream & aWriter ) {
	itsSignature = zip_signature_centralfile;	// Just in case
	itsSignature.Write				( aWriter );
	itsCreatorVersion.Write			( aWriter );
	itsExtractVersion.Write			( aWriter );
	itsBitFlag.Write				( aWriter );
	itsCompressionMethod.Write		( aWriter );
	itsLastMdificationTime.Write	( aWriter );
	itsCrc32.Write					( aWriter );
	itsCompressedSize.Write			( aWriter );
	itsUncompressedSize.Write		( aWriter );
	itsFilenameLength.Write			( aWriter );
	itsExtraFieldLength.Write		( aWriter );
	itsCommentLength.Write			( aWriter );
	itsDiskNumberStart.Write		( aWriter );
	itsInternalFileAttributes.Write	( aWriter );
	itsExternalFileAttributes.Write	( aWriter );
	itsLocalHeaderOffset.Write		( aWriter );
	itsFileName.Write				( aWriter );
	itsExtraField.Write				( aWriter );
	itsComment.Write				( aWriter );
}

// --------------------------------------------------------------------
void	CZipFileHeader::Free( void ) {
	CZipFileHeader::Clear();
}

// --------------------------------------------------------------------
void	CZipFileHeader::Clear( void ) {
	itsSignature.Cleanup();
	itsCreatorVersion.Cleanup();
	itsExtractVersion.Cleanup();
	itsBitFlag.Cleanup();
	itsCompressionMethod.Cleanup();
	itsLastMdificationTime.Cleanup();
	itsCrc32.Cleanup();
	itsCompressedSize.Cleanup();
	itsUncompressedSize.Cleanup();
	itsFilenameLength.Cleanup();
	itsExtraFieldLength.Cleanup();
	itsCommentLength.Cleanup();
	itsDiskNumberStart.Cleanup();
	itsInternalFileAttributes.Cleanup();
	itsExternalFileAttributes.Cleanup();
	itsLocalHeaderOffset.Cleanup();
	itsFileName.Cleanup();
	itsExtraField.Cleanup();
	itsComment.Cleanup();
}

// --------------------------------------------------------------------
void							CZipFileHeader::Cleanup					( void ) { CZipFileHeader::Free(); }
const CZipHeaderSignature &		CZipFileHeader::Signature				( void ) const { return itsSignature; }
const CZipVersion &				CZipFileHeader::CreatorVersion			( void ) const { return itsCreatorVersion; }
const CZipVersion &				CZipFileHeader::ExtractVersion			( void ) const { return itsExtractVersion; }
const CZipHeaderBitFlag	&		CZipFileHeader::BitFlag					( void ) const { return itsBitFlag; }
const CZipCompressionMethod &	CZipFileHeader::CompressionMethod		( void ) const { return itsCompressionMethod; }
const CZipDateTime &			CZipFileHeader::LastMdificationTime		( void ) const { return itsLastMdificationTime; }
const CZipCrc32	&				CZipFileHeader::Crc32					( void ) const { return itsCrc32; }
const CZipDword	&				CZipFileHeader::CompressedSize			( void ) const { return itsCompressedSize; }
const CZipDword	&				CZipFileHeader::UncompressedSize		( void ) const { return itsUncompressedSize; }
const CZipWord &				CZipFileHeader::FilenameLength			( void ) const { return itsFilenameLength; }
const CZipWord &				CZipFileHeader::ExtraFieldLength		( void ) const { return itsExtraFieldLength; }
const CZipWord &				CZipFileHeader::CommentLength			( void ) const { return itsCommentLength; }
const CZipWord &				CZipFileHeader::DiskNumberStart			( void ) const { return itsDiskNumberStart; }
const CZipInternalAttr &		CZipFileHeader::InternalFileAttributes	( void ) const { return itsInternalFileAttributes; }
const CZipExternalAttr &		CZipFileHeader::ExternalFileAttributes	( void ) const { return itsExternalFileAttributes; }
const CZipDword	&				CZipFileHeader::LocalHeaderOffset		( void ) const { return itsLocalHeaderOffset; }
const CZipDataField	&			CZipFileHeader::FileName				( void ) const { return itsFileName; }
const CZipDataField	&			CZipFileHeader::ExtraField				( void ) const { return itsExtraField; }
const CZipDataField	&			CZipFileHeader::Comment					( void ) const { return itsComment; }

void	CZipFileHeader::LocalHeaderOffset	( dword_t aV ) { itsLocalHeaderOffset = aV; }
void	CZipFileHeader::CompressedSize		( dword_t aV ) { itsCompressedSize = aV; }
void	CZipFileHeader::Crc32				( dword_t aV ) { itsCrc32 = aV; }

// --------------------------------------------------------------------
dword_t	CZipFileHeader::ZippedSize	( void ) const {
	dword_t	zize	= 0;
	zize = zize + itsSignature.ZippedSize();
	zize = zize + itsCreatorVersion.ZippedSize();
	zize = zize + itsExtractVersion.ZippedSize();
	zize = zize + itsBitFlag.ZippedSize();
	zize = zize + itsCompressionMethod.ZippedSize();
	zize = zize + itsLastMdificationTime.ZippedSize();
	zize = zize + itsCrc32.ZippedSize();
	zize = zize + itsCompressedSize.ZippedSize();
	zize = zize + itsUncompressedSize.ZippedSize();
	zize = zize + itsFilenameLength.ZippedSize();
	zize = zize + itsExtraFieldLength.ZippedSize();
	zize = zize + itsCommentLength.ZippedSize();
	zize = zize + itsDiskNumberStart.ZippedSize();
	zize = zize + itsInternalFileAttributes.ZippedSize();
	zize = zize + itsExternalFileAttributes.ZippedSize();
	zize = zize + itsLocalHeaderOffset.ZippedSize();
	zize = zize + itsFileName.ZippedSize();
	zize = zize + itsExtraField.ZippedSize();
	zize = zize + itsComment.ZippedSize();
    return zize;
}

// --------------------------------------------------------------------
bool	CZipFileHeader::Filter	( const char * aF ) const {
	return	itsFileName == aF;
}

// --------------------------------------------------------------------
void	CZipFileHeader::AddFileComment	( const char * aComment ) {
	itsCommentLength.Cleanup();
    itsComment.Cleanup();
	if	( aComment ) {
    	if	( *aComment ) {
			itsComment.Data( aComment );
		    itsCommentLength = (word_t)::strlen( aComment );
        }
    }
}

// --------------------------------------------------------------------
// EOF:	CZipFileHeader.cpp
// --------------------------------------------------------------------
