// --------------------------------------------------------------------
// CZipHeaderSignature.cpp
// Whatis:	File data
// Authors:	Esko 'Varpu' Ilola	EIL
// History:	EIL	24-FEB-2003		Created	this source
// --------------------------------------------------------------------
#include	"CZipHeaderSignature.hxx"
#include	"CError.hxx"

// --------------------------------------------------------------------
// local:	Signatures for comparison and setting
// --------------------------------------------------------------------
static	const byte_t	my_signature_set[5][4] = {
	{ 0x50, 0x4B, 0x03, 0x04 },	// 0 local file header signature
	{ 0x50, 0x4B, 0x01, 0x02 },	// 1 central file header signature
	{ 0x50, 0x4B, 0x05, 0x05 },	// 2 digital signature
	{ 0x50, 0x4B, 0x05, 0x06 },	// 3 end of central dir signature
	{ 0x00, 0x00, 0x00, 0x00 }  // 4 unknown
};

// --------------------------------------------------------------------
// public:	Constructors
// --------------------------------------------------------------------
CZipHeaderSignature::CZipHeaderSignature	() {
	CZipHeaderSignature::Clear();
}

// --------------------------------------------------------------------
CZipHeaderSignature::CZipHeaderSignature	( zip_signature_t aType ) {
	CZipHeaderSignature::Clear();
	CZipHeaderSignature::Type( aType );
}

// --------------------------------------------------------------------
CZipHeaderSignature::~CZipHeaderSignature	() {
	CZipHeaderSignature::Free();
}

// --------------------------------------------------------------------
// public:	Copy constructor and assignment operators
// --------------------------------------------------------------------
CZipHeaderSignature::CZipHeaderSignature( const CZipHeaderSignature & aC ) {
	CZipHeaderSignature::Clear();
	CZipHeaderSignature::Type( aC.Type() );
}

// --------------------------------------------------------------------
CZipHeaderSignature & CZipHeaderSignature::operator = ( const CZipHeaderSignature & aC ) {
	CZipHeaderSignature::Free();
	CZipHeaderSignature::Type( aC.Type() );
	return *this;
}

// --------------------------------------------------------------------
CZipHeaderSignature & CZipHeaderSignature::operator = ( zip_signature_t aType ) {
	CZipHeaderSignature::Free();
	CZipHeaderSignature::Type( aType );
	return *this;
}

// --------------------------------------------------------------------
// public:	Comparison operators
// --------------------------------------------------------------------
bool CZipHeaderSignature::operator == ( const CZipHeaderSignature & aC ) const {
	return	itsType == aC.Type();
}

// --------------------------------------------------------------------
bool CZipHeaderSignature::operator != ( const CZipHeaderSignature & aC ) const {
	return	itsType != aC.Type();
}

// --------------------------------------------------------------------
bool CZipHeaderSignature::operator == ( zip_signature_t aT ) const {
	return	itsType == aT;
}

// --------------------------------------------------------------------
bool CZipHeaderSignature::operator != ( zip_signature_t aT ) const {
	return	itsType != aT;
}

// --------------------------------------------------------------------
// public:	Interface
// --------------------------------------------------------------------
void	CZipHeaderSignature::Read	( IZipStream & aReader ) {
	CZipHeaderSignature::Free();
	dword_t	rsz = aReader.Read( itsSignature, 4 );
    if	( rsz < 4 ) {
    	throw CError( aReader.Name(), "Not enough data" );
    }

	int	six	= 0;
    while	( my_signature_set[six][0] != 0 ) {
    	if	( ! ::memcmp( my_signature_set[six], itsSignature, 4 ) ) {
        	break;
        }
        else {
        	six++;
        }
    }

    switch	( six ) {
    	case	0:	CZipHeaderSignature::Type( zip_signature_localfile );	break;
    	case	1:	CZipHeaderSignature::Type( zip_signature_centralfile );	break;
    	case	2:	CZipHeaderSignature::Type( zip_signature_digital );		break;
    	case	3:	CZipHeaderSignature::Type( zip_signature_eocd );		break;
    	case	4:	CZipHeaderSignature::Type( zip_signature_unknown );		break;
		default:    throw CError( "Internal program error" );
    }
}

// --------------------------------------------------------------------
void	CZipHeaderSignature::Write	( IZipStream & aWriter ) {
	aWriter.Write( itsSignature, 4 );
}

// --------------------------------------------------------------------
void	CZipHeaderSignature::Type	( zip_signature_t aType ) {
	switch	( aType ) {
		case	zip_signature_localfile		: ::memcpy( itsSignature, my_signature_set[0], 4 );	break;
		case	zip_signature_centralfile	: ::memcpy( itsSignature, my_signature_set[1], 4 );	break;
		case	zip_signature_digital		: ::memcpy( itsSignature, my_signature_set[2], 4 );	break;
		case	zip_signature_eocd			: ::memcpy( itsSignature, my_signature_set[3], 4 );	break;
		case	zip_signature_unknown		: ::memcpy( itsSignature, my_signature_set[4], 4 );	break;
		default								: throw CError( "Internal program error" );
    }
	itsType = aType;
}

// --------------------------------------------------------------------
void	CZipHeaderSignature::Free	( void ) {
	CZipHeaderSignature::Clear();
}

// --------------------------------------------------------------------
void	CZipHeaderSignature::Clear  ( void ) {
	CZipHeaderSignature::Type( zip_signature_unknown );
}

// --------------------------------------------------------------------
void			CZipHeaderSignature::Cleanup( void ) { CZipHeaderSignature::Free(); }
zip_signature_t	CZipHeaderSignature::Type	( void ) const { return itsType; }

// --------------------------------------------------------------------
dword_t	CZipHeaderSignature::ZippedSize	( void ) const {
	return	4;
}

// --------------------------------------------------------------------
// EOF:	CZipHeaderSignature.cpp
// --------------------------------------------------------------------
