// --------------------------------------------------------------------
// CZipLocalFileHeader.hxx
// Whatis:	Local file header
// Authors:	Esko 'Varpu' Ilola	EIL
// History:	EIL	24-FEB-2003		Created	this source
// --------------------------------------------------------------------
#include	"CZipLocalFileHeader.hxx"
#include	"CError.hxx"

// --------------------------------------------------------------------
// public:	Constructor and destructor
// --------------------------------------------------------------------
CZipLocalFileHeader::CZipLocalFileHeader () {
	CZipLocalFileHeader::Clear();
}

// --------------------------------------------------------------------
CZipLocalFileHeader::CZipLocalFileHeader ( const CZipFlags & aFg, IZipStream * aStream, dword_t aSize, const char * aFileName ) {
	CZipLocalFileHeader::Clear();

    itsSignature							= zip_signature_localfile;
    itsExtractVersion.SetVersion			( 2, 0, zip_data_compat_msdos );
	itsBitFlag.DeflateType					( aFg.DeflateType() );
    itsBitFlag.LateDetails					( false );
	itsCompressionMethod.CompressionMethod	( zip_compression_method_deflated );
    itsLastModificationTime					= ::time( NULL );
    itsCrc32								= (dword_t)0;
    itsCompressedSize						= (dword_t)0;
    itsUncompressedSize						= aSize;
    itsFilenameLength						= (word_t)::strlen( aFileName );
	itsFileName.Data						( aFileName );
}

// --------------------------------------------------------------------
CZipLocalFileHeader::CZipLocalFileHeader ( const CZipFlags & aFg, const char * aLocalFile, const char * aFileName ) {
	struct stat	mystat;

	CZipLocalFileHeader::Clear();

    if	( ::stat( aLocalFile, &mystat ) ) {
    	throw CError( aLocalFile, ::strerror( errno ) );
    }

    itsSignature							= zip_signature_localfile;
    itsExtractVersion.SetVersion			( 2, 0, zip_data_compat_msdos );
	itsBitFlag.DeflateType					( aFg.DeflateType() );
    itsBitFlag.LateDetails					( false );
	itsCompressionMethod.CompressionMethod	( zip_compression_method_deflated );
    itsLastModificationTime					= mystat.st_mtime;
    itsCrc32								= (dword_t)0;
    itsCompressedSize						= (dword_t)0;
    itsUncompressedSize						= mystat.st_size;
    itsFilenameLength						= (word_t)::strlen( aFileName );
	itsFileName.Data						( aFileName );
}

// --------------------------------------------------------------------
CZipLocalFileHeader::CZipLocalFileHeader ( const CZipFlags & aFg, const char * aDataBlock, dword_t aDataLen, const char * aFileName ) {
	CCrc32	crc;

	CZipLocalFileHeader::Clear();
    crc.Calculate( aDataBlock, aDataLen );

    itsSignature							= zip_signature_localfile;
    itsExtractVersion.SetVersion			( 2, 0, zip_data_compat_msdos );
	itsBitFlag.DeflateType					( aFg.DeflateType() );
    itsBitFlag.LateDetails					( false );
	itsCompressionMethod.CompressionMethod	( zip_compression_method_deflated );
    itsLastModificationTime					= ::time( NULL );
    itsCrc32								= crc.Crc32();
    itsCompressedSize						= (dword_t)0;
    itsUncompressedSize						= aDataLen;
    itsFilenameLength						= (word_t)::strlen( aFileName );
	itsFileName.Data						( aFileName );
}

// --------------------------------------------------------------------
CZipLocalFileHeader::CZipLocalFileHeader ( const CGz & aGz, const char * aFileName, time_t aTime ) {
	CZipLocalFileHeader::Clear();

    if	( aGz.Members().size() < 1 ) {
    	throw CError( "GZ Reference", "Not enough GZ data" );
    }

    CGzMember_lci	member = aGz.Members().begin();

    itsSignature							= zip_signature_localfile;
    itsExtractVersion.SetVersion			( 2, 0, zip_data_compat_msdos );
	itsBitFlag.DeflateType					( zip_deflate_type_maximum );
    itsBitFlag.LateDetails					( false );
	itsCompressionMethod.CompressionMethod	( zip_compression_method_deflated );
    itsLastModificationTime					= aTime;
    itsCrc32								= (*member).Data().Crc32();
    itsCompressedSize						= (*member).Data().CompressedDataSize();
    itsUncompressedSize						= (*member).Data().Isize();
    itsFilenameLength						= (word_t)::strlen( aFileName );
	itsFileName.Data						( aFileName );
}

// --------------------------------------------------------------------
CZipLocalFileHeader::~CZipLocalFileHeader () {
	CZipLocalFileHeader::Free();
}

// --------------------------------------------------------------------
// public:	Copy constructor and assignment operator
// --------------------------------------------------------------------
CZipLocalFileHeader::CZipLocalFileHeader ( const CZipLocalFileHeader & aC ) {
	CZipLocalFileHeader::Clear();
	itsSignature			= aC.Signature();
	itsExtractVersion		= aC.ExtractVersion();
	itsBitFlag				= aC.BitFlag();
	itsCompressionMethod	= aC.CompressionMethod();
	itsLastModificationTime	= aC.LastModificationTime();
	itsCrc32				= aC.Crc32();
	itsCompressedSize		= aC.CompressedSize();
	itsUncompressedSize		= aC.UncompressedSize();
	itsFilenameLength		= aC.FilenameLength();
	itsExtraFieldLength		= aC.ExtraFieldLength();
	itsFileName				= aC.FileName();
	itsExtraField			= aC.ExtraField();
}

// --------------------------------------------------------------------
CZipLocalFileHeader & CZipLocalFileHeader::operator =( const CZipLocalFileHeader & aC ) {
	CZipLocalFileHeader::Free();
	itsSignature			= aC.Signature();
	itsExtractVersion		= aC.ExtractVersion();
	itsBitFlag				= aC.BitFlag();
	itsCompressionMethod	= aC.CompressionMethod();
	itsLastModificationTime	= aC.LastModificationTime();
	itsCrc32				= aC.Crc32();
	itsCompressedSize		= aC.CompressedSize();
	itsUncompressedSize		= aC.UncompressedSize();
	itsFilenameLength		= aC.FilenameLength();
	itsExtraFieldLength		= aC.ExtraFieldLength();
	itsFileName				= aC.FileName();
	itsExtraField			= aC.ExtraField();
	return *this;
}

// --------------------------------------------------------------------
// public:	Interfaces
// --------------------------------------------------------------------
void	CZipLocalFileHeader::Read	( const CZipHeaderSignature & aSig, IZipStream & aReader ) {
	CZipLocalFileHeader::Free();
	itsSignature = aSig;
    itsExtractVersion.Read		( aReader );
    itsBitFlag.Read				( aReader );
    itsCompressionMethod.Read	( aReader );
    itsLastModificationTime.Read( aReader );
    itsCrc32.Read				( aReader );
    itsCompressedSize.Read		( aReader );
    itsUncompressedSize.Read	( aReader );
    itsFilenameLength.Read		( aReader );
    itsExtraFieldLength.Read	( aReader );
    itsFileName.Read			( aReader, itsFilenameLength.Value() );
    itsExtraField.Read			( aReader, itsExtraFieldLength.Value() );
}

// --------------------------------------------------------------------
void	CZipLocalFileHeader::Write	( IZipStream & aWriter ) {
	itsSignature = zip_signature_localfile;	// To be sure !
    itsSignature.Write          ( aWriter );
    itsExtractVersion.Write		( aWriter );
    itsBitFlag.Write			( aWriter );
    itsCompressionMethod.Write	( aWriter );
    itsLastModificationTime.Write( aWriter );
    itsCrc32.Write				( aWriter );
    itsCompressedSize.Write		( aWriter );
    itsUncompressedSize.Write	( aWriter );
    itsFilenameLength.Write		( aWriter );
    itsExtraFieldLength.Write	( aWriter );
    itsFileName.Write			( aWriter );
    itsExtraField.Write			( aWriter );
}

// --------------------------------------------------------------------
void	CZipLocalFileHeader::FileName	( const char * aFileName ) {
	itsFilenameLength 	= (word_t)::strlen( aFileName );
    itsFileName.Data( aFileName );
}

// --------------------------------------------------------------------
void	CZipLocalFileHeader::UncompressedSize		( dword_t aSize ) {
	itsUncompressedSize	= aSize;
}

// --------------------------------------------------------------------
void	CZipLocalFileHeader::LastModificationTime	( time_t aTime ) {
	itsLastModificationTime = aTime;
}

// --------------------------------------------------------------------
void	CZipLocalFileHeader::Free ( void ) {
	CZipLocalFileHeader::Clear();
}

// --------------------------------------------------------------------
void	CZipLocalFileHeader::Clear ( void ) {
	itsSignature.Cleanup();
	itsExtractVersion.Cleanup();
	itsBitFlag.Cleanup();
	itsCompressionMethod.Cleanup();
	itsLastModificationTime.Cleanup();
	itsCrc32.Cleanup();
	itsCompressedSize.Cleanup();
	itsUncompressedSize.Cleanup();
	itsFilenameLength.Cleanup();
	itsExtraFieldLength.Cleanup();
	itsFileName.Cleanup();
	itsExtraField.Cleanup();
}

// --------------------------------------------------------------------
void							CZipLocalFileHeader::Cleanup			( void ) { CZipLocalFileHeader::Free(); }
const CZipHeaderSignature &		CZipLocalFileHeader::Signature			( void ) const { return itsSignature;		   }
const CZipVersion &				CZipLocalFileHeader::ExtractVersion		( void ) const { return itsExtractVersion;     }
const CZipHeaderBitFlag &		CZipLocalFileHeader::BitFlag			( void ) const { return itsBitFlag;            }
const CZipCompressionMethod &	CZipLocalFileHeader::CompressionMethod	( void ) const { return itsCompressionMethod;  }
const CZipDateTime &			CZipLocalFileHeader::LastModificationTime( void ) const { return itsLastModificationTime;}
const CZipCrc32 &				CZipLocalFileHeader::Crc32				( void ) const { return itsCrc32;              }
const CZipDword &				CZipLocalFileHeader::CompressedSize		( void ) const { return itsCompressedSize;     }
const CZipDword &				CZipLocalFileHeader::UncompressedSize	( void ) const { return itsUncompressedSize;   }
const CZipWord &				CZipLocalFileHeader::FilenameLength		( void ) const { return itsFilenameLength;     }
const CZipWord &				CZipLocalFileHeader::ExtraFieldLength	( void ) const { return itsExtraFieldLength;   }
const CZipDataField &			CZipLocalFileHeader::FileName			( void ) const { return itsFileName;           }
const CZipDataField &			CZipLocalFileHeader::ExtraField			( void ) const { return itsExtraField;         }

void	CZipLocalFileHeader::CompressedSize	( dword_t aV ) { itsCompressedSize = aV;  }
void	CZipLocalFileHeader::Crc32			( const CZipCrc32 & aC ) { itsCrc32 = aC; }

// --------------------------------------------------------------------
dword_t	CZipLocalFileHeader::ZippedSize	( void ) const {
	dword_t	zize	= 0;
	zize = zize + itsSignature.ZippedSize();
	zize = zize + itsExtractVersion.ZippedSize();
	zize = zize + itsBitFlag.ZippedSize();
	zize = zize + itsCompressionMethod.ZippedSize();
	zize = zize + itsLastModificationTime.ZippedSize();
	zize = zize + itsCrc32.ZippedSize();
	zize = zize + itsCompressedSize.ZippedSize();
	zize = zize + itsUncompressedSize.ZippedSize();
	zize = zize + itsFilenameLength.ZippedSize();
	zize = zize + itsExtraFieldLength.ZippedSize();
	zize = zize + itsFileName.ZippedSize();
	zize = zize + itsExtraField.ZippedSize();
    return	zize;
}

// --------------------------------------------------------------------
// public:	Generate a target path for decompress
// --------------------------------------------------------------------
char *	CZipLocalFileHeader::TargetPath ( const char * aTarget, bool aFlatten ) {
	return	"";
}

// --------------------------------------------------------------------
void	CZipLocalFileHeader::SetLateDetails ( void ) {
	// Set up the late details flag for those out-streams than can't seek
    itsBitFlag.LateDetails( true );
}

// --------------------------------------------------------------------
// EOF:	CZipLocalFileHeader.cpp
// --------------------------------------------------------------------
