// --------------------------------------------------------------------
// Preview.cxx
// Whatis:  CGI for previewing files
// CGI arguments:
//
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 14-JUN-2002     Created this source
// --------------------------------------------------------------------
#include    "CError.hxx"
#include    "CStorage.hxx"
#include    "CCache.hxx"
#include    "CTablePref.hxx"
#include    "CTableFile.hxx"
#include    "CTablePack.hxx"
#include    "CTableAuth.hxx"
#include    "CTableType.hxx"
#include    "CTableRule.hxx"
#include    "CTableXref.hxx"
#include	"CCgiArgs.hxx"
#include	"CMySqlWhere.hxx"
#include	"CUnUtils.hxx"
#include	"CUnLevelInfo.hxx"
#include	"CUnAssaultInfo.hxx"
#include	"CPictureSourceUTX.hxx"
#include	"CPictureSourceJPG.hxx"
#include	"CPictureSourceT3D.hxx"
#include	"CPictureFilterScale.hxx"
#include	"CPictureFilterAlphaBlend.hxx"
#include	"CFurpileLogo.hxx"
#include	"CPicture.hxx"
#include	"C3DPolyList.hxx"

// --------------------------------------------------------------------
// local:	The local data
// --------------------------------------------------------------------
static	CCgiArgs		cgi;
static	CMySqlConnect *	db				= NULL;
static	CCache			pvcache( "preview" );

// --------------------------------------------------------------------
// local:	Some database tables needed
// --------------------------------------------------------------------
static	CTablePref		pref;
static	CTableFile		file;
static	CTablePack		pack;
static	CTableAuth		auth;
static	CTableType		type;
static	CTableRule		rule;
static	CTableXref		xref;

// --------------------------------------------------------------------
// local:	Data associated with those tables
// --------------------------------------------------------------------
static	data_file_t		filedata		= { 0 };
static	data_pack_t		packdata		= { 0 };
static	data_auth_t		authdata		= { 0 };
static	data_type_t		typedata		= { 0 };
static	data_rule_t		previewm		= { 0 };
static	data_xref_t		xrefdata		= { 0 };

// --------------------------------------------------------------------
// local:	Data associated with previewing images
// --------------------------------------------------------------------
static	char			imagename	[1024]	= { 0 };

// --------------------------------------------------------------------
// local:	Data associated with demanding view screen
// --------------------------------------------------------------------
static	char			demandstring[1024]	= { 0 };

// --------------------------------------------------------------------
// local:	Data associated with text output
// --------------------------------------------------------------------
static	char *			textcontent			= NULL;

// --------------------------------------------------------------------
// local:	Unreal package cache management
// --------------------------------------------------------------------
static	char				upackfile	[1024]	= { 0 };
static	CUnUtils *			unrutils			= NULL;
static	CUnLevelInfo *		levelinfo			= NULL;
static	CUnAssaultInfo *	assaultinfo			= NULL;
static	char				screenshot[10][128]	= { { 0 } };
static	dword_t				shotnum				= 0;

// --------------------------------------------------------------------
// local:	Assaultinfo data
// --------------------------------------------------------------------
static	dword_t				targetnum			= 0;
static	char				briefing	[2048]	= { 0 };
static	char				assaultshot	[1024]	= { 0 };

// --------------------------------------------------------------------
// local:	Export previeving system
// --------------------------------------------------------------------
static	bool				xpreview			= false;
static	char				xpreviewer	[1024]	= { 0 };

// --------------------------------------------------------------------
// local:	The polylist for T3D and similar files
// --------------------------------------------------------------------
static	C3DPolyList			xpolylist;
static	char				xpolyshot	[1024]	= { 0 };
static	CPictureFilterAlphaBlend	alphablend( 0x1b, 0x00, 0x3e, 0xff );

// --------------------------------------------------------------------
// local:	Category - there was a type but no category
// --------------------------------------------------------------------
static	void	__category	( void ) {
	CMySqlWhere		w;
	data_type_t		data;
	data_type_tl	list;
	data_type_tli	loop;
	const char *	prevcat = "";
	dword_t			ctgr	= 0;

	w << "type_idnt > 0 order by type_ctgr, type_name";
	list = type.Select( *db, w );

	for	( loop = list.begin(); loop != list.end(); loop++ ) {
		data = *loop;

		// Start of a new category
		if	( ::strcmp( prevcat, data.type_ctgr ) ) {
			prevcat = (*loop).type_ctgr;
			ctgr++;
		}

		// Is the type number same ?
		if	( filedata.file_type == data.type_idnt ) {
			::printf( "%d", ctgr );
			break;
		}
	}
}

// --------------------------------------------------------------------
// local:	Some unreal flags
// --------------------------------------------------------------------
static	const char *	__unrflags( dword_t	aMask ) {
	if	( ! unrutils )	return "???";
	return	(unrutils->UnFile().PackageFlags() & aMask) ? "Yes" : "No";

}

// --------------------------------------------------------------------
// local:	Read a template
// --------------------------------------------------------------------
static	char *	__tmpldata	( const char * aTemplate ) {
	char *	filedata = NULL;

	try	{
		data_pref_tl	list;
		CMySqlWhere		w;

		w << "pref_name='" << aTemplate << "'";
		list = pref.Select( *db, w );

		if	( list.size() < 1 ) {
			throw CError( aTemplate, "Preference not found" );
		}

		filedata = ::my_load_file( (*(list.begin())).pref_valu );
	}

	catch	( ... ) {
		if	( filedata ) delete [] filedata;
		throw;
	}
	return	filedata;
}

// --------------------------------------------------------------------
// local:	Get information about teh file
// --------------------------------------------------------------------
static	void	__get_filedata	( void ) {
	CMySqlWhere		w;
	data_file_tl	filelist;
	data_auth_tl	authlist;
	data_type_tl	typelist;
	data_rule_tl	rulelist;
	data_pref_tl	preflist;

	// ----------------------------------------------------------------
	// If no file ident, give them an error
	// ----------------------------------------------------------------
	if	( ! cgi.Exist( "file_idnt" ) ) {
		throw	CError( "Missing file identifier" );
	}

	// ----------------------------------------------------------------
	// Let's select teh file
	// ----------------------------------------------------------------
	w << "file_idnt=" << (dword_t)::atol( cgi.Arg( "file_idnt" ) );
	filelist = file.Select( *db, w );

	// ----------------------------------------------------------------
	// No such file here ???
	// ----------------------------------------------------------------
	if	( filelist.size() < 1 ) {
		throw CError( "No file available", w.Where() );
	}

	filedata = *(filelist.begin());

	// ----------------------------------------------------------------
	// Dig up teh author name
	// ----------------------------------------------------------------
	w = "";
	w << "auth_idnt=" << filedata.file_auth;
	authlist = auth.Select( *db, w );

	// ----------------------------------------------------------------
	// No such author here ???
	// ----------------------------------------------------------------
	if	( authlist.size() < 1 ) {
		::memset( &authdata, 0, sizeof( authdata ) );
	}
	else {
		authdata = *(authlist.begin());
	}

	// ----------------------------------------------------------------
	// The file type
	// ----------------------------------------------------------------
	w = "";
	w << "type_idnt=" << filedata.file_type;
	typelist = type.Select( *db, w );

	// ----------------------------------------------------------------
	// No such type here ???
	// ----------------------------------------------------------------
	if	( typelist.size() < 1 ) {
		throw CError( "No type available", w.Where() );
	}

	typedata = *(typelist.begin());

	// ----------------------------------------------------------------
	// Preview rule
	// ----------------------------------------------------------------
	w = "";
	w << "rule_type=" << typedata.type_idnt << " and rule_name='preview'";
	rulelist = rule.Select( *db, w );

	// ----------------------------------------------------------------
	// No such rule here ???
	// ----------------------------------------------------------------
	if	( rulelist.size() < 1 ) {
		throw CError( "No preview rule available", w.Where() );
	}

	previewm = *(rulelist.begin());

	// ----------------------------------------------------------------
	// Preview image directory, URL, max number of files and defaultshot
	// ----------------------------------------------------------------
	w = "";
	w << "pref_name='default-screenshot'";
	preflist = pref.Select( *db, w );
	if	( preflist.size() < 1 ) {
		throw CError( "No such preference", w.Where() );
	}
	::strcpy( screenshot[0], (*(preflist.begin())).pref_valu );

	w = "";
	w << "pref_name='default-assaultshot'";
	preflist = pref.Select( *db, w );
	if	( preflist.size() < 1 ) {
		throw CError( "No such preference", w.Where() );
	}
	::strcpy( assaultshot, (*(preflist.begin())).pref_valu );

	// ----------------------------------------------------------------
	// Optional demand string
	// ----------------------------------------------------------------
	w = "";
	w << "rule_type=" << typedata.type_idnt << " and rule_name='demand'";
	rulelist = rule.Select( *db, w );

	if	( rulelist.size() > 0 ) {
		::strcpy( demandstring, (*(rulelist.begin())).rule_rule );
	}
	else {
		::strcpy( demandstring, "Preview in separate window" );
	}
}

// --------------------------------------------------------------------
// local:	Extract the unrutils class
// --------------------------------------------------------------------
static	void	__extract_unrutils	( void ) {
	try {
		unrutils = new CUnUtils( upackfile );
	}

	catch ( ... ) {
		if	( unrutils )	delete unrutils;
		unrutils = NULL;
	}
}

// --------------------------------------------------------------------
// local:	Extract the level info
// --------------------------------------------------------------------
static	void	__extract_levelinfo	( void ) {
	try {
		levelinfo = new CUnLevelInfo(	unrutils->UnFile(),
										unrutils->UnNameTable(),
										unrutils->UnExportTable(),
										unrutils->UnImportTable() );
		char			filepath[1024];
		word_t			i;

		for	( i = 0; i < levelinfo->ShotCount(); i++ ) {

			::sprintf( screenshot[i], "%d-Screenshot%d.jpg", filedata.file_idnt, i );
			pvcache.MakeDirname( filepath, screenshot[i] );

			// Create the file if it does not exist
			if		( ! pvcache.Exist( filepath ) ) {
				CPictureSourceUTX	myutx( levelinfo->Screenshot( i ) );
				CPicture			mypicture;
				CFurpileLogo		logo( "Furpile.utx" );

				mypicture.Load( myutx );
				if	( mypicture.W() > 256 ) {
					CPictureFilterScale	scale( 256, (mypicture.H() * 256) / mypicture.W() );
					mypicture.Filter( scale );
				}

				logo.Plant( mypicture.PixmapRef() );
				CPictureSourceJPG	myjpg( filepath, 70 );
				mypicture.Save( myjpg );

				pvcache.Cache( filepath );
			}
		}
	}

	catch ( ... ) {
		if	( levelinfo )	delete levelinfo;
		levelinfo = NULL;
	}
}

// --------------------------------------------------------------------
// local:	Extract the assault info
// --------------------------------------------------------------------
static	void	__extract_assaultinfo	( void ) {
	try {
		assaultinfo = new CUnAssaultInfo(	unrutils->UnFile(),
											unrutils->UnNameTable(),
											unrutils->UnExportTable(),
											unrutils->UnImportTable() );
	}

	catch ( ... ) {
		if	( assaultinfo )	delete assaultinfo;
		assaultinfo = NULL;
	}
}

// --------------------------------------------------------------------
// local:	Cache an unreal package
// --------------------------------------------------------------------
static	void	__cache_upack		( void ) {
	::sprintf( 	upackfile, "%s/%d-upack.%s",
				pvcache.DirBase(),
				filedata.file_idnt,
				filedata.file_suff );

	if	( ! pvcache.Exist( upackfile ) ) {
		CStorage	storage;
		storage.Load( upackfile, filedata.file_idnt );
		pvcache.Cache( upackfile );
	}
	__extract_unrutils();
	if	( ! unrutils )	return;
	__extract_levelinfo();
	__extract_assaultinfo();

}

// --------------------------------------------------------------------
// local:	Count occurrences of import items
// --------------------------------------------------------------------
static	void	__import_count		( void ) {
	CMySqlWhere		w;
	CMySqlQuote		q;

	w << "file_name='" << q.Quote( xrefdata.xref_name ) << "'";
	::printf( "%d", file.Count( *db, w ) );
}

// --------------------------------------------------------------------
// local:	Model technical statistics
// --------------------------------------------------------------------
static	void	__pll_vertexcount( void ) {
	dword_t	count	= 0;
	C3DPolygon_lci	loop;

	for	(	loop  = xpolylist.PolyList().begin();
			loop != xpolylist.PolyList().end();
			loop++ ) {
		count = count + (*loop).VertexList().size();
	}
	::printf( "%d", count );
}

// --------------------------------------------------------------------
static	void	__pll_xsize( void ) {
	double			minv	=  1000000.0;
	double			maxv	= -1000000.0;
	C3DPolygon_lci	loop;
	C3DVertex_lci	vloop;

	for	(	loop  = xpolylist.PolyList().begin();
			loop != xpolylist.PolyList().end();
			loop++ ) {
		for	(	vloop  =  (*loop).VertexList().begin();
				vloop  != (*loop).VertexList().end();
				vloop++ ) {
			if	( minv > (*vloop).X() )	minv = (*vloop).X();
			if	( maxv < (*vloop).X() )	maxv = (*vloop).X();
		}
	}
	::printf( "%lf", maxv - minv );
}

// --------------------------------------------------------------------
static	void	__pll_ysize( void ) {
	double			minv	=  1000000.0;
	double			maxv	= -1000000.0;
	C3DPolygon_lci	loop;
	C3DVertex_lci	vloop;

	for	(	loop  = xpolylist.PolyList().begin();
			loop != xpolylist.PolyList().end();
			loop++ ) {
		for	(	vloop  =  (*loop).VertexList().begin();
				vloop  != (*loop).VertexList().end();
				vloop++ ) {
			if	( minv > (*vloop).Y() )	minv = (*vloop).Y();
			if	( maxv < (*vloop).Y() )	maxv = (*vloop).Y();
		}
	}
	::printf( "%lf", maxv - minv );
}

// --------------------------------------------------------------------
static	void	__pll_zsize( void ) {
	double			minv	=  1000000.0;
	double			maxv	= -1000000.0;
	C3DPolygon_lci	loop;
	C3DVertex_lci	vloop;

	for	(	loop  = xpolylist.PolyList().begin();
			loop != xpolylist.PolyList().end();
			loop++ ) {
		for	(	vloop  =  (*loop).VertexList().begin();
				vloop  != (*loop).VertexList().end();
				vloop++ ) {
			if	( minv > (*vloop).Z() )	minv = (*vloop).Z();
			if	( maxv < (*vloop).Z() )	maxv = (*vloop).Z();
		}
	}
	::printf( "%lf", maxv - minv );
}

// --------------------------------------------------------------------
static	void	__pll_itemcount( void ) {
	dword_t			count	= 0;
	const char *	refs	= "";
	C3DPolygon_lci	loop;
	C3DPolygon_lci	loopp;

	loop = xpolylist.PolyList().begin();
	while	( loop != xpolylist.PolyList().end() ) {
		refs = (*loop).Item();

		loopp = xpolylist.PolyList().begin();
		while	( loopp != loop ) {
			if	( ! ::strcmp( refs, (*loopp).Item() ) )	break;
			loopp++;
		}
		if	( loop == loopp )	count++;

		loop++;
	}
	::printf( "%d", count );
}

// --------------------------------------------------------------------
static	void	__pll_textcount( void ) {
	dword_t			count	= 0;
	const char *	refs	= "";
	C3DPolygon_lci	loop;
	C3DPolygon_lci	loopp;

	loop = xpolylist.PolyList().begin();
	while	( loop != xpolylist.PolyList().end() ) {
		refs = (*loop).Texture();

		loopp = xpolylist.PolyList().begin();
		while	( loopp != loop ) {
			if	( ! ::strcmp( refs, (*loopp).Texture() ) )	break;
			loopp++;
		}
		if	( loop == loopp )	count++;

		loop++;
	}
	::printf( "%d", count );
}

// --------------------------------------------------------------------
static	void	__pll_itemlist( void ) {
	const char *	refs	= "";
	C3DPolygon_lci	loop;
	C3DPolygon_lci	loopp;

	loop = xpolylist.PolyList().begin();
	while	( loop != xpolylist.PolyList().end() ) {
		refs = (*loop).Item();

		loopp = xpolylist.PolyList().begin();
		while	( loopp != loop ) {
			if	( ! ::strcmp( refs, (*loopp).Item() ) )	break;
			loopp++;
		}
		if	( loop == loopp ) {
			::printf( " " );
			::HTML_quote( *refs ? refs : "default" );
		}

		loop++;
	}
}

// --------------------------------------------------------------------
static	void	__pll_textlist( void ) {
	const char *	refs	= "";
	C3DPolygon_lci	loop;
	C3DPolygon_lci	loopp;

	loop = xpolylist.PolyList().begin();
	while	( loop != xpolylist.PolyList().end() ) {
		refs = (*loop).Texture();

		loopp = xpolylist.PolyList().begin();
		while	( loopp != loop ) {
			if	( ! ::strcmp( refs, (*loopp).Texture() ) )	break;
			loopp++;
		}
		if	( loop == loopp ) {
			::printf( " " );
			::HTML_quote( *refs ? refs : "default" );
		}

		loop++;
	}
}

// --------------------------------------------------------------------
// local:	Parser questions are sent here for answering
// --------------------------------------------------------------------
static	void	__answer	( const char * aQ ) {

	// ----------------------------------------------------------------
	// Local parsing
	// ----------------------------------------------------------------
	if		( ! ::strcmp( aQ, "imagename" ) )		::printf( "%s", imagename );
	else if	( ! ::strcmp( aQ, "imageurl" ) )		::printf( "%s", pvcache.UrlBase() );
	else if	( ! ::strcmp( aQ, "demandstring" ) )	::printf( "%s", demandstring );
	else if	( ! ::strcmp( aQ, "textcontent" ) )		::HTML_quote( textcontent ? textcontent : "" );
	else if	( ! ::strcmp( aQ, "screenshot" ) )		::printf( "%s", screenshot[shotnum] );

	// ----------------------------------------------------------------
	// For linking reviews
	// ----------------------------------------------------------------
	else if	( ! ::strcmp( aQ, "category" ) )		__category();
	

	// ----------------------------------------------------------------
	// Extracting data from level info
	// ----------------------------------------------------------------
	else if	( ! ::strcmp( aQ, "lvl-author" ) )		::HTML_quote( levelinfo ? levelinfo->Author() : "" );
	else if	( ! ::strcmp( aQ, "lvl-title" ) )		::HTML_quote( levelinfo ? levelinfo->Title() : "" );
	else if	( ! ::strcmp( aQ, "lvl-enter" ) )		::HTML_quote( levelinfo ? levelinfo->EnterText() : "" );
	else if	( ! ::strcmp( aQ, "lvl-ideal" ) )		::HTML_quote( levelinfo ? levelinfo->IdealPlayers() : "" );

	// ----------------------------------------------------------------
	// Extracting data from assault info
	// ----------------------------------------------------------------
	else if	( ! ::strcmp( aQ, "ass-target" ) )		::printf( "%d", targetnum );
	else if	( ! ::strcmp( aQ, "ass-brief" ) )		::HTML_quote( briefing );
	else if	( ! ::strcmp( aQ, "ass-shot" ) )		::HTML_quote( assaultshot );

	// ----------------------------------------------------------------
	// Counting
	// ----------------------------------------------------------------
	else if	( ! ::strcmp( aQ, "impcount" ) )		__import_count();

	// ----------------------------------------------------------------
	// Extracting data from Unreal header
	// ----------------------------------------------------------------
	else if	( ! ::strcmp( aQ, "pak-version" ) )		::printf( "%d", unrutils ? unrutils->UnFile().PackageVersion() : 0 );
	else if	( ! ::strcmp( aQ, "pak-licmode" ) )		::printf( "%d", unrutils ? unrutils->UnFile().LicenseeMode() : 0 );
	else if	( ! ::strcmp( aQ, "pak-allowdl" ) )		::printf( "%s", __unrflags( PKG_AllowDownload ) );
	else if	( ! ::strcmp( aQ, "pak-cliopti" ) )		::printf( "%s", __unrflags( PKG_ClientOptional ) );
	else if	( ! ::strcmp( aQ, "pak-servero" ) )		::printf( "%s", __unrflags( PKG_ServerSideOnly ) );
	else if	( ! ::strcmp( aQ, "pak-broklin" ) )		::printf( "%s", __unrflags( PKG_BrokenLinks ) );
	else if	( ! ::strcmp( aQ, "pak-unsecur" ) )		::printf( "%s", __unrflags( PKG_Unsecure ) );
	else if	( ! ::strcmp( aQ, "pak-neededp" ) )		::printf( "%s", __unrflags( PKG_Need ) );
	else if	( ! ::strcmp( aQ, "pak-guid" ) )		::printf( "%s", unrutils ? unrutils->UnFile().GUID() : "" );

	// ----------------------------------------------------------------
	// Export previewing
	// ----------------------------------------------------------------
	else if	( ! ::strcmp( aQ, "xp_s" ) )			::printf( "%s", xpreview ? "" : "<!-- " );
	else if	( ! ::strcmp( aQ, "xp_e" ) )			::printf( "%s", xpreview ? "" : " -->" );
	else if	( ! ::strcmp( aQ, "xp" ) )				::printf( "%s", xpreviewer );

	// ----------------------------------------------------------------
	// T3D and similar technical data
	// ----------------------------------------------------------------
	else if	( ! ::strcmp( aQ, "pll_polycount" ) )	::printf( "%d", (dword_t)xpolylist.PolyList().size() );
	else if	( ! ::strcmp( aQ, "pll_vertexcount" ) )	__pll_vertexcount();
	else if	( ! ::strcmp( aQ, "pll_xsize" ) )		__pll_xsize();
	else if	( ! ::strcmp( aQ, "pll_ysize" ) )		__pll_ysize();
	else if	( ! ::strcmp( aQ, "pll_zsize" ) )		__pll_zsize();
	else if	( ! ::strcmp( aQ, "pll_itemcount" ) )	__pll_itemcount();
	else if	( ! ::strcmp( aQ, "pll_texcount" ) )	__pll_textcount();
	else if	( ! ::strcmp( aQ, "pll_itemlist" ) )	__pll_itemlist();
	else if	( ! ::strcmp( aQ, "pll_texlist" ) )		__pll_textlist();
	else if	( ! ::strcmp( aQ, "pll_preview" ) )		::printf( "%s", xpolyshot );

	// ----------------------------------------------------------------
	// Parsing table data
	// ----------------------------------------------------------------
	else if	( ::MySqlTableParser( file.Layout(), "file", (const char *)(&filedata), aQ ) )	;
	else if	( ::MySqlTableParser( pack.Layout(), "pack", (const char *)(&packdata), aQ ) )	;
	else if	( ::MySqlTableParser( auth.Layout(), "auth", (const char *)(&authdata), aQ ) )	;
	else if	( ::MySqlTableParser( type.Layout(), "type", (const char *)(&typedata), aQ ) )	;
	else if	( ::MySqlTableParser( xref.Layout(), "xref", (const char *)(&xrefdata), aQ ) )	;

	// ----------------------------------------------------------------
	// Parsing the CGI data
	// ----------------------------------------------------------------
	else if	( ::cgi_parser( cgi, aQ ) )	;

	// ----------------------------------------------------------------
	// Everything else goes to default parser
	// ----------------------------------------------------------------
	else	::my_default_answer( aQ );
}

// --------------------------------------------------------------------
// local:	Execute a template
// --------------------------------------------------------------------
static	void	__tempexec	( const char * aTemplate ) {
	char *	tmpldata = __tmpldata( aTemplate );
	try	{

		::my_parse( __answer, tmpldata );
		delete [] tmpldata;
	}
	catch	( ... ) {
		if	( tmpldata ) delete [] tmpldata;
		throw;
	}
}

// --------------------------------------------------------------------
// local:	Common previewer
// --------------------------------------------------------------------
static	void	__preview_file	(	const char *	aTemplate ) {
	char	filepath[1024];
	::sprintf(	imagename, "%d-preview.%s",
				filedata.file_idnt,
				filedata.file_suff );
	pvcache.MakeDirname( filepath, imagename );
	if	( ! pvcache.Exist( filepath ) ) {
		CStorage	storage;
		storage.Load( filepath, filedata.file_idnt );
		pvcache.Cache( filepath );
	}
	if	( ! ::strcmp( previewm.rule_rule, "t3d" ) ) {
		xpolylist.T3DParseFile( filepath );

		int		camera	= T3D_CAMERA_FRONT;

		if	( cgi.Exist( "cmra" ) ) {
			camera = ::atoi( cgi.Arg( "cmra" ) );
		}

		char	xspath[1024];

		::sprintf( xpolyshot, "xpshot-%d-%d.jpg", (dword_t)camera, filedata.file_idnt );
		pvcache.MakeDirname( xspath, xpolyshot );

		// Create the file if it does not exist
		if	( ! pvcache.Exist( xspath ) ) {
			CPictureSourceT3D	myt3d( filepath );
			CPictureSourceJPG	myjpg( xspath, 70 );
			CPicture			mypicture;
			CFurpileLogo		logo( "Furpile.utx" );

			myt3d.Camera( camera );

			while	( ( myt3d.W() < 512 ) && ( myt3d.H() < 512 ) ) {
				myt3d.Factor( myt3d.Factor() * 1.2 );
			}


			mypicture.Load( myt3d );

			if	( mypicture.W() > 640 ) {
				CPictureFilterScale	scale( 640, (mypicture.H() * 640) / mypicture.W() );
				mypicture.Filter( scale );
			}

			mypicture.Filter( alphablend );
			logo.Plant( mypicture.PixmapRef() );
			mypicture.Save( myjpg );
			pvcache.Cache( filepath );
		}
	}
	if	( aTemplate )	__tempexec	( aTemplate );
}

// --------------------------------------------------------------------
// local:	Preview method: preview text
// --------------------------------------------------------------------
static	void	__preview_text		( void ) {
	char			filepath[1024];
	__preview_file	( NULL );
	textcontent = ::my_load_file( pvcache.MakeDirname( filepath, imagename ) );
	__tempexec	( "tmpl-preview-text" );
}

// --------------------------------------------------------------------
// local:	Preview assault info
// --------------------------------------------------------------------
static	void	__preview_assault	( void ) {
	char *	tmpldata = __tmpldata( "tmpl-preview-ass-body" );
	char	defashot [1024];
	dword_t	targ;

	::strcpy( defashot, assaultshot );

	try	{
		__tempexec	( "tmpl-preview-ass-head" );

		for	( targ = 0; targ < assaultinfo->Count(); targ++ ) {

			targetnum = targ + 1;

			if	( assaultinfo->Desc( targ ) ) {
				::my_strfit( briefing, sizeof( briefing ), assaultinfo->Desc( targ ) );
			}
			else {
				::strcpy( briefing, "No briefing available" );
			}

			if	( assaultinfo->Shot( targ ) ) {
				char			filepath[1024];

				::sprintf( assaultshot, "%d-ashot-%d.jpg", filedata.file_idnt, targetnum );
				pvcache.MakeDirname( filepath, assaultshot );

				// Create the file if it does not exist
				if		( ! pvcache.Exist( filepath ) ) {

					CPictureSourceUTX	myutx( assaultinfo->Shot( targ ) );
					CPictureSourceJPG	myjpg( filepath, 70 );
					CPicture			mypicture;
					CFurpileLogo		logo( "Furpile.utx" );

					mypicture.Load( myutx );

					if	( mypicture.W() > 256 ) {
						CPictureFilterScale	scale( 256, (mypicture.H() * 256) / mypicture.W() );
						mypicture.Filter( scale );
					}

					logo.Plant( mypicture.PixmapRef() );

					mypicture.Save( myjpg );
					pvcache.Cache( filepath );

				}
			}
			else {
				::strcpy( assaultshot, defashot );
			}

			::my_parse( __answer, tmpldata );
		}

		__tempexec	( "tmpl-preview-ass-tail" );
		delete [] tmpldata;
	}

	catch	( ... ) {
		if	( tmpldata )	delete [] tmpldata;
	}
}

// --------------------------------------------------------------------
// local:	Preview the import list
// --------------------------------------------------------------------
static	void	__preview_import	( void ) {
	char *	tmpldata = __tmpldata( "tmpl-preview-imp-body" );

	try	{
		__tempexec	( "tmpl-preview-imp-head" );

		// Get package names needed
		CMySqlWhere		w;
		data_xref_tl	xlist;
		data_xref_tli	xloop;

		w << "xref_file=" << filedata.file_idnt << " and xref_expo=0 order by xref_name";
		xlist = xref.Select( *db, w );

		for	( xloop = xlist.begin(); xloop != xlist.end(); xloop++ ) {
			xrefdata = *xloop;
			::my_parse( __answer, tmpldata );
		}

		__tempexec	( "tmpl-preview-imp-tail" );
		delete [] tmpldata;
	}

	catch	( ... ) {
		if	( tmpldata )	delete [] tmpldata;
	}
}

// --------------------------------------------------------------------
// local:	Preview the export list
// --------------------------------------------------------------------
static	void	__preview_export	( void ) {
	char *	tmpldata = __tmpldata( "tmpl-preview-exp-body" );

	try	{
		__tempexec	( "tmpl-preview-exp-head" );

		// Get package names needed
		CMySqlWhere		w;
		CMySqlQuote		q;
		data_xref_tl	xlist;
		data_xref_tli	xloop;
		data_pref_tl	plist;

		w << "xref_file=" << filedata.file_idnt << " and xref_expo=1 order by xref_name";
		xlist = xref.Select( *db, w );

		for	( xloop = xlist.begin(); xloop != xlist.end(); xloop++ ) {
			xrefdata = *xloop;

			w = "";
			w << "pref_name='xp-" << q.Quote( xrefdata.xref_name ) << "'";
			plist = pref.Select( *db, w );
			if	( plist.size() > 0 ) {
				xpreview	= true;
				::strcpy( xpreviewer, (*(plist.begin())).pref_valu );
			}
			else {
				xpreview	= false;
				*xpreviewer	= 0;
			}

			::my_parse( __answer, tmpldata );
		}

		__tempexec	( "tmpl-preview-exp-tail" );
		delete [] tmpldata;
	}

	catch	( ... ) {
		if	( tmpldata )	delete [] tmpldata;
	}
}

// --------------------------------------------------------------------
// local:	Preview level info head part
// --------------------------------------------------------------------
static	void	__preview_fileinfo	( void ) {
	char *	tmpldata = __tmpldata( "tmpl-preview-unr-body" );

	try	{
		__tempexec	( "tmpl-preview-unr-head" );

		if	( levelinfo->ShotCount() < 1 ) {
			shotnum = 0;
			::my_parse( __answer, tmpldata );
		}
		else {
			for	( shotnum = 0; shotnum < levelinfo->ShotCount(); shotnum++ ) {
				::my_parse( __answer, tmpldata );
			}
		}

		__tempexec	( "tmpl-preview-unr-tail" );
		delete [] tmpldata;
	}

	catch	( ... ) {
		if	( tmpldata )	delete [] tmpldata;
	}
}

// --------------------------------------------------------------------
// local:	Preview method: preview Unreal Tournament game level
// --------------------------------------------------------------------
static	void	__preview_ulevel	( void ) {
	__cache_upack();
	__preview_fileinfo();
	if	( levelinfo ) {
		__tempexec	( "tmpl-preview-levelinfo" );
	}

	if	( assaultinfo ) {
		__preview_assault();
	}
	__preview_import();
	__preview_export();
}

// --------------------------------------------------------------------
// local:	Memory cleanup
// --------------------------------------------------------------------
static	void	__memclean	( void ) {
	try	{
		if	( assaultinfo )	delete assaultinfo;
		assaultinfo = NULL;
	
		if	( levelinfo )	delete levelinfo;
		levelinfo = NULL;
	
		if	( unrutils )	delete unrutils;
		unrutils = NULL;
	
		if	( textcontent ) delete [] textcontent;
		textcontent = NULL;
	
		if	( db )			delete db;
		db = NULL;
	}

	catch	( ... ) {
	}
}

// --------------------------------------------------------------------
// public:  Program entrypoint
// --------------------------------------------------------------------
extern  int     main( 	int aAc, char ** aAv ) {
	::srand( (unsigned int)::time( NULL ) );
	::HTTP_text( "en" );

	try	{
		db = new CMySqlConnect( "quest", "", "UTCMS" );
		__get_filedata();

		if		( ! ::strcmp( previewm.rule_rule, "skip" ) )		__tempexec		( "tmpl-preview-skip" );
		else if	( ! ::strcmp( previewm.rule_rule, "image" ) )		__preview_file	( "tmpl-preview-image" );
		else if	( ! ::strcmp( previewm.rule_rule, "demand" ) )		__preview_file	( "tmpl-preview-demand" );
		else if	( ! ::strcmp( previewm.rule_rule, "html" ) )		__preview_file	( "tmpl-preview-html" );
		else if	( ! ::strcmp( previewm.rule_rule, "text" ) )		__preview_text	();
		else if	( ! ::strcmp( previewm.rule_rule, "utlevel" ) )		__preview_ulevel();
		else if	( ! ::strcmp( previewm.rule_rule, "u1level" ) )		__preview_ulevel();
		else if	( ! ::strcmp( previewm.rule_rule, "upack" ) )		__preview_ulevel();
		else if	( ! ::strcmp( previewm.rule_rule, "t3d" ) )			__preview_file	( "tmpl-preview-t3d" );
		else	throw CError( "Missing handler for rule", previewm.rule_rule );
		__memclean();
	}

	catch 	( CError e ) {
		::printf( "<p>Error: " );
		::HTML_quote( e.Error() );
		::printf( "</p>" );
		__memclean();
		::fflush( stdout );
		::fclose( stdout );
		return	-1;
	}

	catch	( ... ) {
		::printf( "<p>Error: Unknown error</p>" );
		__memclean();
		::fflush( stdout );
		::fclose( stdout );
		return	-1;
	}
	::fflush( stdout );
	::fclose( stdout );
    return	0;
}

// --------------------------------------------------------------------
// EOF: Preview.cxx
// --------------------------------------------------------------------
