// --------------------------------------------------------------------
// UServerPoll.cxx
// Whatis:  A process that polls for unreal servers
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 25-APR-2003     Created this source
// --------------------------------------------------------------------
#include    "CError.hxx"
#include    "CUnServerInfo.hxx"
#include    "CUnMasterServer.hxx"
#include	"CFileScore.hxx"
#include	"CTableSrvr.hxx"
#include	"CTableFsrv.hxx"
#include	"CTableFile.hxx"
#include	"CMySqlWhere.hxx"
#include	"CCpuMutex.hxx"

// --------------------------------------------------------------------
// Definitions (limitations)
// --------------------------------------------------------------------
#define	MAX_POLLER_THREADS	8

// --------------------------------------------------------------------
// Class used by the threads to poll the servers
// --------------------------------------------------------------------
class	CServerPoller	{

	// ----------------------------------------------------------------
	public:		// Constructor
	// ----------------------------------------------------------------
	CServerPoller	();

	// ----------------------------------------------------------------
	public:		// Destructor
	// ----------------------------------------------------------------
	~CServerPoller	();

	// ----------------------------------------------------------------
	public:		// Run from thread
	// ----------------------------------------------------------------
	void			Run		( void );
	int				Slot	( void ) const;

	// ----------------------------------------------------------------
	private:	// Private helpers
	// ----------------------------------------------------------------
	data_srvr_tl	GetNextRecord		( void );
	void			PollUtServer		( void );
	void			PollUt2k3Server		( void );
	void			UpdateDatabase		( void );
	void			FillServerInfo		( const CUnServerInfo & );

	// ----------------------------------------------------------------
	private:	// Instance data
	// ----------------------------------------------------------------
	data_srvr_t		itsNewRecord;
	data_srvr_t		itsOldRecord;
};

// --------------------------------------------------------------------
// local:	Some static data
// --------------------------------------------------------------------
static	dword_t				__next_record_number	= 0;
static	CCpuMutex			__database_mutex;
static	int					__cur_threads			= 0;
static	int					__max_threads			= 0;

// --------------------------------------------------------------------
// CServerPoller: Constructor
// --------------------------------------------------------------------
CServerPoller::CServerPoller	() {
}

CServerPoller::~CServerPoller	() {
}

// --------------------------------------------------------------------
// CServerPoller: Run from thread
// --------------------------------------------------------------------
void	CServerPoller::Run	( void ) {
	data_srvr_tl	list;
	data_srvr_tli	loop;
	try {
		// First, get our portion of the stuff
		list = CServerPoller::GetNextRecord();

		// Loop through all in there
		for	( loop = list.begin(); loop != list.end(); loop++ ) {
			itsOldRecord = *loop;
			itsNewRecord = *loop;

			// Depending on server type do an action
			if		( ! ::strcmp( itsNewRecord.srvr_type, "UT" ) )		CServerPoller::PollUtServer();
			else if	( ! ::strcmp( itsNewRecord.srvr_type, "UT2003" ) )	CServerPoller::PollUt2k3Server();
	
			// If the data did change update it to database but only if we got the data
			if	( itsNewRecord.srvr_pmax != 0 ) {
				if	( ::memcmp( &itsNewRecord, &itsOldRecord, sizeof( itsNewRecord ) ) ) {
					CServerPoller::UpdateDatabase();
				}
			}
	
		}
	}
	catch ( ... ) {
	}
}

// --------------------------------------------------------------------
// CServerPoller: 	Get next available record from database
// --------------------------------------------------------------------
data_srvr_tl	CServerPoller::GetNextRecord	( void ) {
	CMySqlWhere		w;
	CTableSrvr		srvr;
	data_srvr_tl	list;
	data_srvr_tli	loop;
	data_srvr_tl	result;
	const char *	thisip;

	__database_mutex.Lock();
	try	{
		CMySqlConnect	db( "quest", "", "UTCMS" );

		w = "";
		w << "srvr_quep>0 order by srvr_addr,srvr_quep LIMIT " << __next_record_number << ",60";
		list = srvr.Select( db, w );

		if	( list.size() < 1 ) {
			__next_record_number = 0;
			w = "";
			w << "srvr_quep>0 order by srvr_addr,srvr_quep LIMIT " << __next_record_number << ",60";
			list = srvr.Select( db, w );
		}

		if	( list.size() > 0 ) {
			thisip = (*(list.begin())).srvr_addr;
			for	( loop = list.begin(); loop != list.end(); loop++ ) {
				if	( ::strcmp( thisip, (*loop).srvr_addr ) )	break;
				result.push_back( *loop );
				__next_record_number++;
			}
		}
	}

	catch	( ... ) {
	}
	__database_mutex.Unlock();
	return	result;
}

// --------------------------------------------------------------------
// CServerPoller: Poll an UT server
// --------------------------------------------------------------------
void	CServerPoller::PollUtServer	( void ) {
	try {
		CUnServerInfo	myinfo( itsNewRecord.srvr_addr, itsNewRecord.srvr_quep, unreal_server_ut );
		if	( myinfo.GamePort() != 0 ) {
			CServerPoller::FillServerInfo( myinfo );
		}
	}

	catch	( ... ) {
		::memset( &itsNewRecord, 0, sizeof( itsNewRecord ) );
	}
}

// --------------------------------------------------------------------
// CServerPoller: Poll an UT2003 server
// --------------------------------------------------------------------
void	CServerPoller::PollUt2k3Server	( void ) {
	try {
		CUnServerInfo	myinfo( itsNewRecord.srvr_addr, itsNewRecord.srvr_quep, unreal_server_2003 );
		if	( myinfo.GamePort() != 0 ) {
			CServerPoller::FillServerInfo( myinfo );
		}
	}

	catch	( ... ) {
		::memset( &itsNewRecord, 0, sizeof( itsNewRecord ) );
	}
}

// --------------------------------------------------------------------
// CServerPoller:	Fill in server info
// --------------------------------------------------------------------
void	CServerPoller::FillServerInfo	( const CUnServerInfo & aInfo ) {
	::memset( &itsNewRecord, 0, sizeof( itsNewRecord ) );
	::my_strfit( itsNewRecord.srvr_addr, sizeof( itsNewRecord.srvr_addr ) - 1, aInfo.IpAddress() );
	::my_strfit( itsNewRecord.srvr_name, sizeof( itsNewRecord.srvr_name ) - 1, aInfo.ServerName() );
	::my_strfit( itsNewRecord.srvr_gmap, sizeof( itsNewRecord.srvr_gmap ) - 1, aInfo.GameMap() );
	::my_strfit( itsNewRecord.srvr_gttl, sizeof( itsNewRecord.srvr_gttl ) - 1, aInfo.MapTitle() );
	::my_strfit( itsNewRecord.srvr_gtyp, sizeof( itsNewRecord.srvr_gtyp ) - 1, aInfo.GameType() );
	itsNewRecord.srvr_pcur = aInfo.CurrentPlayers();
	itsNewRecord.srvr_pmax = aInfo.MaximumPlayers();
	itsNewRecord.srvr_port = aInfo.GamePort();
	itsNewRecord.srvr_quep = aInfo.QueryPort();
	itsNewRecord.srvr_time = ::time( NULL );
	switch	( aInfo.Type() ) {
		case	unreal_server_ut:	::strcpy( itsNewRecord.srvr_type, "UT" );		break;
		case	unreal_server_2003:	::strcpy( itsNewRecord.srvr_type, "UT2003" );	break;
	}
}

// --------------------------------------------------------------------
// CServerPoller: Update database
// --------------------------------------------------------------------
void	CServerPoller::UpdateDatabase	( void ) {
	CTableSrvr		srvr;
	CTableFile		file;
	CTableFsrv		fsrv;
	CMySqlWhere		w;
	CMySqlQuote		q;
	data_file_tl	list;
	data_fsrv_t		fsrvdata;

	__database_mutex.Lock();

	try	{
		CMySqlConnect	db( "quest", "", "UTCMS" );

		// Update the data record
		srvr.Delete( db, &itsOldRecord );
		srvr.Delete( db, &itsNewRecord );
		srvr.Insert( db, &itsNewRecord );

		// Do we have one and only one file for it ?
		w << "file_name='" << q.Quote( itsNewRecord.srvr_gmap ) << "'";
		if	( ! ::strcmp( itsNewRecord.srvr_type, "UT" ) )	w << " and file_suff='unr'";
		else												w << " and file_suff='ut2'";

		list = file.Select( db, w );
		if	( list.size() == 1 ) {
			CFileScore	myscore( (*(list.begin())).file_idnt );

			// Calculate how well the server is utilized
			dword_t	seats	= itsNewRecord.srvr_pmax;
			dword_t	plays	= itsNewRecord.srvr_pcur;

			if	( seats == 0 )		seats = 1;
			if	( plays > seats )	plays = seats;
			plays = plays * 10 + 5;
			plays = plays / seats;
			if	( plays > 10 )	plays = 10;

			switch	( plays ) {

				case   10:	myscore.Rate2( 9 );	myscore.Rate1( 0 );	myscore.Rate0( 0 );	break;
				case	9:	myscore.Rate2( 9 );	myscore.Rate1( 0 );	myscore.Rate0( 0 );	break;
				case	8:	myscore.Rate2( 8 );	myscore.Rate1( 0 );	myscore.Rate0( 0 );	break;
				case	7:	myscore.Rate2( 7 );	myscore.Rate1( 0 );	myscore.Rate0( 0 );	break;
				case	6:	myscore.Rate2( 6 );	myscore.Rate1( 0 );	myscore.Rate0( 0 );	break;
				case	5:	myscore.Rate2( 5 );	myscore.Rate1( 0 );	myscore.Rate0( 0 );	break;
				case	4:	myscore.Rate2( 4 );	myscore.Rate1( 0 );	myscore.Rate0( 0 );	break;
				case	3:	myscore.Rate2( 3 );	myscore.Rate1( 0 );	myscore.Rate0( 0 );	break;
				case	2:	myscore.Rate2( 2 );	myscore.Rate1( 0 );	myscore.Rate0( 0 );	break;
				case	1:	myscore.Rate2( 1 );	myscore.Rate1( 1 );	myscore.Rate0( 0 );	break;
				case	0:	myscore.Rate2( 0 );	myscore.Rate1( 1 );	myscore.Rate0( 1 );	break;
			}

			myscore.Players( itsNewRecord.srvr_pcur );
			myscore.Update();

			// Remove all entries that are older than 2 weeks
			w = "";
			w << "to_days('now') - to_days(fsrv_time) > 14";
			fsrv.Delete( db, w );

			// Create the record
			::memset( &fsrvdata, 0, sizeof( fsrvdata ) );
			fsrvdata.fsrv_file = (*(list.begin())).file_idnt;
			::strcpy( fsrvdata.fsrv_addr, itsNewRecord.srvr_addr );
			fsrvdata.fsrv_quep = itsNewRecord.srvr_quep;
			fsrvdata.fsrv_pmax = itsNewRecord.srvr_pmax;
			fsrvdata.fsrv_pcur = itsNewRecord.srvr_pcur;
			fsrvdata.fsrv_time = ::time( NULL );

			// Update the table
			fsrv.Delete( db, &fsrvdata );
			fsrv.Insert( db, &fsrvdata );
		}
	}

	catch	( ... ) {
	}
	__database_mutex.Unlock();

}

//---------------------------------------------------------------------------
static	bool	__server_in_database( 	const char *	aIp,
										dword_t			aQp ) {
	bool	result = true;

	__database_mutex.Lock();
	try	{
		CMySqlConnect	db( "quest", "", "UTCMS" );
		CTableSrvr		srvr;
		CMySqlWhere		w;

		w << "srvr_addr='" << aIp << "' and srvr_quep=" << aQp;
		if	( srvr.Count( db, w ) == 0 ) {
			result = false;
		}
	}
	catch	( ... ) {
		result = true;
	}
	__database_mutex.Unlock();
	return	result;
}

//---------------------------------------------------------------------------
static	void	__poll_master_server(	const char *	aIp,
										word_t			aPort,
										unreal_server_t	aType ) {
	try	{
		CUnMasterServerItem_tl		list;
		CUnMasterServerItem_tlci	loop;
		data_srvr_t					data;
		int							retry;

		for	( retry = 0; retry < 5; retry++ ) {
			try {
				CUnMasterServer		master( aIp, aPort, aType );
				list = master.ServerList();
			}
			catch ( ... ) {
			}
			if	( list.size() > 0 )	break;
		}

	    for	( 	loop  = list.begin();
	            loop != list.end();
	            loop++ ) {

			// Check if this server already exists in the database
			if	( ! __server_in_database( (*loop).IpAddress(), (*loop).QueryPort() ) ) {
				::memset( &data, 0, sizeof( data ) );
				::my_strfit( data.srvr_addr, sizeof( data.srvr_addr ) - 1, (*loop).IpAddress() );
				data.srvr_quep = (*loop).QueryPort();
				data.srvr_time = ::time( NULL );
				switch	( (*loop).ServerType() ) {
					case	unreal_server_ut:	::strcpy( data.srvr_type, "UT" );		break;
					case	unreal_server_2003:	::strcpy( data.srvr_type, "UT2003" );	break;
				}

				__database_mutex.Lock();

				try {
					CMySqlConnect	db( "quest", "", "UTCMS" );
					CTableSrvr		srvr;
					srvr.Insert( db, &data );
				}

				catch ( ... ) {
				}

				__database_mutex.Unlock();
			}
		}
	}

	catch	( ... ) {
	}
}

// --------------------------------------------------------------------
// local:	The thread for one poller
// --------------------------------------------------------------------
static	void *	__client_thread ( void * ) {
	bool	keepmealive = true;

	while	( keepmealive ) {
		try	{
			CServerPoller	poller;
			poller.Run();
			SLEEP(1000);
		}

		catch ( ... ) {
		}

		__database_mutex.Lock();
		keepmealive = ( __cur_threads <= __max_threads );
		__database_mutex.Unlock();
	}

	__database_mutex.Lock();
	__cur_threads--;
	if	( __cur_threads < 0 )	__cur_threads = 0;
	__database_mutex.Unlock();
	return	NULL;
}

// --------------------------------------------------------------------
// local:	Every 60 seconds adjust the max thread count
// --------------------------------------------------------------------
static	void	__adjust_max_thread_count	( void ) {
	int		mycount	= 0;
	char	myline	[512];
	FILE *	mypipe;

	try {
	 	mypipe = ::popen( "ps -C proftpd --format pid,args --cols 400", "r" );
		while	( mypipe ) {
			if	( ::fgets( myline, 511, mypipe ) ) {
				if	( ::strstr( myline, "RETR" ) )	mycount += 2;
			}
			else {
				::pclose( mypipe );
				mypipe = NULL;
			}
		}
	
	 	mypipe = ::popen( "ps -C SendZip --format pid,args --cols 400", "r" );
		while	( mypipe ) {
			if	( ::fgets( myline, 511, mypipe ) ) {
				if	( ::strstr( myline, "SendZip" ) )	mycount += 2;
			}
			else {
				::pclose( mypipe );
				mypipe = NULL;
			}
		}
	
	 	mypipe = ::popen( "ps -C ZipDaemon --format pid,args --cols 400", "r" );
		mycount -= 2;
		while	( mypipe ) {
			if	( ::fgets( myline, 511, mypipe ) ) {
				if	( ::strstr( myline, "ZipDaemon" ) )	mycount += 2;
			}
			else {
				::pclose( mypipe );
				mypipe = NULL;
			}
		}
	
		// Set new maximum for threads
		__max_threads = MAX_POLLER_THREADS - mycount;
		if	( __max_threads <= 0 ) {
			__max_threads = 1;
		}
	}
	catch	( ... ) {
		__max_threads = 1;
	}
}

// --------------------------------------------------------------------
// local:	Delete records older tha 10 days
// --------------------------------------------------------------------
static	void	__del_old_records	( void ) {
	CTableSrvr		srvr;
	CTableFsrv		fsrv;
	CMySqlConnect	db( "quest", "", "UTCMS" );

	
	srvr.Delete( db, " where to_days(now())-to_days(srvr_time) > 10" );
	fsrv.Delete( db, " where to_days(now())-to_days(fsrv_time) > 10" );
}

// --------------------------------------------------------------------
// local:	The process
// --------------------------------------------------------------------
static	void	__process	( void ) {
	pthread_t	mythread;
	int			counter;
	time_t		lastadjust;
	time_t		lastmaster;

	lastadjust = ::time( NULL ) + 60;
	lastmaster = ::time( NULL ) + 30;
	while	( true ) {

		// Each minute adjust max number of threads
		if	( lastadjust < ::time( NULL ) ) {
			__adjust_max_thread_count();
			lastadjust = ::time( NULL ) + 60;
		}

		// Each day poll the master servers
		if	( lastmaster < ::time( NULL ) ) {
			// Cool down threads
			__max_threads = 0;
			counter = 300;
			while	( ( __cur_threads > 0 ) && ( counter > 0 ) ) {
				SLEEP( 1000 );
				counter--;
			}
			__cur_threads = 0;
			try {
				__poll_master_server( "unreal.epicgames.com", 			28900, unreal_server_ut );
				__poll_master_server( "ut2003master.epicgames.com", 	   80, unreal_server_2003 );
				__poll_master_server( "master.gamespy.com", 			28900, unreal_server_ut );
				__poll_master_server( "master.mplayer.com",				28900, unreal_server_ut );
				lastmaster = ::time( NULL ) + (60 * 60 * 4);	// Next refresh after 4 hours
				__adjust_max_thread_count();
				lastadjust = ::time( NULL ) + 60;
				__del_old_records();
			}
			catch ( ... ) {
			}
		}

		// Do we need another thread ?
		if	( __cur_threads < __max_threads ) {
			__database_mutex.Lock();
			try {
				mythread = 0;
				::pthread_create( &mythread, NULL, __client_thread, NULL );
				::pthread_detach( mythread );
				__cur_threads++;
			}
			catch ( ... ) {
			}
			__database_mutex.Unlock();
		}
		SLEEP( 1000 );
	}
}

// --------------------------------------------------------------------
// public:  Program entrypoint
// --------------------------------------------------------------------
extern  int     main( 	int aAc, char ** aAv ) {
	::daemon( 1, 0 );
	__process();
    return  0;
}

// --------------------------------------------------------------------
// EOF: Motd.cxx
// --------------------------------------------------------------------
