// --------------------------------------------------------------------
// ZipDaemon.cxx
// Whatis:  Actually a HTTP server for sending files to clients
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 03-MAY-2003     Created this source
// --------------------------------------------------------------------
#include    "CError.hxx"
#include    "CCpuMutex.hxx"
#include    "CTcpSocket.hxx"
#include    "CTableAuth.hxx"
#include    "CTableRule.hxx"
#include    "CTableFile.hxx"
#include    "CTablePack.hxx"
#include    "CTablePackFile.hxx"
#include	"CCgiArgs.hxx"
#include	"CZip.hxx"
#include	"CMySqlWhere.hxx"
#include	"CFileScore.hxx"
#include	"CZipStreamSock.hxx"

// --------------------------------------------------------------------
// local:	Some local data
// --------------------------------------------------------------------
#define	ZIPDAEMON_PORT		(unsigned short)1337
#define	MAX_THREAD_COUNT	64
#define	ZIPDAEMON_LOGFILE	"/var/log/zipdaemon/zipdaemon.log"

//---------------------------------------------------------------------------
// local:	Structure used to store request data
//---------------------------------------------------------------------------
typedef	struct {
	char	mode	[5];	// Contains 'pack' or 'file'
	char	idnt	[7];	// Contains the identifier
	char	file	[128];	// Contains name of the file
	char	prot	[16];	// Contains the protocol 'HTTP/1.1'
	bool	ranu;			// True if there was a range request

	dword_t	rans;			// Start of range
	dword_t	rane;			// End of range

}	http_request_t;

//---------------------------------------------------------------------------
// local:	Structure used to store client information
//---------------------------------------------------------------------------
typedef struct	{
	char	addr	[64];	// IP address
	dword_t	count;			// Number of threads for him
}	client_info_t;

// --------------------------------------------------------------------
// Days and months
// --------------------------------------------------------------------
static const char *    __weekday[] = {
    "Sun", "Mon", "Tue", "Wed", "Thu",
    "Fri", "Sat"
};
static const char *    __month[] = {
    "Jan", "Feb", "Mar", "Apr",
    "May", "Jun", "Jul", "Aug",
    "Sep", "Oct", "Nov", "Dec"
};

//---------------------------------------------------------------------------
// local:	Common data for threads
//---------------------------------------------------------------------------
static	CCpuMutex		__mutex;
static	int				__number_of_threads					= 0;
static	client_info_t	__client_info[ MAX_THREAD_COUNT ]	= { { { 0 } } };

// --------------------------------------------------------------------
// local:	Write entry to the log file
// --------------------------------------------------------------------
static	void	__writelog			(	const char *	aIp,
										const char *	aMsg ) {
	FILE *		log	= NULL;
	time_t		tim	= ::time( NULL );
	struct tm *	tms	= ::gmtime( &tim );

	__mutex.Lock();

	try {
		log = ::fopen( ZIPDAEMON_LOGFILE, "a+" );
		if	( ! log ) {
			log = ::fopen( ZIPDAEMON_LOGFILE, "w" );
		}
		if	( log ) {
			::fprintf(	log, "%02d-%s-%4d %02d:%02d:%02d %s %s\n",
						tms->tm_mday,
						__month[tms->tm_mon],
						tms->tm_year + 1900,
						tms->tm_hour,
						tms->tm_min,
						tms->tm_sec,
						aIp, aMsg );
			::fclose( log );
			
		}
	}

	catch	( ... ) {
		if	( log )	::fclose( log );
	}
	__mutex.Unlock();
}

// --------------------------------------------------------------------
// local:	Receive a line from a socket - throw if disconnected
// --------------------------------------------------------------------
static	void	__read_sock_line	(	char *			aBuf,
										size_t			aLen,
										CTcpSocket *	aSock ) {
	time_t	mytime	= ::time( NULL ) + 10;
	size_t	offs	= 0;
	int		rch;

	while	( true ) {
		rch = aSock->Read();
		if		( rch == -1 ) {
			if	( mytime < ::time( NULL ) ) {
				throw	0;
			}
		}
		else if	( rch == -2 ) {
			throw	0;
		}
		else if	( rch >= 0 ) {
			if	( rch != 0x0d ) {
				if	( rch == 0x0a )	break;
				aBuf[offs++] = rch;
			}
			if	( offs == aLen ) {
				throw 0;
			}
		}
	}
	aBuf[offs] = 0;
}

// --------------------------------------------------------------------
// local:	Move pointer to next non-whitespace character
// --------------------------------------------------------------------
static	char *	__skip_blanks	( char * aS ) {
	while	( ( *aS > 0 ) && ( *aS <= ' ' ) )	aS++;
	return	aS;
}

// --------------------------------------------------------------------
// local:	Grab the request
// --------------------------------------------------------------------
static	void	__read_request	(	http_request_t &	aRequest,
									CTcpSocket *		aSock ) {
	char	myline	[1024];
	char *	s;
	char *	e;

	::memset( &aRequest, 0, sizeof( aRequest ) );

	// The first line
	__read_sock_line( myline, sizeof( myline ), aSock );

	// The first four chars should read 'GET '
	if	( ::strncmp( myline, "GET ", 4 ) )	throw 0;
	
	// The next thing should be '/pack/' or '/file/'
	s = __skip_blanks( myline + 4 );

	if	( ( ::strncmp( s, "/pack/", 6 ) ) &&
		  ( ::strncmp( s, "/file/", 6 ) ) )	throw 0;
	s++; e = s + 4; *e = 0;
	::strcpy( aRequest.mode, s );
	s = e + 1;

	// Next we should have 1 .. 6 characters of numbers - this is the ID followed by '/'
	e = s;
	while	( ( *e >= '0' ) && ( *e <= '9' ) )	e++;
	if	( *e != '/' )	throw 0;
	*e = 0;
	if	( ::strlen( s ) > 6 )	throw 0;
	::strcpy( aRequest.idnt, s );
	s = e + 1;

	// Next characters are the zip name (URL encoded) - terminates to a blanc
	e = s;
	while	( ( *e < 0 ) || ( *e > ' ' ) )	e++;
	if	( *e != ' ' )	throw 0;
	*e = 0;
	if	( ::strlen( s ) > 127 )	throw 0;
	::strcpy( aRequest.file, s );

	// The last thing should be the protocol string
	s = __skip_blanks( e + 1 );
	if	( ::strlen( s ) > 15 )	throw 0;
	::strcpy( aRequest.prot, s );

	// Collect rest of the header information
	while	( true ) {
		__read_sock_line( myline, sizeof( myline ), aSock );
		if	( *myline == 0 )	break;

		// Let's see if it has the 'Range: bytes='
		if	( ! ::strncmp( myline, "Range:", 6 ) ) {

			s = __skip_blanks( myline + 6 );
			if	( ::strncmp( s, "bytes", 5 ) )	continue;
			s = __skip_blanks( s + 5 );
			if	( *s != '=' )					continue;
			s = __skip_blanks( s + 1 );

			e = s+1;
			while	( ( *e != '-' ) && ( *e ) )	e++;
			if	( *e != '-' )					continue;
			*e = 0; e++;

			// Range start
			aRequest.rans = (dword_t)::atol( s );

			s = e;
			while	( ( *e >= '0' ) && ( *e <= '9' ) )	e++;
			*e = 0;
	
			// Range end
			aRequest.rane = (dword_t)::atol( s );
			aRequest.ranu = true;
		}
	}

	char	detailbuf[1024];
	::sprintf(	detailbuf, "mode='%s' idnt='%s' file='%s' prot='%s' start=%d stop=%d",
				aRequest.mode,
				aRequest.idnt,
				aRequest.file,
				aRequest.prot,
				aRequest.rans,
				aRequest.rane );
	__writelog( aSock->PeerAddr(), detailbuf );
}

// --------------------------------------------------------------------
// local:	Insert one file for download
// --------------------------------------------------------------------
static	dword_t	__insert_file	( 	CZip &			aZip,
									CMySqlConnect & aDb,
									dword_t			aF ) {
	CTableFile		file;
	CTableRule		rule;
	CMySqlWhere		w;
	data_file_tl	flist;
	data_rule_tl	rlist;
	data_file_t		fdata;

	// Get file data
	w << "file_idnt=" << aF;
	flist = file.Select( aDb, w );
	if	( flist.size() < 1 )	return 0;
	fdata = *(flist.begin());

	// Check if this file can be downloaded
	if	( ( fdata.file_flag & FLAG_NODOWNLOAD ) != 0 )	return 0;

	// Determine the destination path name
	char	destination[1024];
	char *	installdir;

	// Determine what is the subdirectory name
	w = "";
	w << "rule_type=" << fdata.file_type << " and rule_name='installdir'";
	rlist = rule.Select( aDb, w );
	if	( rlist.size() > 0 ) {
		installdir = (*(rlist.begin())).rule_rule;
	}
	else {
		installdir = "";
	}
	*destination = 0;

	// Is there a filename component in the install directory ?
	if		( ! ::strncmp( installdir, "$(filename)/", 12 ) ) {
		::strcat( destination, fdata.file_name );
		::strcat( destination, "/" );
		installdir = installdir + 12;
	}
	::strcat( destination, installdir );
	if	( *destination )	::strcat( destination, "/" );
	::strcat( destination, fdata.file_name );
	if	( fdata.file_suff[0] != 0 ) {
		::strcat( destination, "." );
		::strcat( destination, fdata.file_suff );
	}

	// Grab the GZ into memory
	CGz		mygz;
	mygz.Read( fdata.file_data );

	// Add the stuff into the outgoing zip
	aZip.AddDirectoryEntry( mygz, destination, fdata.file_ctim );

	// Add a comment of the original author - if available
	char	mycomment[2048];
	::strcpy( mycomment, "- Original author: " );
	if	( fdata.file_auth > 0 ) {
		CTableAuth		auth;
		data_auth_tl	alist;
		w = "";
		w << "auth_idnt=" << fdata.file_auth;
		alist = auth.Select( aDb, w );
		if	( alist.size() > 0 ) {
			::strcat( mycomment, (*(alist.begin())).auth_name );
		}
		else {
			::strcat( mycomment, "unknown" );
		}
	}
	else {
		::strcat( mycomment, "unknown" );
	}
	aZip.AddFileComment( destination, mycomment );

	// Finally the statistics
	CFileScore	myscore( aF );
	myscore.Download();
	myscore.Update();
	return	1;
}

// --------------------------------------------------------------------
// local:	Collect a single file into the zip
// --------------------------------------------------------------------
static	void	__collect_file	( CZip & aZip, dword_t aF ) {
	CMySqlConnect	db( "quest", "", "UTCMS" );
	CMySqlWhere		w;
	CTableFile		file;
	CTableAuth		auth;
	data_file_tl	flist;
	data_auth_tl	alist;
	char			mycomment[16384];	// Large enough

	if	( ! __insert_file( aZip, db, aF ) ) {
		throw CError( "That file is not valid for download" );
	}
	else {
		// Generate the comment
		::strcpy( mycomment, "------------------------------------------------\n" );
		::strcat( mycomment, "This file was downloaded from UTCMS file service\n" );
		::strcat( mycomment, "http://furpile.com/UTCMS/\n" );
		::strcat( mycomment, "(c) Esko 'Varpu' Ilola 2002, 2003, 2004\n" );
		::strcat( mycomment, "------------------------------------------------\n" );
		::strcat( mycomment, "The authors of the contained files are:\n" );
		w = "";
		w << "file_idnt=" << aF;
		flist = file.Select( db, w );

		::strcat( mycomment, (*(flist.begin())).file_name );
		if	( (*(flist.begin())).file_suff[0] ) {
			::strcat( mycomment, "." );
			::strcat( mycomment, (*(flist.begin())).file_suff );
		}
		::strcat( mycomment, " - " );

		if	( (*(flist.begin())).file_auth > 0 ) {
			w = "";
			w << "auth_idnt=" << (*(flist.begin())).file_auth;
			alist = auth.Select( db, w );
			if	( alist.size() > 0 ) {
				::strcat( mycomment, (*(alist.begin())).auth_name );
			}
			else {
				::strcat( mycomment, "author unknown" );
			}
		}
		else {
			::strcat( mycomment, "author unknown" );
		}
		::strcat( mycomment, "\n" );
		::strcat( mycomment, "------------------------------------------------\n" );
		aZip.AddZipComment( mycomment );
	}
}

// --------------------------------------------------------------------
// local:	Collect an entire pack into the zip
// --------------------------------------------------------------------
static	void	__collect_pack	( CZip & aZip, dword_t aP ) {
	CMySqlConnect		db( "quest", "", "UTCMS" );
	CMySqlWhere			w;
	CTablePackFile		packfile;
	CTableFile			file;
	CTableAuth			auth;
	data_packfile_tl	pflist;
	data_packfile_tli	pfloop;
	data_file_tl		flist;
	data_auth_tl		alist;
	dword_tl			filist;
	dword_tli			filoop;
	dword_t				analmax;
	dword_t				analnow;
	char				mycomment[16384];	// Large enough

	// Get the files in this package
	w << "packfile_pack=" << aP;
	pflist = packfile.Select( db, w );
	if	( pflist.size() < 1 )	throw CError( "No pack with that identifier" );

	// Loop all them through and count those that were added for transmission
	for	( pfloop = pflist.begin(); pfloop != pflist.end(); pfloop++ ) {
		if	( __insert_file( aZip, db, (*pfloop).packfile_file ) ) {
			filist.push_back( (*pfloop).packfile_file );
		}
	}

	if	( filist.size() < 1 )	throw CError( "No files were valid for download" );

	// Generate the comment
	::strcpy( mycomment, "------------------------------------------------\n" );
	::strcat( mycomment, "This file was downloaded from UTCMS file service\n" );
	::strcat( mycomment, "http://furpile.com/UTCMS/\n" );
	::strcat( mycomment, "(c) Esko 'Varpu' Ilola 2002, 2003, 2004\n" );
	::strcat( mycomment, "------------------------------------------------\n" );
	::strcat( mycomment, "The authors of the contained files are:\n" );


	// Fisrt, detect the longest file name in this pack
	analmax = 0;
	for	( filoop = filist.begin(); filoop != filist.end(); filoop++ ) {
		analnow = 0;
		w = "";
		w << "file_idnt=" << (*filoop);
		flist = file.Select( db, w );

		analnow = ::strlen( (*(flist.begin())).file_name );
		if	( (*(flist.begin())).file_suff[0] ) {
			analnow = analnow + 1 + ::strlen( (*(flist.begin())).file_suff );
		}
		analmax = analnow > analmax ? analnow : analmax;
	}

	for	( filoop = filist.begin(); filoop != filist.end(); filoop++ ) {
		w = "";
		w << "file_idnt=" << (*filoop);
		flist = file.Select( db, w );

		analnow = ::strlen( (*(flist.begin())).file_name );
		::strcat( mycomment, (*(flist.begin())).file_name );
		if	( (*(flist.begin())).file_suff[0] ) {
			::strcat( mycomment, "." );
			::strcat( mycomment, (*(flist.begin())).file_suff );
			analnow = analnow + 1 + ::strlen( (*(flist.begin())).file_suff );
		}

		// Embed spaces to fill up to analmax - makes it all look better
		while	( analnow < analmax ) {
			::strcat( mycomment, " " );
			analnow++;
		}
		::strcat( mycomment, " - " );

		if	( (*(flist.begin())).file_auth > 0 ) {
			w = "";
			w << "auth_idnt=" << (*(flist.begin())).file_auth;
			alist = auth.Select( db, w );
			if	( alist.size() > 0 ) {
				::strcat( mycomment, (*(alist.begin())).auth_name );
			}
			else {
				::strcat( mycomment, "author unknown" );
			}
		}
		else {
			::strcat( mycomment, "author unknown" );
		}
		::strcat( mycomment, "\n" );
		if	( ::strlen( mycomment ) > sizeof( mycomment ) - 1000 )	break;
	}

	::strcat( mycomment, "------------------------------------------------\n" );
	aZip.AddZipComment( mycomment );
}

// --------------------------------------------------------------------
// local:	Collect data into the zip
// --------------------------------------------------------------------
static	void	__collect_zip	( CZip & aZip, http_request_t & aRequest ) {
	dword_t		idnt =	(dword_t)::atol( aRequest.idnt );

	if	( ! ::strcmp( aRequest.mode, "pack" ) )	__collect_pack( aZip, idnt );
	else										__collect_file( aZip, idnt );
}

// --------------------------------------------------------------------
// local:   Time output in MIME compliant form
// --------------------------------------------------------------------
static  void    __mime_time ( CTcpSocket * aSock, time_t aTime ) {
    struct tm * tms = ::gmtime( &aTime );
    dword_t		year= (tms->tm_year) % 100;

	aSock->Write( __weekday[tms->tm_wday] );	aSock->Write( ", " );
	aSock->Write( (dword_t)tms->tm_mday );		aSock->Write( " " );
	aSock->Write( __month[tms->tm_mon] );		aSock->Write( " " );

	aSock->Write( year < 10 ? "0" : "" );			aSock->Write( year );
	aSock->Write( tms->tm_hour < 10 ? "0" : "" );	aSock->Write( (dword_t)tms->tm_hour );
	aSock->Write( tms->tm_min < 10 ? "0" : "" );	aSock->Write( (dword_t)tms->tm_min );
	aSock->Write( tms->tm_sec < 10 ? "0" : "" );	aSock->Write( (dword_t)tms->tm_sec );
#ifdef	WIN32
    aSock->Write( "GMT\n" );
#else
    aSock->Write( tms->tm_zone );	aSock->Write( "\n" );
#endif
}

// --------------------------------------------------------------------
// local:	Register the client
// --------------------------------------------------------------------
static	bool	__register_client	( CTcpSocket *	aSock ) {
	bool	result = false;
	int		slot;

	__mutex.Lock();

	try {
		// Try to find this client
		for	( slot = 0; slot < MAX_THREAD_COUNT; slot++ ) {
			if	( ! ::strcmp( __client_info[slot].addr, aSock->PeerAddr() ) ) {
				break;
			}
		}
		// If found we just add to it's thread count
		if	( slot < MAX_THREAD_COUNT ) {
			if	( __client_info[slot].count < 5 ) {
				__client_info[slot].count++;
				result = true;
			}
		}

		// Didn't exist. We have to find an empty slot
		else {
			for	( slot = 0; slot < MAX_THREAD_COUNT; slot++ ) {
				if	( __client_info[slot].addr[0] == 0 ) {
					::strcpy( __client_info[slot].addr, aSock->PeerAddr() );
					__client_info[slot].count = 1;
					result = true;
					break;
				}
			}
		}
	}

	catch	( ... ) {
	}

	__mutex.Unlock();
	return	result;
}

// --------------------------------------------------------------------
// local:	Register the client
// --------------------------------------------------------------------
static	void	__unregister_client	( CTcpSocket *	aSock ) {
	int		slot;

	__mutex.Lock();

	try {
		for	( slot = 0; slot < MAX_THREAD_COUNT; slot++ ) {
			if	( ! ::strcmp( __client_info[slot].addr, aSock->PeerAddr() ) ) {
				__client_info[slot].count--;
				if	( __client_info[slot].count == 0 ) {
					__client_info[slot].addr[0] = 0;
				}
				break;
			}
		}
	}

	catch	( ... ) {
	}

	__mutex.Unlock();
}

// --------------------------------------------------------------------
// local:	Process one request
// --------------------------------------------------------------------
static	void		__process_request	( CTcpSocket *	aSock ) {
	CZipStreamSock	sockout( aSock, 10 );
	http_request_t	request;
	CZip			zip;
	dword_t			zipsize;
	bool			usehtml= true;


	try {
		usehtml = false;
		__read_request( request, aSock );
		usehtml = true;
		__collect_zip ( zip, request );
		usehtml = false;

		// Send out a proper header
		aSock->Write( request.prot );
		if	( request.ranu ) {
			aSock->Write( " 206 Partial Content\n" );
		}
		else {
			aSock->Write( " 200 OK\n" );
		}

		aSock->Write( "Date: " );			__mime_time( aSock, ::time( NULL ) );
		aSock->Write( "Expires: ");			__mime_time( aSock, ::time( NULL ) - (time_t)(24 * 60 * 60 * 7) );
		aSock->Write( "Server: ZipDaemon/0.0.9 (Unix)\n" );
		aSock->Write( "Pragma: no-cache\n" );
		aSock->Write( "Keep-Alive: timeout=15, max=100\n" );
		aSock->Write( "Connection: Keep-Alive\n" );
		aSock->Write( "Accept-Ranges: bytes\n" );

		// Set the portion parameters on the header
		zipsize = zip.ZippedSize();

		if	( request.rane == 0 )	request.rane = zipsize - 1;

		aSock->Write( "Content-Length: " );
		if	( request.ranu ) {
			aSock->Write( (dword_t)( request.rane - request.rans + 1 ) );
		}
		else {
			aSock->Write( (dword_t)zipsize );
		}
		aSock->Write( "\n" );

		if	( request.ranu ) {
			aSock->Write( "Content-Range: bytes " );
			aSock->Write( request.rans );	aSock->Write( "-" );
			aSock->Write( request.rane );	aSock->Write( "/" );
			aSock->Write( zipsize );		aSock->Write( "\n" );
		}

		aSock->Write( "Content-Type: application/zip\n\n" );
		aSock->Flush();

		// Initialize our skip bytes count and the amount to output
		sockout.StartPoint	( request.rans );
		sockout.EndPoint	( request.rane );


		// Last but not least - the throttle is disabled on lan !
		if	( ! ::strncmp( aSock->PeerAddr(), "10.10.", 6 ) ) {
			__writelog( aSock->PeerAddr(), "Disabling throttle for local client" );
			sockout.Throttle( 1000 );
		}

		zip.Write( sockout );
		aSock->Flush();
	}

	catch ( CError e ) {
		__writelog( aSock->PeerAddr(), e.Error() );
		if	( usehtml ) {
			aSock->Write( request.prot );
			aSock->Write( " 404 Not Found\n" );
			aSock->Write( "Date: " );			__mime_time( aSock, ::time( NULL ) );
			aSock->Write( "Expires: ");			__mime_time( aSock, ::time( NULL ) - (time_t)(24 * 60 * 60 * 7) );
			aSock->Write( "Server: ZipDaemon/0.0.9 (Unix)\n" );
			aSock->Write( "Pragma: no-cache\n" );
			aSock->Write( "Keep-Alive: timeout=15, max=100\n" );
			aSock->Write( "Connection: Keep-Alive\n" );
			aSock->Write( "Accept-Ranges: bytes\n" );
			aSock->Write( "Content-Type: text/html\n\n" );
			aSock->Write( "<html><body>The file you asked was not found on this server</body></html>\n" );
			aSock->Flush();
		}
	}

	catch ( ... ) {
		__writelog( aSock->PeerAddr(), "Possible file not found error" );
		if	( usehtml ) {
			aSock->Write( request.prot );
			aSock->Write( " 404 Not Found\n" );
			aSock->Write( "Date: " );			__mime_time( aSock, ::time( NULL ) );
			aSock->Write( "Expires: ");			__mime_time( aSock, ::time( NULL ) - (time_t)(24 * 60 * 60 * 7) );
			aSock->Write( "Server: ZipDaemon/0.0.9 (Unix)\n" );
			aSock->Write( "Pragma: no-cache\n" );
			aSock->Write( "Keep-Alive: timeout=15, max=100\n" );
			aSock->Write( "Connection: Keep-Alive\n" );
			aSock->Write( "Accept-Ranges: bytes\n" );
			aSock->Write( "Content-Type: text/html\n\n" );
			aSock->Write( "<html><body>The file you asked was not found on this server</body></html>\n" );
			aSock->Flush();
		}
	}
}

// --------------------------------------------------------------------
// local:	Process the client - normal way
// --------------------------------------------------------------------
static	void *	__client_thread ( void * aArg ) {
	CTcpSocket *	sock = (CTcpSocket *)aArg;
	__process_request( sock );
	__unregister_client( sock );
	delete	sock;
	__mutex.Lock();
	__number_of_threads--;
	__mutex.Unlock();
	return	NULL;
}

// --------------------------------------------------------------------
// local:	The process
// --------------------------------------------------------------------
static	void	__process	( void ) {
	CTcpSocket			mysocket( (socket_type_t)socket_type_stream );
	CTcpSocket *		client = NULL;
	pthread_t			mythread;

	mysocket.Bind( ZIPDAEMON_PORT );
	mysocket.Listen();

	do	{
		try	{
			client = mysocket.Accept();
			if	( client ) {
				if		( __number_of_threads >= MAX_THREAD_COUNT ) {
					__writelog( client->PeerAddr(), "Disposed - too many clients" );
					delete client;
				}
				else if	( ! __register_client( client ) ) {
					__writelog( client->PeerAddr(), "Disposed - unable to register" );
					delete client;
				}
				else {
					__mutex.Lock();
					try {
						mythread = 0;
						::pthread_create( &mythread, NULL, __client_thread, (void *)client );
						__number_of_threads++;
						::pthread_detach( mythread );
					}
					catch	( ... ) {
					}
					__mutex.Unlock();
				}
			}
			else {
				SLEEP( 1000 );
			}
		}
		catch	( CError e ) {
			__writelog( "????", e.Error() );
		}
		catch	( ... ) {
			__writelog( "????", "Unknown error" );
		}
	}	while	( true );
}



// --------------------------------------------------------------------
// public:  Program entrypoint
// --------------------------------------------------------------------
extern  int     main( 	int aAc, char ** aAv ) {
	__writelog( "???", "Daemon starting" );
	::daemon( 1, 0 );
	__process();
    return  0;
}

// --------------------------------------------------------------------
// EOF: ZipDaemon.cxx
// --------------------------------------------------------------------
