// --------------------------------------------------------------------
// CMySqlData.cxx
// Whatis:  Data manipulation for all mysal tables
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 24-NOV-2001     Created this source
// --------------------------------------------------------------------
#include    "CError.hxx"
#include    "CMySqlData.hxx"

// --------------------------------------------------------------------
// Remove comment from the followin to enable printing the SQL statements
// --------------------------------------------------------------------
// #define PRINT_SQL_STATEMENTS    1

// --------------------------------------------------------------------
// public:      Constructor and destructor of the base class
// --------------------------------------------------------------------
CMySqlData::CMySqlData  ( data_struct_descriptor_tp aStruct, word_t aSize ) {
    itsDataStructDescriptor = aStruct;
    itsStructSize           = aSize;
}

// --------------------------------------------------------------------
CMySqlData::~CMySqlData ( ) {
}

// --------------------------------------------------------------------
// public:      Make a select max(*) into the database
// --------------------------------------------------------------------
dword_t CMySqlData::Maximum(    		CMySqlConnect_t &   aDataBase,
                                        const char *        aTableName,
                                        const char *        aColumnName ) {
    char            clause[4096];
    MYSQL_RES *     myr = NULL;
    MYSQL_ROW       row;
    dword_t         maximum = 0;

    try {

	    // ------------------------------------------------------------
	    // Build the select statement
	    // ------------------------------------------------------------
	    ::sprintf( clause, "SELECT max(%s) FROM %s", aColumnName, aTableName );

	    // ------------------------------------------------------------
	    // Make the query
	    // ------------------------------------------------------------
#ifdef  PRINT_SQL_STATEMENTS
::printf( "%s\n", clause );
#endif
	    if  ( ::mysql_query( aDataBase.Mysql(), clause ) ) {
	        throw CError( "SELECT", aTableName, ::mysql_error( aDataBase.Mysql() ) );
	    }

	    // ------------------------------------------------------------
	    // Fetch in all data immediately
	    // ------------------------------------------------------------
        myr = ::mysql_store_result( aDataBase.Mysql() );
        if  ( ! myr )   throw CError( "FETCH", aTableName, ::mysql_error( aDataBase.Mysql() ) );

	    // ------------------------------------------------------------
        // Get the data
	    // ------------------------------------------------------------
        if  ( ::mysql_num_rows( myr ) > 0 ) {
    	    row = ::mysql_fetch_row( myr );
			if	( row ) {
				if	( row[0] ) {
		            maximum = ::atol( row[0] );
				}
			}
        }
    }

    catch ( ... ) {
        if  ( myr ) ::mysql_free_result( myr );
        throw;
    }

    if  ( myr )  ::mysql_free_result( myr );
    return  maximum;
}

// --------------------------------------------------------------------
// public:      Make a select count(*) into the database
// --------------------------------------------------------------------
dword_t CMySqlData::MakeSelectCount(    CMySqlConnect_t &   aDataBase,
                                        const char *        aTableName,
	                                    bool				aDist,
                                        const char *        aWhereClause ) {
    char            clause[4096];
    MYSQL_RES *     myr = NULL;
    MYSQL_ROW       row;
    dword_t         count = 0;

    try {

	    // ------------------------------------------------------------
	    // Build the select statement
	    // ------------------------------------------------------------
		if	( aDist )	::strcpy( clause, "SELECT DISTINCT count(*) FROM " );
		else			::strcpy( clause, "SELECT count(*) FROM " );
	    ::strcat( clause, aTableName );
	    if  ( aWhereClause ) {
	        ::strcat( clause, " " );
	        ::strcat( clause, aWhereClause );
	    }

	    // ------------------------------------------------------------
	    // Make the query
	    // ------------------------------------------------------------
#ifdef  PRINT_SQL_STATEMENTS
::printf( "%s\n", clause );
#endif
	    if  ( ::mysql_query( aDataBase.Mysql(), clause ) ) {
	        throw CError( "SELECT", aTableName, ::mysql_error( aDataBase.Mysql() ) );
	    }

	    // ------------------------------------------------------------
	    // Fetch in all data immediately
	    // ------------------------------------------------------------
        myr = ::mysql_store_result( aDataBase.Mysql() );
        if  ( ! myr )   throw CError( "FETCH", aTableName, ::mysql_error( aDataBase.Mysql() ) );

	    // ------------------------------------------------------------
        // Get the data
	    // ------------------------------------------------------------
        if  ( ::mysql_num_rows( myr ) > 0 ) {
    	    row = ::mysql_fetch_row( myr );
            count = ::atol( row[0] );
        }
    }

    catch ( ... ) {
        if  ( myr ) ::mysql_free_result( myr );
        throw;
    }

    if  ( myr )  ::mysql_free_result( myr );
    return  count;
}

// --------------------------------------------------------------------
// public:      Make a select into the database
// --------------------------------------------------------------------
void ** CMySqlData::MakeSelect( CMySqlConnect_t &   aDataBase,
                                const char *        aTableName,
                                bool				aDist,
                                const char *        aWhereClause,
                                const char *        aOrderClause ) {
    char            clause[4096];
    int             i, rownow, colnow;
    MYSQL_RES *     myr = NULL;
    void **         res = NULL;
    MYSQL_ROW       row;
    my_ulonglong    rowcount;

    try {

	    // ------------------------------------------------------------
	    // Build the select statement
	    // ------------------------------------------------------------
		if	( aDist )	::strcpy( clause, "SELECT DISTINCT " );
		else			::strcpy( clause, "SELECT " );
	    for ( i = 0; itsDataStructDescriptor[i].name; i++ ) {
            if  ( i > 0 )   ::strcat( clause, "," );
			::strcat( clause, aTableName );
			::strcat( clause, "." );
	        ::strcat( clause, itsDataStructDescriptor[i].name );
	    }
	    ::strcat( clause, " FROM " );
	    ::strcat( clause, aTableName );
	    if  ( aWhereClause ) {
	        ::strcat( clause, " " );
	        ::strcat( clause, aWhereClause );
	    }
	    if  ( aOrderClause ) {
	        ::strcat( clause, " " );
	        ::strcat( clause, aOrderClause );
	    }

	    // ------------------------------------------------------------
	    // Make the query
	    // ------------------------------------------------------------
#ifdef  PRINT_SQL_STATEMENTS
::printf( "%s\n", clause );
#endif
	    if  ( ::mysql_query( aDataBase.Mysql(), clause ) ) {
	        throw CError( "SELECT", aTableName, ::mysql_error( aDataBase.Mysql() ) );
	    }

	    // ------------------------------------------------------------
	    // Fetch in all data immediately
	    // ------------------------------------------------------------
        myr = ::mysql_store_result( aDataBase.Mysql() );
        if  ( ! myr )   throw CError( "FETCH", aTableName, ::mysql_error( aDataBase.Mysql() ) );

	    // ------------------------------------------------------------
        // Allocate container for the data
	    // ------------------------------------------------------------
        rowcount = ::mysql_num_rows( myr );
        res = new void * [ rowcount + 1 ];
        ::memset( res, 0, sizeof( void * ) * ( rowcount + 1 ) );

#ifdef  PRINT_SQL_STATEMENTS
::printf( "\nGot %d rows\n\n", rowcount );
#endif
	    // ------------------------------------------------------------
        // Collect data for each row
	    // ------------------------------------------------------------
        rownow = 0;
        while   ( ( row = ::mysql_fetch_row( myr ) ) ) {
            char *  data = new char [ itsStructSize ];
            ::memset( data, 0, itsStructSize );
            res[ rownow++ ] = (void *)data;

            for ( colnow = 0; itsDataStructDescriptor[colnow].name; colnow++ ) {
                switch  ( itsDataStructDescriptor[colnow].type ) {
                    case    data_struct_type_time:
                    CMySqlData::ImportTime ( data, itsDataStructDescriptor[colnow], row[colnow] );
                    break;

                    case    data_struct_type_bool:
                    CMySqlData::ImportBool ( data, itsDataStructDescriptor[colnow], row[colnow] );
                    break;

                    case    data_struct_type_dword:
                    CMySqlData::ImportDword( data, itsDataStructDescriptor[colnow], row[colnow] );
                    break;

                    case    data_struct_type_ascz:
                    CMySqlData::ImportAscz ( data, itsDataStructDescriptor[colnow], row[colnow] );
                    break;
                }
            }
        }
    }

    catch ( ... ) {
        if  ( res ) CMySqlData::Destroy( res );
        if  ( myr ) ::mysql_free_result( myr );
        throw;
    }

    if  ( myr )  ::mysql_free_result( myr );
    return  res;
}

// --------------------------------------------------------------------
// public:      Make insert
// --------------------------------------------------------------------
void    CMySqlData::MakeInsert( CMySqlConnect_t &   aDataBase,
                                const char *        aTableName,
                                const void **       aData ) {
    // ----------------------------------------------------------------
    // For each row in the data
    // ----------------------------------------------------------------
    for ( int rownow = 0; aData[rownow]; rownow++ ) {
        char    clause[4096];
		::memset( clause, 0, sizeof( clause ) );
        ::sprintf( clause, "INSERT %s SET ", aTableName );
        for ( int colnow = 0; itsDataStructDescriptor[colnow].name; colnow++ ) {
            if  ( colnow > 0 )  ::strcat( clause, "," );
            ::strcat( clause, itsDataStructDescriptor[colnow].name );
            ::strcat( clause, "=" );
            switch  ( itsDataStructDescriptor[colnow].type ) {
                case    data_struct_type_time:
                CMySqlData::ExportTime ( clause, itsDataStructDescriptor[colnow], (const char *)aData[rownow] );
                break;

                case    data_struct_type_bool:
                CMySqlData::ExportBool ( clause, itsDataStructDescriptor[colnow], (const char *)aData[rownow] );
                break;

                case    data_struct_type_dword:
                CMySqlData::ExportDword( clause, itsDataStructDescriptor[colnow], (const char *)aData[rownow] );
                break;

                case    data_struct_type_ascz:
                CMySqlData::ExportAscz ( clause, itsDataStructDescriptor[colnow], (const char *)aData[rownow] );
                break;
            }
        }

#ifdef  PRINT_SQL_STATEMENTS
::printf( "%s\n\n", clause );
#endif
        if  ( ::mysql_query( aDataBase.Mysql(), clause ) ) {
            throw CError( "INSERT", aTableName, ::mysql_error( aDataBase.Mysql() ) );
        }
    }
}

// --------------------------------------------------------------------
// public:      Make update
// --------------------------------------------------------------------
void    CMySqlData::MakeUpdate( CMySqlConnect_t &   aDataBase,
                                const char *        aTableName,
                                const CMySqlWhere & aWhere,
                                const void **       aData ) {
    CMySqlData::MakeUpdate( aDataBase, aTableName, aWhere.Where(), aData );
}

// --------------------------------------------------------------------
void    CMySqlData::MakeUpdate( CMySqlConnect_t &   aDataBase,
                                const char *        aTableName,
                                const char *        aWhereClause,
                                const void **       aData ) {
    // ----------------------------------------------------------------
    // For each row in the data
    // ----------------------------------------------------------------
    for ( int rownow = 0; aData[rownow]; rownow++ ) {
        char    clause[4096];
		::memset( clause, 0, sizeof( clause ) );
        ::sprintf( clause, "UPDATE %s SET ", aTableName );
        for ( int colnow = 0; itsDataStructDescriptor[colnow].name; colnow++ ) {
            if  ( colnow > 0 )  ::strcat( clause, "," );
            ::strcat( clause, itsDataStructDescriptor[colnow].name );
            ::strcat( clause, "=" );
            switch  ( itsDataStructDescriptor[colnow].type ) {
                case    data_struct_type_time:
                CMySqlData::ExportTime ( clause, itsDataStructDescriptor[colnow], (const char *)aData[rownow] );
                break;

                case    data_struct_type_bool:
                CMySqlData::ExportBool ( clause, itsDataStructDescriptor[colnow], (const char *)aData[rownow] );
                break;

                case    data_struct_type_dword:
                CMySqlData::ExportDword( clause, itsDataStructDescriptor[colnow], (const char *)aData[rownow] );
                break;

                case    data_struct_type_ascz:
                CMySqlData::ExportAscz ( clause, itsDataStructDescriptor[colnow], (const char *)aData[rownow] );
                break;
            }
        }
	    if  ( aWhereClause ) {
	        ::strcat( clause, " " );
	        ::strcat( clause, aWhereClause );
	    }

#ifdef  PRINT_SQL_STATEMENTS
::printf( "%s\n\n", clause );
#endif
        if  ( ::mysql_query( aDataBase.Mysql(), clause ) ) {
	        throw CError( "UPDATE", aTableName, ::mysql_error( aDataBase.Mysql() ) );
        }
    }
}

// --------------------------------------------------------------------
// public:      Make delete
// --------------------------------------------------------------------
void    CMySqlData::MakeDelete( CMySqlConnect_t &   aDataBase,
                                const char *        aTableName,
                                const char *        aWhereClause ) {
    char    clause[4096];
	::memset( clause, 0, sizeof( clause ) );
    ::sprintf( clause, "DELETE FROM %s", aTableName );
    if  ( aWhereClause ) {
        ::strcat( clause, " " );
        ::strcat( clause, aWhereClause );
    }

#ifdef  PRINT_SQL_STATEMENTS
::printf( "%s\n\n", clause );
#endif

    if  ( ::mysql_query( aDataBase.Mysql(), clause ) ) {
        throw CError( "DELETE", aTableName, ::mysql_error( aDataBase.Mysql() ) );
    }
}

// --------------------------------------------------------------------
// public:      Dispose used data array
// --------------------------------------------------------------------
void    CMySqlData::Destroy   ( void ** aData ) {
    for ( int i = 0; aData[i]; i++ ) {
        delete [] ((char *)aData[i]);
        aData[i] = NULL;
    }
    delete [] aData;
}

// --------------------------------------------------------------------
// private:     Importing
// --------------------------------------------------------------------
void    CMySqlData::ImportTime( char * aB, data_struct_descriptor_t & aF, const char * aD ) {
    time_t      tnw = ::time( NULL );
    struct  tm  tms = *localtime( &tnw );
    char        cpy[20];

    // Format in the string is
    // YYYY-MM-DD HH:MM:SS
    // 0000000000111111111
    // 0123456789012345678

    ::memcpy( cpy, aD, 19 ); cpy[19] = 0;
    
    tms.tm_sec  = ::atoi( cpy + 17 );   cpy[16] = 0;
    tms.tm_min  = ::atoi( cpy + 14 );   cpy[13] = 0;
    tms.tm_hour = ::atoi( cpy + 11 );   cpy[10] = 0;
    tms.tm_mday = ::atoi( cpy +  8 );   cpy[ 7] = 0;
    tms.tm_mon  = ::atoi( cpy +  5 )-1; cpy[ 4] = 0;
    tms.tm_year = ::atoi( cpy ) - 1900;

    *DATA_LOCATION( aB, aF, time_t ) = ::mktime( &tms );
}

// --------------------------------------------------------------------
void    CMySqlData::ImportBool( char * aB, data_struct_descriptor_t & aF, const char * aD ) {
    *DATA_LOCATION( aB, aF, bool ) = (aD[0] == '1');
}

// --------------------------------------------------------------------
void    CMySqlData::ImportDword( char * aB, data_struct_descriptor_t & aF, const char * aD ) {
    *DATA_LOCATION( aB, aF, dword_t ) = (dword_t)atol( aD );
}

// --------------------------------------------------------------------
void    CMySqlData::ImportAscz( char * aB, data_struct_descriptor_t & aF, const char * aD ) {
    char *  buf = DATA_LOCATION( aB, aF, char );

    // ----------------------------------------------------------------
    // Decrypt as needed
    // ----------------------------------------------------------------
    if  ( aF.crypt ) {
        char    copy[1024];
        ::strcpy( copy, aD );
        ::my_decrypt( copy );
        if ( (int)::strlen( aD ) > (int)(aF.size - 1) ) {
            ::memcpy( buf, copy, aF.size - 1 );
        }
        else {
            ::strcpy( buf, copy );
        }
    }
    else if ( (int)::strlen( aD ) > (int)(aF.size - 1) ) {
        ::memcpy( buf, aD, aF.size - 1 );
    }
    else {
        ::strcpy( buf, aD );
    }
    ::my_strtail( ::my_strhead( buf ) );
}

// --------------------------------------------------------------------
// private:     Exporting
// --------------------------------------------------------------------
void    CMySqlData::ExportTime( char * aB, data_struct_descriptor_t & aF, const char * aD ) {
    time_t          tnw = *DATA_LOCATION( aD, aF, time_t );
    struct  tm *    tms = ::localtime( & tnw );

    ::sprintf(  aB + ::strlen( aB ), "'%d-", tms->tm_year + 1900 );

    if  ( tms->tm_mon > 8 )     ::sprintf( aB + ::strlen( aB ), "%d-",  tms->tm_mon + 1 );
    else                        ::sprintf( aB + ::strlen( aB ), "0%d-", tms->tm_mon + 1 );

    if  ( tms->tm_mday > 9 )    ::sprintf( aB + ::strlen( aB ), "%d ",  tms->tm_mday );
    else                        ::sprintf( aB + ::strlen( aB ), "0%d ", tms->tm_mday );

    if  ( tms->tm_hour > 9 )    ::sprintf( aB + ::strlen( aB ), "%d:",  tms->tm_hour );
    else                        ::sprintf( aB + ::strlen( aB ), "0%d:", tms->tm_hour );

    if  ( tms->tm_min > 9 )     ::sprintf( aB + ::strlen( aB ), "%d:",  tms->tm_min );
    else                        ::sprintf( aB + ::strlen( aB ), "0%d:", tms->tm_min );

    if  ( tms->tm_sec > 9 )     ::sprintf( aB + ::strlen( aB ), "%d'",   tms->tm_sec );
    else                        ::sprintf( aB + ::strlen( aB ), "0%d'",  tms->tm_sec );
}

// --------------------------------------------------------------------
void    CMySqlData::ExportBool( char * aB, data_struct_descriptor_t & aF, const char * aD ) {
    bool    val = *DATA_LOCATION( aD, aF, bool );
    ::strcat( aB, val ? "'1'" : "'0'" );
}

// --------------------------------------------------------------------
void    CMySqlData::ExportDword( char * aB, data_struct_descriptor_t & aF, const char * aD ) {
    ::sprintf( aB + ::strlen( aB ), "%d", *DATA_LOCATION( aD, aF, dword_t ) );
}

// --------------------------------------------------------------------
void    CMySqlData::ExportAscz( char * aB, data_struct_descriptor_t & aF, const char * aD ) {

    ::strcat( aB, "'" );

    // ----------------------------------------------------------------
    // Encrypt as needed
    // ----------------------------------------------------------------
    if  ( aF.crypt ) {
        char *  ep = aB + ::strlen( aB );
        ::strcat( aB, DATA_LOCATION( aD, aF, char ) );
        ::my_encrypt( ep );
    }
    else {
        CMySqlQuote q;
        ::strcat( aB, q.Quote( DATA_LOCATION( aD, aF, char ) ) );
    }
    ::strcat( aB, "'" );
}

// --------------------------------------------------------------------
// EOF: CMySqlData.cxx
// --------------------------------------------------------------------
