// --------------------------------------------------------------------
// CFtpSocket.cpp
// Whatis:  Socket class to be used by the FTP system
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 10-NOV-2003		Created this source
// --------------------------------------------------------------------
#include	"CFtpSocket.hxx"

// --------------------------------------------------------------------
static unsigned long __stdcall __MyFtpSocketThread( void * aFtpSocket ) {
	try	{
		((CFtpSocket *)aFtpSocket)->Run();
    }
    catch	( ... ) {
    }
	::ExitThread( 0 );
	return	0;
}

// --------------------------------------------------------------
CFtpSocket::CFtpSocket  (	const char *	aHost,
							unsigned short	aPort,
							CFtpStream *	aInput,
							CFtpStream *	aOutput ) {
	CFtpSocket::Cleanup();
	itsHost		= ::my_private_strdup( aHost );
	itsPort		= aPort;
	itsInput	= aInput;
	itsOutput	= aOutput;

	if	( ::CreateThread(
    			NULL,					// Security
                32768,					// Stack
                __MyFtpSocketThread,	// Thread
                (void *)this,			// Argument
                0,						// Flags
    			&itsThreadId ) <= 0 ) {
		itsError = CError( "Unable to start thread" );
		itsState = ftp_socket_state_error;
	}
}

// --------------------------------------------------------------
CFtpSocket::CFtpSocket  (	CTcpSocket *	aListen,
							CFtpStream *	aInput,
							CFtpStream *	aOutput ) {
	CFtpSocket::Cleanup();
	itsListen	= aListen;
	itsInput	= aInput;
	itsOutput	= aOutput;

	if	( ::CreateThread(
    			NULL,					// Security
                32768,					// Stack
                __MyFtpSocketThread,	// Thread
                (void *)this,			// Argument
                0,						// Flags
    			&itsThreadId ) <= 0 ) {
		itsError = CError( "Unable to start thread" );
		itsState = ftp_socket_state_error;
	}
}

// --------------------------------------------------------------
CFtpSocket::~CFtpSocket () {
	while	( itsRunning ) {
		CFtpSocket::Abort();
	}
	CFtpSocket::Free();
}

// --------------------------------------------------------------
void	CFtpSocket::Run			( void ) {
	itsRunning = true;
	while	( itsState != ftp_socket_state_abort ) {
		itsMutex.Lock();
		try {
			switch	( itsState ) {
				case	ftp_socket_state_connect:
				CFtpSocket::Connect();
				break;

				case	ftp_socket_state_transfer:
				CFtpSocket::Transfer();
                break;

				default:
                break;
			}
		}
		catch ( CError e ) {
			itsError = e;
			itsState = ftp_socket_state_error;
		}
		catch ( ... ) {
			itsError = CError( "Unknown error" );
			itsState = ftp_socket_state_error;
		}
		itsMutex.Unlock();
		switch	( itsState ) {
			case	ftp_socket_state_connect:	SLEEP( 100 );	break;
			case	ftp_socket_state_hangup:	SLEEP( 100 );	break;
		}
	}
	itsRunning = false;
}

// --------------------------------------------------------------
void	CFtpSocket::Abort		( void ) {
	itsMutex.Lock();
	itsState = ftp_socket_state_abort;
	itsMutex.Unlock();
}

// --------------------------------------------------------------
const char *		CFtpSocket::Host	( void ) const {
	return	itsHost;
}

// --------------------------------------------------------------
unsigned short		CFtpSocket::Port	( void ) const {
	return	itsPort;
}

// --------------------------------------------------------------
ftp_socket_state_t	CFtpSocket::State	( void ) const {
	return	itsState;
}

// --------------------------------------------------------------
CError				CFtpSocket::Error	( void ) const {
	return	itsError;
}

// --------------------------------------------------------------
CTcpSocket *		CFtpSocket::Socket	( void ) {
	return	itsSocket;
}

// --------------------------------------------------------------
void	CFtpSocket::Connect		( void ) {
	if	( itsListen ) {
		itsSocket = itsListen->Accept();

		if	( itsSocket ) {
			itsState = ftp_socket_state_transfer;
		}
	}
	else {
		if	( ! itsSocket ) {
			itsSocket = new CTcpSocket( socket_type_stream );
		}
		if	( itsSocket->Connect( itsHost, itsPort ) ) {
			itsState = ftp_socket_state_transfer;
		}
	}
}

// --------------------------------------------------------------
void	CFtpSocket::Transfer	( void ) {
	int		rxch;

	// -------------------------
	// Copy from input to socket
	// -------------------------
	if	( itsInput ) {
    	rxch = itsInput->Read();
        if		( rxch >= 0 ) {
        	itsSocket->Write( (char)rxch );
        }
        else if	( rxch == FTP_STREAM_READ_IDLE ) {
        	itsSocket->Flush();
        }
        else if	( rxch == FTP_STREAM_READ_EOF ) {
        	itsSocket->Flush();
			itsState = ftp_socket_state_hangup;
        }
        else if	( rxch == FTP_STREAM_READ_ERROR ) {
        	throw CError( "Cant read input stream" );
        }
    }

	// -------------------------
	// Copy from socket to output
	// -------------------------
    if	( itsOutput ) {
		rxch = itsSocket->Read();
        if		( rxch >= 0 ) {
        	itsOutput->Write( (char)rxch );
        }
        else if	( rxch == SOCKET_READ_IDLE ) {
        	itsOutput->Flush();
        }
        else if	( rxch == SOCKET_READ_DISCONNECT ) {
        	itsOutput->Flush();
			itsState = ftp_socket_state_hangup;
        }
        else if	( rxch == FTP_STREAM_READ_ERROR ) {
        	throw CError( "Cant read input stream" );
        }
    }
}

// --------------------------------------------------------------
void	CFtpSocket::Cleanup		( void ) {
	itsThreadId	= 0;
	itsHost		= NULL;
	itsPort		= 0;
	itsState	= ftp_socket_state_connect;
	itsRunning	= false;
	itsListen	= NULL;
	itsSocket	= NULL;
	itsInput	= NULL;
	itsOutput	= NULL;
}

// --------------------------------------------------------------
void	CFtpSocket::Free		( void ) {
	if	( itsHost )		delete [] itsHost;
	if	( itsSocket )	delete itsSocket;
	CFtpSocket::Cleanup();
}

// --------------------------------------------------------------
// EOF: CFtpSocket.cpp
// --------------------------------------------------------------
