// --------------------------------------------------------------------
// CFurserClient.cpp
// Whatis:  Furser Client class
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 23-DEC-2003     Created this source
// --------------------------------------------------------------------
#include	"CFurserClient.h"
#include	"CError.hxx"

#define		__FURSER_IDENT	"FURSER V1.0\n"

// --------------------------------------------------------------------
CFurserClient::CFurserClient	( CTcpSocket * aSock, const char * aPwd ) {
	CFurserClient::Cleanup();
	itsSocket = aSock;
    try {
    	::strcpy( itsAddress, aSock->PeerAddr() );
		CFurserClient::RxResponse( __FURSER_IDENT );
		CFurserClient::TxResponse( "Server OK\n" );
		CFurserClient::ReadPasswd( aPwd );
		CFurserClient::TxResponse( "Password OK\n" );
    }
	catch	( CError e ) {
		CFurserClient::Free();
		throw	e;
	}
	catch	( ... ) {
		CFurserClient::Free();
        throw CError( "Connection failed" );
	}
}

// --------------------------------------------------------------------
CFurserClient::CFurserClient	( const char * aIp, unsigned short aPort, const char * aPwd ) {
	time_t	mytime;
	CFurserClient::Cleanup();
	itsSocket = new CTcpSocket( socket_type_stream );

	mytime = ::time( NULL ) + 10;
	try	{
    	::strcpy( itsAddress, aIp );
		while	( ! itsSocket->Connect( aIp, aPort ) ) {
			if	( mytime < ::time( NULL ) ) {
				throw 1;
			}
		}
		CFurserClient::TxResponse( __FURSER_IDENT );
		CFurserClient::RxResponse( "Server OK\n" );
		CFurserClient::SendPasswd( aPwd );
		try {
			CFurserClient::RxResponse( "Password OK\n" );
        }
        catch ( ... ) {
        	throw CError( "Invalid password" );
        }
	}
	catch	( CError e ) {
		CFurserClient::Free();
		throw	e;
	}
	catch	( ... ) {
		CFurserClient::Free();
        throw CError( "Connection failed" );
	}
}

// --------------------------------------------------------------------
CFurserClient::~CFurserClient	() {
	CFurserClient::Free();
}

// --------------------------------------------------------------------
void	CFurserClient::TxResponse	( const char * aData ) {
	CFurserClient::TxData( aData, ::strlen( aData ) + 1 );
}

// --------------------------------------------------------------------
void	CFurserClient::RxResponse	( const char * aWait ) {
	char *	rsp = CFurserClient::RxData();
    if	( ::strcmp( rsp, aWait ) ) {
    	delete [] rsp;
        throw CError( "Invalid response" );
    }
    delete [] rsp;
}

// --------------------------------------------------------------------
void	CFurserClient::TxData		( const char * aD, dword_t aS ) {
	try {
		if	( itsSocket ) {
            CFurserClient::TxDataDword( aS );
			itsSocket->Write( aD, aS );
			itsSocket->Flush();
		}
	}
	catch	( CError e ) {
		CFurserClient::Free();
		throw	e;
	}
	catch	( ... ) {
		CFurserClient::Free();
        throw CError( "Connection failed" );
	}
}

// --------------------------------------------------------------------
void	CFurserClient::TxDataDword	( dword_t aDw ) {
	try {
		if	( itsSocket ) {
			int		i;
			for	( i = 0; i < 4; i++ ) {
				itsSocket->Write( (char)(aDw & 0xff) );
				aDw = aDw >> 8;
			}
			itsSocket->Flush();
		}
	}
	catch	( CError e ) {
		CFurserClient::Free();
		throw	e;
	}
	catch	( ... ) {
		CFurserClient::Free();
        throw CError( "Connection failed" );
	}
}

// --------------------------------------------------------------------
void	CFurserClient::TxDataInt	( int aInt ) {
	try {
		if	( itsSocket ) {
			if	( aInt < 0 ) {
				itsSocket->Write( (char)1 );
                CFurserClient::TxDataDword( (dword_t)(-aInt) );
            }
            else {
				itsSocket->Write( (char)0 );
                CFurserClient::TxDataDword( (dword_t)aInt );
            }
		}
	}
	catch	( CError e ) {
		CFurserClient::Free();
		throw	e;
	}
	catch	( ... ) {
		CFurserClient::Free();
        throw CError( "Connection failed" );
	}
}

// --------------------------------------------------------------------
void	CFurserClient::TxDataBool	( bool aBool ) {
	try {
		if	( itsSocket ) {
			itsSocket->Write( (char)(aBool ? 1 : 0) );
			itsSocket->Flush();
		}
	}
	catch	( CError e ) {
		CFurserClient::Free();
		throw	e;
	}
	catch	( ... ) {
		CFurserClient::Free();
        throw CError( "Connection failed" );
	}
}

// --------------------------------------------------------------------
void	CFurserClient::TxDataTime	( time_t aTime ) {
	try {
		if	( itsSocket ) {
			struct tm *	mytm = ::localtime( &aTime );
			CFurserClient::TxDataInt( mytm->tm_sec );
			CFurserClient::TxDataInt( mytm->tm_min );
			CFurserClient::TxDataInt( mytm->tm_hour );
			CFurserClient::TxDataInt( mytm->tm_mday );
			CFurserClient::TxDataInt( mytm->tm_mon );
			CFurserClient::TxDataInt( mytm->tm_year );
		}
	}
	catch	( CError e ) {
		CFurserClient::Free();
		throw	e;
	}
	catch	( ... ) {
		CFurserClient::Free();
        throw CError( "Connection failed" );
	}
}

// --------------------------------------------------------------------
char *	CFurserClient::RxData		( void ) {
	char *	mydata = NULL;

	try {
		if	( itsSocket ) {
			dword_t	i;
			dword_t	s = CFurserClient::RxDataDword();
			if	( s > 100000 )	throw CError( "Invalid data block" );
			if	( s == 0 )		throw CError( "Invalid data block" );
			mydata = new char [s];
			for	( i = 0; i < s; i++ ) {
				mydata[i] = CFurserClient::ReadByte();
			}
		}
	}
	catch	( CError e ) {
		CFurserClient::Free();
		if	( mydata ) {
			delete [] mydata;
			mydata = NULL;
		}
		throw	e;
	}
	catch	( ... ) {
		CFurserClient::Free();
		if	( mydata ) {
			delete [] mydata;
			mydata = NULL;
		}
        throw CError( "Connection failed" );
	}
	return	mydata;
}

// --------------------------------------------------------------------
dword_t	CFurserClient::RxDataDword		( void ) {
	dword_t	mydword	= 0;

	try {
		if	( itsSocket ) {
			int	i;
			for	( i = 0; i < 4; i++ ) {
				mydword = mydword >> 8;
				mydword = mydword | ((dword_t)CFurserClient::ReadByte() << 24);
			}
		}
	}
	catch	( CError e ) {
		CFurserClient::Free();
		throw	e;
	}
	catch	( ... ) {
		CFurserClient::Free();
        throw CError( "Connection failed" );
	}
	return	mydword;
}

// --------------------------------------------------------------------
int		CFurserClient::RxDataInt		( void ) {
	int	myint	= 0;

	try {
		if	( itsSocket ) {
			if	( CFurserClient::ReadByte() ) {
				myint = (int)CFurserClient::RxDataDword();
                myint = -myint;
            }
            else {
				myint = (int)CFurserClient::RxDataDword();
            }
		}
	}
	catch	( CError e ) {
		CFurserClient::Free();
		throw	e;
	}
	catch	( ... ) {
		CFurserClient::Free();
        throw CError( "Connection failed" );
	}
	return	myint;
}

// --------------------------------------------------------------------
bool	CFurserClient::RxDataBool	( void ) {
	dword_t	mybool	= false;

	try {
		if	( itsSocket ) {
			mybool = (CFurserClient::ReadByte() != 0);
		}
	}
	catch	( CError e ) {
		CFurserClient::Free();
		throw	e;
	}
	catch	( ... ) {
		CFurserClient::Free();
        throw CError( "Connection failed" );
	}
	return	mybool;
}

// --------------------------------------------------------------------
time_t	CFurserClient::RxDataTime	( void ) {
	time_t	mytime;
	try {
		if	( itsSocket ) {
			struct tm	mytm;
            mytime = ::time( NULL );
            ::memcpy( &mytm, ::localtime( &mytime ), sizeof( mytm ) );
			mytm.tm_sec    = CFurserClient::RxDataInt();
			mytm.tm_min    = CFurserClient::RxDataInt();
			mytm.tm_hour   = CFurserClient::RxDataInt();
			mytm.tm_mday   = CFurserClient::RxDataInt();
			mytm.tm_mon    = CFurserClient::RxDataInt();
			mytm.tm_year   = CFurserClient::RxDataInt();
            mytime = ::mktime( &mytm );
		}
	}
	catch	( CError e ) {
		CFurserClient::Free();
		throw	e;
	}
	catch	( ... ) {
		CFurserClient::Free();
        throw CError( "Connection failed" );
	}
    return	mytime;
}

// --------------------------------------------------------------------
void	CFurserClient::SendPasswd	( const char * aPw ) {
	char	pwd	[512];
	::memset( pwd, 0, sizeof( pwd ) );
	::strcpy( pwd, aPw );
	::my_encrypt( pwd );
	CFurserClient::TxData( pwd, sizeof( pwd ) );
}

// --------------------------------------------------------------------
void	CFurserClient::ReadPasswd	( const char * aPw ) {
	char *	pwd = CFurserClient::RxData();
	::my_decrypt( pwd );
	if	( ::strcmp( aPw, pwd ) ) {
		delete [] pwd;
		throw CError( "Invalid Password" );
	}
	else {
		delete [] pwd;
	}
}

// --------------------------------------------------------------------
byte_t	CFurserClient::ReadByte	( void ) {
	int		mybyte = 0;
	time_t	mytime = ::time( NULL ) + 10;
	if	( itsSocket ) {
		do	{
			if	( mytime < ::time( NULL ) ) {
				throw CError( "Connection timed out" );
			}
			mybyte = itsSocket->Read();
			if		( mybyte == SOCKET_READ_IDLE )	continue;
			else if	( mybyte < 0 )	throw CError( "Connection failed" );
		}	while	( mybyte < 0 );
	}
	return	(byte_t)(mybyte & 0xff);
}

// --------------------------------------------------------------------
byte_t	CFurserClient::ReadCommand	( void ) {
	int		mybyte = 0;
	if	( itsSocket ) {
		do	{
			mybyte = itsSocket->Read();
			if		( mybyte == SOCKET_READ_IDLE ) {
	        	SLEEP( 1000 );
            	continue;
            }
			else if	( mybyte < 0 )	throw CError( "Connection failed" );
		}	while	( mybyte < 0 );
	}
	return	(byte_t)(mybyte & 0xff);
}

// --------------------------------------------------------------------
byte_t	CFurserClient::ReadCommand	( bool & aAbort ) {
	int		mybyte = 0;
	if	( itsSocket ) {
		do	{
			mybyte = itsSocket->Read();
			if		( mybyte == SOCKET_READ_IDLE ) {
	        	SLEEP( 1000 );
            	continue;
            }
			else if	( mybyte < 0 )	throw CError( "Connection failed" );
		}	while	( ( mybyte < 0 ) && ( aAbort == false ) );
	}
	return	(byte_t)(mybyte & 0xff);
}

// --------------------------------------------------------------------
void	CFurserClient::SendCommand	( byte_t aC ) {
	if	( itsSocket ) {
		itsSocket->Write( (char)aC );
        itsSocket->Flush();
	}
}

// --------------------------------------------------------------------
const char *	CFurserClient::Address( void ) const {
	return itsAddress;
}

// --------------------------------------------------------------------
void	CFurserClient::Free		( void ) {
	if	( itsSocket ) {
		delete	itsSocket;
	}
	CFurserClient::Cleanup();
}

// --------------------------------------------------------------------
void	CFurserClient::Cleanup	( void ) {
	itsSocket		= NULL;
    itsAddress[0]	= 0;
}

// --------------------------------------------------------------------
// EOF:	CFurserClient.cpp
// --------------------------------------------------------------------
