// --------------------------------------------------------------------
// FurserScheduleThread.cpp
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 23-DEC-2003     Created this source
// --------------------------------------------------------------------
#include	"FurserDaemon.h"
#include	"CFurserSrData.h"
#include	"CError.hxx"

//---------------------------------------------------------------------------
// Scheduled tasks currently running
//---------------------------------------------------------------------------
static	CFurserTask_l	__MyTaskList;

//---------------------------------------------------------------------------
// Remove a schedule entry from the queue
//---------------------------------------------------------------------------
static	void	__RemoveScheduleEntry	(	const CFurserSrSrvr & aSrvr ) {
	CFurserTask_li	task;
    try	{
    	::FurserLock();
        for	(	task  = __MyTaskList.begin();
        		task != __MyTaskList.end();
                task++ ) {
			if	( aSrvr != (*task).Server() )	continue;
            ::FurserLog( "TASK REMOVE SERVER", aSrvr.ServerName() );
			::FurserStopGameServer( *task );
            __MyTaskList.erase( task );
            break;
		}
    	::FurserUnlock();
    }
    catch ( ... ) {
    	::FurserUnlock();
    }
}

//---------------------------------------------------------------------------
// Start
//---------------------------------------------------------------------------
static	void	__StartScheduleEntry(	const CFurserSrSrvr & aSrvr ) {
	CFurserTask_li	task;
    CFurserTask		newtask;
    try	{
    	::FurserLock();
		// Ensure there is no entry already
        for	(	task  = __MyTaskList.begin();
        		task != __MyTaskList.end();
                task++ ) {
			if	( aSrvr == (*task).Server() ) {
               	return;
            }
		}
		newtask.Server( aSrvr.ServerName() );
		::FurserStartGameServer( newtask, aSrvr );
		__MyTaskList.push_back( newtask );
    	::FurserUnlock();
    }
    catch ( CError e ) {
		::FurserLog( "ERROR", e.Error() );
    	::FurserUnlock();
    }
    catch ( ... ) {
		::FurserLog( "ERROR", "Unknown error" );
    	::FurserUnlock();
    }
}

//---------------------------------------------------------------------------
// Do something with the schedule entry
//---------------------------------------------------------------------------
static	void	__ExecuteScheduleEntry	(	const CFurserSrSrvr & aSrvr,
                                            int aTask ) {
	switch	( aTask ) {
		case	EVENT_TASK_NOTHING:
		::FurserLog( "TASK NOP SERVER", aSrvr.ServerName() );
		break;

		case	EVENT_TASK_START:
		::FurserLog( "TASK START SERVER", aSrvr.ServerName() );
        __StartScheduleEntry( aSrvr );
        break;

		case	EVENT_TASK_STOP:
		::FurserLog( "TASK STOP SERVER", aSrvr.ServerName() );
        __RemoveScheduleEntry( aSrvr );
        break;

		case	EVENT_TASK_RESTART:
		::FurserLog( "TASK RESTART SERVER", aSrvr.ServerName() );
        __RemoveScheduleEntry( aSrvr );
        __StartScheduleEntry( aSrvr );
        break;
    }
}

//---------------------------------------------------------------------------
// Inspect if a schedule entry is ready fo action
//---------------------------------------------------------------------------
static	void	__InspectScheduleEntry	(	const CFurserSrSrvr & aSrvr,
											const CFurserSrEvnt & aEvnt,
                                            time_t	aTime ) {
	time_t	mystart	= aEvnt.StartTime() - (aEvnt.StartTime() % 60);
	time_t	myend	= aEvnt.EndTime() - (aEvnt.EndTime() % 60);

	// How should this entry be inspected ?
    switch	( aEvnt.EventType() ) {
    	case	EVENT_TYPE_ONCE:
        if		( aTime == mystart  ) {
        	__ExecuteScheduleEntry( aSrvr, aEvnt.StartTask() );
        }
        else if	( aTime == myend ) {
        	__ExecuteScheduleEntry( aSrvr, aEvnt.EndTask() );
        }
        break;

    	case	EVENT_TYPE_HOURLY:
        if	( (aTime % 3600) == (mystart % 3600) ) {
        	__ExecuteScheduleEntry( aSrvr, aEvnt.StartTask() );
        }
        break;

    	case	EVENT_TYPE_DAILY:
        if	( (aTime % (3600*24)) == (mystart % (3600*24)) ) {
        	__ExecuteScheduleEntry( aSrvr, aEvnt.StartTask() );
        }
        break;

    	case	EVENT_TYPE_WEEKLY:
        if	( (aTime % (3600*24*7)) == (mystart % (3600*24*7)) ) {
        	__ExecuteScheduleEntry( aSrvr, aEvnt.StartTask() );
            break;
        }
        break;
    }
}

//---------------------------------------------------------------------------
// Process a server
//---------------------------------------------------------------------------
static	void	__ProcessServer		( const CFurserSrSrvr & aSrvr, time_t aTime ) {
	CFurserSrEvnt_lci	loop;
	for	(	loop  = aSrvr.EventList().begin();
    		loop != aSrvr.EventList().end();
			loop++ ) {
    	if		( ! aSrvr.Running() ) {
        	__RemoveScheduleEntry( aSrvr );
        }
        else if	( ! (*loop).Enabled() ) {
        	__RemoveScheduleEntry( aSrvr );
        }
        else {
        	__InspectScheduleEntry( aSrvr, *loop, aTime );
        }
    }
}

//---------------------------------------------------------------------------
// Process a schedule
//---------------------------------------------------------------------------
static	void	__ProcessSchedule	( time_t aTime ) {
	CWinIniFile		ini;
    CFurserSrSrvr	srvr;
	CFurserTask_li	task;
    char			srvrsect[64];
    int				count, ix;

	::FurserLoadServerData();
   	::FurserLock();
	ini = FurserIni;
   	::FurserUnlock();

    count = ini.IntValue( "Servers", "ServerCount" );

	// Remove any entries that have no more server configuration
   	::FurserLock();
    try	{
		task = __MyTaskList.begin();
        while	( task != __MyTaskList.end() ) {
		    // Loop through all servers
		    for	( ix = 0; ix < count; ix++ ) {
    			::sprintf( srvrsect, "Server_%i", ix );
				if	( ! ::strcmp( (*task).Server(), ini.Value( srvrsect, "ServerName" ) ) ) {
					break;
                }
		    }
            if	( ix < count ) {
            	task++;
            }
            else {
	            ::FurserLog( "TASK REMOVE UNCONFIGURED SERVER", (*task).Server() );
				::FurserStopGameServer( *task );
        	    __MyTaskList.erase( task );
				task = __MyTaskList.begin();
            }
		}
    	::FurserUnlock();
    }
    catch ( ... ) {
    	::FurserUnlock();
    }

    // Loop through all servers
    for	( ix = 0; ix < count; ix++ ) {
    	::sprintf( srvrsect, "Server_%i", ix );
		srvr.Load( ini, srvrsect );
        __ProcessServer( srvr, aTime );
    }

	// Restart all entries that have crashed
    try	{
    	::FurserLock();
        for	(	task  = __MyTaskList.begin();
        		task != __MyTaskList.end();
                task++ ) {
        	::FurserCheckRunningState( *task );
		}
    	::FurserUnlock();
    }
    catch ( ... ) {
    	::FurserUnlock();
    }

}

//---------------------------------------------------------------------------
// The schedule process
//---------------------------------------------------------------------------
extern	void	FurserScheduleThread	( void * ) {
	time_t	lasttime;
	::FurserLock();
	FurserStatus.running++;
	::FurserUnlock();

	try	{
    	::FurserLog( "THREAD+", "FurserScheduleThread" );
		lasttime = ::time( NULL );
		while	( ! FurserStatus.abort ) {
			// Make this thread active in 60 second periods
			SLEEP( 100 );
            if	( (::time( NULL ) - lasttime ) < 60 )	continue;

            lasttime = ::time( NULL );

            // If the service is paused ???
            if	( FurserStatus.pause )	continue;

			__ProcessSchedule( lasttime - (lasttime % 60) );
		}
    }

    catch ( ... ) {
    }
   	::FurserLog( "THREAD-", "FurserScheduleThread" );
	::FurserLock();
	FurserStatus.running--;
	::FurserUnlock();
}

//---------------------------------------------------------------------------
// Immediately start a server
//---------------------------------------------------------------------------
extern void	FurserServerImmediateStart( CFurserClient & aClient ) {
	CFurserSrData	mydata;
	char *			srvname = NULL;
	::FurserLog( aClient.Address(), "Immediate Game Server Start" );
	if	( ! FurserStatus.error ) {
    	::FurserLock();
		try {
			::FurserLoadServerData();
		    mydata.Load( FurserIni );
	    	srvname = aClient.RxData();
    		mydata.NetImmStart( aClient, srvname, false );
            if	( mydata.HasServer( srvname ) ) {
				__ExecuteScheduleEntry( mydata.GetServer( srvname ),  EVENT_TASK_START );
			}
			delete [] srvname;
            ::FurserUnlock();
    	}
    	catch ( ... ) {
			if	( srvname ) delete [] srvname;
            ::FurserUnlock();
			throw;
	    }
	}
	else {
		throw CError( FurserStatus.error );
	}
}

//---------------------------------------------------------------------------
// Immediately stop a server
//---------------------------------------------------------------------------
extern void	FurserServerImmediateStop( CFurserClient & aClient ) {
	CFurserSrData	mydata;
	char *			srvname = NULL;
	::FurserLog( aClient.Address(), "Immediate Game Server Stop" );
	if	( ! FurserStatus.error ) {
    	::FurserLock();
		try {
			::FurserLoadServerData();
		    mydata.Load( FurserIni );
	    	srvname = aClient.RxData();
    		mydata.NetImmStop( aClient, srvname, false );
            if	( mydata.HasServer( srvname ) ) {
				__ExecuteScheduleEntry( mydata.GetServer( srvname ),  EVENT_TASK_STOP );
			}
			delete [] srvname;
            ::FurserUnlock();
    	}
    	catch ( ... ) {
			if	( srvname ) delete [] srvname;
            ::FurserUnlock();
			throw;
	    }
	}
	else {
		throw CError( FurserStatus.error );
	}
}

//---------------------------------------------------------------------------
// Scan running servers
//---------------------------------------------------------------------------
extern void	FurserScanRunningServers( CFurserSrData & aData ) {
	CFurserSrSrvr_li	srvr;
	CFurserTask_li		task;
    for	(	srvr  = aData.ServerMList().begin();
    		srvr != aData.ServerMList().end();
            srvr++ ) {
    	::FurserLock();
        for	(	task  = __MyTaskList.begin();
        		task != __MyTaskList.end();
                task++ ) {
			if	( (*srvr) == (*task).Server() ) {
				(*srvr).GameRunning( (*task).Running() );
            }
		}
        ::FurserUnlock();
    }
}

// --------------------------------------------------------------------
// EOF: FurserScheduleThread.cpp
// --------------------------------------------------------------------
