// --------------------------------------------------------------------
// WinFurserDaemonService.cpp
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 23-DEC-2003     Created this source
// --------------------------------------------------------------------
#include	"FurserDaemon.h"
#include	"CError.hxx"

//---------------------------------------------------------------------------
// local:	Stuff needed for service
//---------------------------------------------------------------------------
static  const char *			__MyServiceName			= "FurserDaemonService";
static  const char *			__MyInstanceName		= "FurserDaemonInstance";
static  SERVICE_STATUS_HANDLE	__MyStatusHandle		= NULL;
static  HANDLE					__MyMutexHandle			= NULL;
static  DWORD                   __MyServiceStatus		= SERVICE_STOPPED;

// ----------------------------------------------------------------
// local:	Throw and log system error message
// ----------------------------------------------------------------
static	void	__WinSystemError( void ) {
	LPVOID	lpMsgBuf	= NULL;

    try	{
		FormatMessage	(
			FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
			NULL,
    		GetLastError(),
			MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
			(LPTSTR) &lpMsgBuf,
    		0,
    		NULL );
		if	( lpMsgBuf ) {
        	::my_strfix( (char *)lpMsgBuf );
			throw	CError( (const char *)lpMsgBuf );
        }
        else {
        	throw	CError( "Unknown system error" );
        }
    }
	catch	( ... ) {
		if	( lpMsgBuf )	LocalFree( lpMsgBuf );
		throw;
    }
}

// ----------------------------------------------------------------
// local:	WIN Service status update
// ----------------------------------------------------------------
static  bool	__winServiceStatus( DWORD dwCurrentState,
                                    DWORD dwWin32ExitCode,
                                    DWORD dwServiceSpecificExitCode,
                                    DWORD dwCheckPoint,
                                    DWORD dwWaitHint ) {
    SERVICE_STATUS serviceStatus;

    // Fill in all of the SERVICE_STATUS fields
    serviceStatus.dwServiceType = SERVICE_WIN32_OWN_PROCESS;
    serviceStatus.dwCurrentState = dwCurrentState;

    // If in the process of something, then accept
    // no control events, else accept anything
    if ( dwCurrentState == SERVICE_START_PENDING ) {
        serviceStatus.dwControlsAccepted = 0;
    }
    else {
        serviceStatus.dwControlsAccepted =
            SERVICE_ACCEPT_STOP |
            SERVICE_ACCEPT_PAUSE_CONTINUE |
            SERVICE_ACCEPT_SHUTDOWN;
    }

    // if a specific exit code is defined, set up
    // the Win32 exit code properly
    if  ( dwServiceSpecificExitCode == 0 ) {
        serviceStatus.dwWin32ExitCode = dwWin32ExitCode;
    }
    else {
        serviceStatus.dwWin32ExitCode = ERROR_SERVICE_SPECIFIC_ERROR;
    }
    serviceStatus.dwServiceSpecificExitCode =   dwServiceSpecificExitCode;
    serviceStatus.dwCheckPoint = dwCheckPoint;
    serviceStatus.dwWaitHint = dwWaitHint;

    // Pass the status record to the SCM
    if	( ! ::SetServiceStatus ( __MyStatusHandle, &serviceStatus ) ) {
		FurserStatus.abort	= true;
        __MyServiceStatus	= SERVICE_STOPPED;
        return false;
    }
    return true;
}

// ----------------------------------------------------------------
// local:	WIN Service control dispatcher
// ----------------------------------------------------------------
static  VOID    __winServiceCtlHandler ( DWORD controlCode ) {
    switch( controlCode ) {

        // Pause the service
        case    SERVICE_CONTROL_PAUSE:
		::FurserLog( "SERVICE_CONTROL_PAUSE", __MyServiceName );
        if  ( !FurserStatus.pause && !FurserStatus.abort ) {
            // Tell the SCM we're about to Pause.
            __winServiceStatus( SERVICE_PAUSE_PENDING, NO_ERROR, 0, 1, 1000 );
            FurserStatus.pause = true;
            __MyServiceStatus = SERVICE_PAUSED;
         }
         break;

        // Resume from a pause
        case    SERVICE_CONTROL_CONTINUE:
		::FurserLog( "SERVICE_CONTROL_CONTINUE", __MyServiceName );
        if  ( FurserStatus.pause && !FurserStatus.abort ) {
            // Tell the SCM we're about to Resume.
            __winServiceStatus( SERVICE_CONTINUE_PENDING, NO_ERROR, 0, 1, 1000);
            FurserStatus.pause = true;
            __MyServiceStatus = SERVICE_RUNNING;
        }
        break;

        // Update the current status for the SCM.
        case SERVICE_CONTROL_INTERROGATE:
        break;

        case SERVICE_CONTROL_SHUTDOWN:
        case SERVICE_CONTROL_STOP:
		::FurserLog( "SERVICE_CONTROL_STOP", __MyServiceName );
        // Tell the SCM we're about to Stop.
        __MyServiceStatus = SERVICE_STOP_PENDING;
        __winServiceStatus(SERVICE_STOP_PENDING, NO_ERROR, 0, 1, 5000 );
		FurserStatus.abort	= true;
        __MyServiceStatus	= SERVICE_STOPPED;
		for	( int i = 0; i < 10; i++ ) {
        	if	( ! FurserStatus.running )	break;
        	SLEEP( 1000 );
        }
        break;

        default:
        break;
    }
    __winServiceStatus( __MyServiceStatus, NO_ERROR, 0, 0, 0 );
}

// ----------------------------------------------------------------
// local:	WIN Service dispatcher
// ----------------------------------------------------------------
static  void    __winServiceDispatcher ( DWORD, LPTSTR * ) {

	::FurserLog( "WINSERVICE MAIN THREAD", __MyServiceName );

    // First we must call the Registration function
    __MyStatusHandle = :: RegisterServiceCtrlHandler( __MyServiceName, (LPHANDLER_FUNCTION)__winServiceCtlHandler );
    if  ( ! __MyStatusHandle ) {
        return;
    }

    // Next Notify the Service Control Manager of progress
    if	( ! __winServiceStatus( SERVICE_START_PENDING, NO_ERROR, 0, 1, 5000 ) ) {
        return;
    }

    // Notify the SCM of progress again
    if	( ! __winServiceStatus( SERVICE_START_PENDING, NO_ERROR, 0, 2, 1000 ) ) {
        return;
    }

    // Start the service execution thread by calling our StartServiceThread function...
	::FurserStartThread( FurserDaemonProcess, NULL );

    // The service is now running.  Notify the SCM of this fact.
    if	( ! __winServiceStatus( SERVICE_RUNNING, NO_ERROR, 0, 0, 0 ) ) {
        return;
    }

    // Now just wait for our killed service signal, and then exit, which
    // terminates the service!
    while	( !FurserStatus.abort ) {
    	SLEEP( 1000 );
    }
}

//---------------------------------------------------------------------------
// Test for existence of the service
//---------------------------------------------------------------------------
extern	bool	FurserDaemonTestWinService( void ) {
	SC_HANDLE		myService	= 0;
    SC_HANDLE		scm;

    scm = ::OpenSCManager( 0, 0, SC_MANAGER_CREATE_SERVICE );
   	if  (  scm ) {
	    myService = ::OpenService( scm, __MyServiceName, SERVICE_ALL_ACCESS | DELETE );
    }

	if	( myService )	::CloseServiceHandle(myService);
	if	( scm )			::CloseServiceHandle(scm);

	return	( ( scm ) && ( myService ) );
}

//---------------------------------------------------------------------------
// Test for existence of process
//---------------------------------------------------------------------------
extern	bool	FurserDaemonTestWinInstance( void ) {
	HANDLE	mutexhndl;
    mutexhndl = OpenMutex( MUTEX_ALL_ACCESS, FALSE, __MyInstanceName );
	if	( mutexhndl )	CloseHandle( mutexhndl );
	return	mutexhndl != NULL;
}

//---------------------------------------------------------------------------
// Create named mutex to block others from coming in
//---------------------------------------------------------------------------
extern	void	FurserDaemonCreateWinInstance( void ) {
    __MyMutexHandle = CreateMutex( NULL, FALSE, __MyInstanceName );
}

//---------------------------------------------------------------------------
// Remove the named mutex
//---------------------------------------------------------------------------
extern	void	FurserDaemonDeleteWinInstance( void ) {
	if	( __MyMutexHandle )	CloseHandle( __MyMutexHandle );
    __MyMutexHandle = NULL;
}

//---------------------------------------------------------------------------
// Installing a windows service
//---------------------------------------------------------------------------
extern	void	FurserDaemonInstallWinService( const char * aExeName ) {
	SC_HANDLE	myService	= 0;
    SC_HANDLE	scm			= 0;

	::FurserLog( "WINSERVICE INSTALL", __MyServiceName );
	try	{
	    scm = ::OpenSCManager( 0, 0, SC_MANAGER_CREATE_SERVICE );
    	if  ( ! scm )	__WinSystemError();
		myService = ::CreateService(	scm,
        								__MyServiceName,
								        "FurserDaemon: Remote Game Server Administrator",
								        SERVICE_ALL_ACCESS,
								        SERVICE_WIN32_OWN_PROCESS,
								        SERVICE_AUTO_START,
								        SERVICE_ERROR_NORMAL,
								        aExeName,
							   	        NULL,
                                        0,
                                        NULL,
                                        NULL,
                                        NULL );
		if  ( ! myService )	__WinSystemError();
		::FurserLog( "WINSERVICE INSTALL", "Done" );
		::CloseServiceHandle(myService);
		::CloseServiceHandle(scm);
    }
	catch	( CError e ) {
		::FurserLog( "WINSERVICE INSTALL ERROR", e.Error() );
		if	( myService )	::CloseServiceHandle(myService);
		if	( scm )			::CloseServiceHandle(scm);
        throw;
    }
    catch	( ... ) {
		::FurserLog( "WINSERVICE INSTALL ERROR", "Unknown error" );
		if	( myService )	::CloseServiceHandle(myService);
		if	( scm )			::CloseServiceHandle(scm);
        throw;
    }
}

//---------------------------------------------------------------------------
// Removing a windows service
//---------------------------------------------------------------------------
extern	void	FurserDaemonRemoveWinService( const char * aExeName ) {
    SC_HANDLE       myService	= 0;
    SC_HANDLE		scm			= 0;
    SERVICE_STATUS  status;

    try	{
		::FurserLog( "WINSERVICE REMOVE", __MyServiceName );

    	// Open a Service Control Manager connection
	    scm = ::OpenSCManager( NULL, NULL, SC_MANAGER_CREATE_SERVICE );
    	if  ( ! scm )	__WinSystemError();

		// Get the service's handle
	    myService = ::OpenService( scm, __MyServiceName, SERVICE_ALL_ACCESS | DELETE );
		if  ( ! myService )	__WinSystemError();

		// Get service status
	    if  ( ! ::QueryServiceStatus( myService, &status ) ) __WinSystemError();

        // Stop service if it is running
		if  ( status.dwCurrentState != SERVICE_STOPPED ) {
	        ::printf( "%s: Stopping service %s ... \n", aExeName, __MyServiceName );

	        if  ( ! ::ControlService( myService, SERVICE_CONTROL_STOP, &status ) ) __WinSystemError();
			SLEEP( 1000 );
	    }

		// Remove the service
		if ( ! ::DeleteService( myService ) ) __WinSystemError();
		::FurserLog( "WINSERVICE REMOVE", "Done" );
		FurserStatus.pause = false;
		FurserStatus.abort = true;
	    ::CloseServiceHandle( myService );
    	::CloseServiceHandle( scm );
    }
	catch	( CError e ) {
		::FurserLog( "WINSERVICE REMOVE ERROR", e.Error() );
		if	( myService )	::CloseServiceHandle(myService);
		if	( scm )			::CloseServiceHandle(scm);
        throw;
    }
    catch	( ... ) {
		::FurserLog( "WINSERVICE REMOVE ERROR", "Unknown error" );
		if	( myService )	::CloseServiceHandle(myService);
		if	( scm )			::CloseServiceHandle(scm);
        throw;
    }
}

// ----------------------------------------------------------------
// Start service
// ----------------------------------------------------------------
extern	void	FurserDaemonStartWinService( void ) {
    SC_HANDLE       myService	= 0;
    SC_HANDLE		scm			= 0;
    SERVICE_STATUS  status;

    try	{
		::FurserLog( "WINSERVICE START", __MyServiceName );

    	// Open a Service Control Manager connection
	    scm = ::OpenSCManager( NULL, NULL, SC_MANAGER_CREATE_SERVICE );
    	if  ( ! scm )	__WinSystemError();

		// Get the service's handle
	    myService = ::OpenService( scm, __MyServiceName, SERVICE_ALL_ACCESS | DELETE );
		if  ( ! myService )	__WinSystemError();

		// Get service status
	    if  ( ! ::QueryServiceStatus( myService, &status ) ) __WinSystemError();

        // Start service if it is not running
		if	( status.dwCurrentState == SERVICE_PAUSED ) {
	        if  ( ! ::ControlService( myService, SERVICE_CONTROL_CONTINUE, &status ) ) __WinSystemError();
			::FurserLog( "WINSERVICE START", "Done" );
            FurserStatus.pause = false;
	    }
		else if	( status.dwCurrentState == SERVICE_STOPPED ) {
	        if  ( ! ::StartService( myService, 0, NULL ) ) __WinSystemError();
			::FurserLog( "WINSERVICE START", "Done" );
            FurserStatus.pause = false;
        }
		else if	( status.dwCurrentState == SERVICE_RUNNING ) {
			::FurserLog( "WINSERVICE START", "Service is already running" );
        }
        else {
			::FurserLog( "WINSERVICE START", "Service is not in controllable state" );
        }

	    ::CloseServiceHandle( myService );
    	::CloseServiceHandle( scm );
    }
	catch	( CError e ) {
		::FurserLog( "WINSERVICE START ERROR", e.Error() );
		if	( myService )	::CloseServiceHandle(myService);
		if	( scm )			::CloseServiceHandle(scm);
        throw;
    }
    catch	( ... ) {
		::FurserLog( "WINSERVICE START ERROR", "Unknown error" );
		if	( myService )	::CloseServiceHandle(myService);
		if	( scm )			::CloseServiceHandle(scm);
        throw;
    }
}

//---------------------------------------------------------------------------
// Fully stop the service
//---------------------------------------------------------------------------
extern	void	FurserDaemonStopWinService( void ) {
    SC_HANDLE       myService	= 0;
    SC_HANDLE		scm			= 0;
    SERVICE_STATUS  status;

    try	{
		::FurserLog( "WINSERVICE STOP", __MyServiceName );

    	// Open a Service Control Manager connection
	    scm = ::OpenSCManager( NULL, NULL, SC_MANAGER_CREATE_SERVICE );
    	if  ( ! scm )	__WinSystemError();

		// Get the service's handle
	    myService = ::OpenService( scm, __MyServiceName, SERVICE_ALL_ACCESS | DELETE );
		if  ( ! myService )	__WinSystemError();

		// Get service status
	    if  ( ! ::QueryServiceStatus( myService, &status ) ) __WinSystemError();

        // Stop service if it is running
		if  ( status.dwCurrentState != SERVICE_STOPPED ) {
	        if  ( ! ::ControlService( myService, SERVICE_CONTROL_STOP, &status ) ) __WinSystemError();
			SLEEP( 1000 );
	    }

		::FurserLog( "WINSERVICE STOP", "Done" );
		FurserStatus.pause = false;
		FurserStatus.abort = true;
	    ::CloseServiceHandle( myService );
    	::CloseServiceHandle( scm );
    }
	catch	( CError e ) {
		::FurserLog( "WINSERVICE STOP ERROR", e.Error() );
		if	( myService )	::CloseServiceHandle(myService);
		if	( scm )			::CloseServiceHandle(scm);
        throw;
    }
    catch	( ... ) {
		::FurserLog( "WINSERVICE STOP ERROR", "Unknown error" );
		if	( myService )	::CloseServiceHandle(myService);
		if	( scm )			::CloseServiceHandle(scm);
        throw;
    }
}

//---------------------------------------------------------------------------
// Pause the service
//---------------------------------------------------------------------------
extern	void	FurserDaemonPauseWinService( void ) {
    SC_HANDLE       myService	= 0;
    SC_HANDLE		scm			= 0;
    SERVICE_STATUS  status;

    try	{
		::FurserLog( "WINSERVICE PAUSE", __MyServiceName );

    	// Open a Service Control Manager connection
	    scm = ::OpenSCManager( NULL, NULL, SC_MANAGER_CREATE_SERVICE );
    	if  ( ! scm )	__WinSystemError();

		// Get the service's handle
	    myService = ::OpenService( scm, __MyServiceName, SERVICE_ALL_ACCESS | DELETE );
		if  ( ! myService )	__WinSystemError();

		// Get service status
	    if  ( ! ::QueryServiceStatus( myService, &status ) ) {
        	throw CError( "Failed to get service status" );
	    }

        // Stop service if it is running
		if  ( status.dwCurrentState != SERVICE_PAUSED ) {
	        if  ( ! ::ControlService( myService, SERVICE_CONTROL_PAUSE, &status ) ) __WinSystemError();
			SLEEP( 1000 );
	    }

		::FurserLog( "WINSERVICE PAUSE", "Done" );
		FurserStatus.pause = false;
	    ::CloseServiceHandle( myService );
    	::CloseServiceHandle( scm );
    }
	catch	( CError e ) {
		::FurserLog( "WINSERVICE PAUSE ERROR", e.Error() );
		if	( myService )	::CloseServiceHandle(myService);
		if	( scm )			::CloseServiceHandle(scm);
        throw;
    }
    catch	( ... ) {
		::FurserLog( "WINSERVICE PAUSE ERROR", "Unknown error" );
		if	( myService )	::CloseServiceHandle(myService);
		if	( scm )			::CloseServiceHandle(scm);
        throw;
    }
}

//---------------------------------------------------------------------------
// Register the dispatcher
//---------------------------------------------------------------------------
extern	void	FurserDaemonRegisterDispatcher( void ) {
    SERVICE_TABLE_ENTRY myServiceTable[] = {
        { (char *)__MyServiceName, (LPSERVICE_MAIN_FUNCTION) __winServiceDispatcher },
        { NULL, NULL }
    };
	::FurserLog( "WINSERVICE REGISTER DISPATCHER", __MyServiceName );
   	::StartServiceCtrlDispatcher( myServiceTable );
}

// --------------------------------------------------------------------
// EOF: WinFurserDaemonService.cpp
// --------------------------------------------------------------------
