// --------------------------------------------------------------------
// CPictureFilterScale.cxx
// Whatis:  Scale the image to given size
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 24-NOV-2001     Created this source
// --------------------------------------------------------------------
#include	"CError.hxx"
#include	"CPictureFilterScale.hxx"

// --------------------------------------------------------------------
// public:	The data processor
// --------------------------------------------------------------------
void	CPictureFilterScale::Process	( CPicturePixmap & aPixmap ) {

	// Lil bit of optimization - image size does not change
	if	( ( itsW == aPixmap.W() ) && ( itsH == aPixmap.H() ) )	return;

	// We should have some size
	if	( ( itsW < 1 ) || ( itsH < 1 ) ) {
		throw CError( "Image size too small" );
	}

	// Pixel ratios
	double	ws	= (double)itsW / (double)aPixmap.W();
	double	hs	= (double)itsH / (double)aPixmap.H();

	// Do the smaller change up first
	if	( ws < hs ) {
		if	( ws > 1.0 ) {
			CPictureFilterScale::WScale( aPixmap );
			ws = 1.0;
		}
		if	( hs > 1.0 ) {
			CPictureFilterScale::HScale( aPixmap );
			hs = 1.0;
		}
	}
	else {
		if	( hs > 1.0 ) {
			CPictureFilterScale::HScale( aPixmap );
			hs = 1.0;
		}
		if	( ws > 1.0 ) {
			CPictureFilterScale::WScale( aPixmap );
			ws = 1.0;
		}
	}

	// Do the smaller change down first
	if	( ws < hs ) {
		if	( hs < 1.0 )	CPictureFilterScale::HScale( aPixmap );
		if	( ws < 1.0 )	CPictureFilterScale::WScale( aPixmap );
	}
	else {
		if	( ws < 1.0 )	CPictureFilterScale::WScale( aPixmap );
		if	( hs < 1.0 )	CPictureFilterScale::HScale( aPixmap );
	}
}

// --------------------------------------------------------------------
// private:	Horizontal (Width) scale up
// --------------------------------------------------------------------
void	CPictureFilterScale::WScale	( CPicturePixmap & aPixmap ) {
	CPicturePixmap	clone = aPixmap;
	aPixmap.Create( itsW, clone.H() );

	CPicturePixel *	srow = new CPicturePixel[clone.W()];
	CPicturePixel *	trow = new CPicturePixel[itsW];

	dword_t		x, y;
	for	( y = 0; y < clone.H(); y++ ) {
		for	( x = 0; x < clone.W(); x++ ) {
			srow[x] = clone.Pixel( x, y );
		}
		if	( itsW > clone.W() ) {
			CPictureFilterScale::ScaleUp( trow, itsW, srow, clone.W() );
		}
		else {
			CPictureFilterScale::ScaleDn( trow, itsW, srow, clone.W() );
		}
		for	( x = 0; x < itsW; x++ ) {
			aPixmap.Pixel( x, y, trow[x] );
		}
	}

	delete []	trow;
	delete []	srow;
}

// --------------------------------------------------------------------
// private:	Vertical (Height) scale up
// --------------------------------------------------------------------
void	CPictureFilterScale::HScale	( CPicturePixmap & aPixmap ) {
	CPicturePixmap	clone = aPixmap;
	aPixmap.Create( clone.W(), itsH );

	CPicturePixel *	srow = new CPicturePixel[clone.H()];
	CPicturePixel *	trow = new CPicturePixel[itsH];

	dword_t		x, y;

	for	( x = 0; x < clone.W(); x++ ) {
		for	( y = 0; y < clone.H(); y++ ) {
			srow[y] = clone.Pixel( x, y );
		}
		if	( itsH > clone.H() ) {
			CPictureFilterScale::ScaleUp( trow, itsH, srow, clone.H() );
		}
		else {
			CPictureFilterScale::ScaleDn( trow, itsH, srow, clone.H() );
		}
		for	( y = 0; y < itsH; y++ ) {
			aPixmap.Pixel( x, y, trow[y] );
		}
	}

	delete []	trow;
	delete []	srow;
}

// --------------------------------------------------------------------
// private:	Scale up
// --------------------------------------------------------------------
void	CPictureFilterScale::ScaleUp	 ( CPicturePixel *	aT,
										   dword_t			aTs,
										   CPicturePixel *	aS,
										   dword_t			aSs ) {
	double	factor	= (double)aSs / (double)aTs;
	dword_t	t, s;
	for	( t = 0; t < aTs; t++ ) {
		s = (dword_t)( (double)t * factor + 0.5 );
		if	( s > aSs ) s = aSs;
		aT[t] = aS[s];
	}
}

// --------------------------------------------------------------------
// private:	Scale Down
// --------------------------------------------------------------------
void	CPictureFilterScale::ScaleDn	 ( CPicturePixel *	aT,
							 			   dword_t			aTs,
										   CPicturePixel *	aS,
										   dword_t			aSs ) {
	double			factor	= (double)aSs / (double)aTs;
	dword_t			s, t, f, l, r, g, b, a;

	for	( t = 0; t < aTs; t++ ) {
		f = (dword_t)( (double)t * factor + 0.5 );
		l = (dword_t)( (double)(t + 1) * factor + 0.5 );

		r = g = b = a = 0;
		for	( s = f; s <= l; s++ ) {
			r += aS[s].R();
			g += aS[s].G();
			b += aS[s].B();
			a += aS[s].A();
		}
		l = (l - f + 1);
		aT[t].R( (byte_t)(r / l) );
		aT[t].G( (byte_t)(g / l) );
		aT[t].B( (byte_t)(b / l) );
		aT[t].A( (byte_t)(a / l) );
	}
}

// --------------------------------------------------------------------
// EOF: CPictureFilterScale.cxx
// --------------------------------------------------------------------
