// --------------------------------------------------------------------
// CPictureSourceT3D.cxx
// Whatis:  UTX Picture manipulation class
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 24-NOV-2001     Created this source
// --------------------------------------------------------------------
#include	"CError.hxx"
#include	"CPictureSourceT3D.hxx"

// --------------------------------------------------------------------
// public:	Constructor
// --------------------------------------------------------------------
CPictureSourceT3D::CPictureSourceT3D(	const char * aT3Dfile ) {
	itsPolyList.T3DParseFile( aT3Dfile );
	itsFactor 	= 1.0;
	itsCamera	= T3D_CAMERA_FRONT;
	itsBG.RGBA( 0x00, 0x00, 0x00, 0x00 );	// Transparent black
	itsFG.RGBA( 0xff, 0xff, 0xff, 0xff );	// Opaque white
}

// --------------------------------------------------------------------
// public:	Interface
// --------------------------------------------------------------------
CPicturePixmap	CPictureSourceT3D::Read	( void ) const {
	CPicturePixmap	mypixmap;

	// Create the pixmap image space
	mypixmap.Create( CPictureSourceT3D::W(), CPictureSourceT3D::H() );

	// Fill it with background color
	dword_t	x, y;
	for	( x = 0; x < mypixmap.W(); x++ ) {
		for	( y = 0; y < mypixmap.H(); y++ ) {
			mypixmap.Pixel( x, y, itsBG );
		}
	}

	// Get polygon center with current camera orientation
	double	wcp	= CPictureSourceT3D::WcP();
	double	hcp	= CPictureSourceT3D::HcP();

	// Loop through the polygons drawing each of them
	C3DPolygon_lci	ploop;
	for	(	ploop  = itsPolyList.PolyList().begin();
			ploop != itsPolyList.PolyList().end();
			ploop++ ) {
		CPictureSourceT3D::DrawPolygon( mypixmap, wcp, hcp, *ploop );
	}

	return	mypixmap;
}

// --------------------------------------------------------------------
void	CPictureSourceT3D::Write ( const CPicturePixmap & aPixmap ) {
	throw	CError( "Can't write T3D files" );
}

// --------------------------------------------------------------------
// public:	Calculate width of the resulting image
// --------------------------------------------------------------------
dword_t	CPictureSourceT3D::W ( void ) const {
	double			minv	=  1000000.0;
	double			maxv	= -1000000.0;
	double			valu	= 0.0;
	C3DPolygon_lci	ploop;
	C3DVertex_lci	vloop;

	for	(	ploop  = itsPolyList.PolyList().begin();
			ploop != itsPolyList.PolyList().end();
			ploop++ ) {
		for	(	vloop  = (*ploop).VertexList().begin();
				vloop != (*ploop).VertexList().end();
				vloop++ ) {
			switch	( itsCamera ) {

				// Y as width
				case	T3D_CAMERA_LEFT:
				valu = (*vloop).Y();
				break;

				// X as width
				case	T3D_CAMERA_TOP:
				case	T3D_CAMERA_FRONT:
				valu = (*vloop).X();
				break;
			}
			if	( minv > valu )	minv = valu;
			if	( maxv < valu ) maxv = valu;
		}
	}
	return	(dword_t)( itsFactor * ( maxv - minv ) );
}

// --------------------------------------------------------------------
// public:	Calculate height of the resulting image
// --------------------------------------------------------------------
dword_t	CPictureSourceT3D::H ( void ) const {
	double			minv	=  1000000.0;
	double			maxv	= -1000000.0;
	double			valu	= 0.0;
	C3DPolygon_lci	ploop;
	C3DVertex_lci	vloop;

	for	(	ploop  = itsPolyList.PolyList().begin();
			ploop != itsPolyList.PolyList().end();
			ploop++ ) {
		for	(	vloop  = (*ploop).VertexList().begin();
				vloop != (*ploop).VertexList().end();
				vloop++ ) {
			switch	( itsCamera ) {

				// Y as height
				case	T3D_CAMERA_TOP:
				valu = (*vloop).Y();
				break;

				// Z as height
				case	T3D_CAMERA_LEFT:
				case	T3D_CAMERA_FRONT:
				valu = (*vloop).Z();
				break;
			}
			if	( minv > valu )	minv = valu;
			if	( maxv < valu ) maxv = valu;
		}
	}
	return	(dword_t)( itsFactor * ( maxv - minv ) );
}

// --------------------------------------------------------------------
// private:	Calculate horizontal offset position
// --------------------------------------------------------------------
double	CPictureSourceT3D::WcP ( void ) const {
	double			minv	= 1000000.0;
	double			valu	= 0.0;
	C3DPolygon_lci	ploop;
	C3DVertex_lci	vloop;

	for	(	ploop  = itsPolyList.PolyList().begin();
			ploop != itsPolyList.PolyList().end();
			ploop++ ) {
		for	(	vloop  = (*ploop).VertexList().begin();
				vloop != (*ploop).VertexList().end();
				vloop++ ) {
			switch	( itsCamera ) {

				// Y as width
				case	T3D_CAMERA_LEFT:
				valu = (*vloop).Y();
				break;

				// X as width
				case	T3D_CAMERA_TOP:
				case	T3D_CAMERA_FRONT:
				valu = (*vloop).X();
				break;
			}
			if	( minv > valu )	minv = valu;
		}
	}

	return	-(minv * itsFactor);
}

// --------------------------------------------------------------------
// private:	Calculate vertical center position
// --------------------------------------------------------------------
double	CPictureSourceT3D::HcP ( void ) const {
	double			minv	=  1000000.0;
	double			valu	= 0.0;
	C3DPolygon_lci	ploop;
	C3DVertex_lci	vloop;

	for	(	ploop  = itsPolyList.PolyList().begin();
			ploop != itsPolyList.PolyList().end();
			ploop++ ) {
		for	(	vloop  = (*ploop).VertexList().begin();
				vloop != (*ploop).VertexList().end();
				vloop++ ) {
			switch	( itsCamera ) {

				// Y as height
				case	T3D_CAMERA_TOP:
				valu = (*vloop).Y();
				break;

				// Z as height
				case	T3D_CAMERA_LEFT:
				case	T3D_CAMERA_FRONT:
				valu = (*vloop).Z();
				break;
			}
			if	( minv > valu )	minv = valu;
		}
	}

	return	-(minv * itsFactor);
}

// --------------------------------------------------------------------
// private:	Draw a polygon into a pixmap
// --------------------------------------------------------------------
void	CPictureSourceT3D::DrawPolygon(	CPicturePixmap &	aPixmap,
										double				aWc,
										double				aHc,
										const C3DPolygon &	aPolygon ) const {
	C3DVertex_lci	vloop;
	dword_t			xf, xp, x;
	dword_t			yf, yp, y;
	dword_t			w	= aPixmap.W();
	dword_t			h	= aPixmap.H();

	// Some initializations to make compiler happy
	xf = xp = x = yf = yp = y = 0;

	// Remember the first vertex coordinates
	vloop = aPolygon.VertexList().begin();
	CPictureSourceT3D::CalcCoords( xf, yf, aWc, aHc, *vloop );

	if	( xf >= w ) xf = w - 1;
	if	( yf >= h ) yf = h - 1;

	// We assume in cold blood that we draw it !
	x = xf;
	y = yf;

	// Start from the second vertex
	vloop++;
	while	( vloop != aPolygon.VertexList().end() ) {
		// Remember previous position
		xp = x;
		yp = y;

		// Calculate a new position
		CPictureSourceT3D::CalcCoords( x, y, aWc, aHc, *vloop );
		if	( x >= w ) x = w - 1;
		if	( y >= h ) y = h - 1;

		// Draw line into that position
		aPixmap.Line( w - x - 1,  h - y - 1,
					  w - xp - 1, h - yp - 1,
					  itsFG );

		// Advance to next vertex
		vloop++;
	}

	// Draw line between first and last vertexes
	aPixmap.Line( w - x - 1,  h - y - 1,
				  w - xf - 1, h - yf - 1,
				  itsFG );
}

// --------------------------------------------------------------------
// private:	Calculate coordinates
// --------------------------------------------------------------------
void	CPictureSourceT3D::CalcCoords(	dword_t &			aX,
										dword_t &			aY,
										double				aWc,
										double				aHc,
										const C3DVertex &	aVertex ) const {
	double	x, y;

	x = y = 0.0;
	switch	( itsCamera ) {
		// Front X = X, Y = Z
		case	T3D_CAMERA_FRONT:
		x = aVertex.X();
		y = aVertex.Z();
		break;

		// Top X = X, Y = Y
		case	T3D_CAMERA_TOP:
		x = aVertex.X();
		y = aVertex.Y();
		break;

		// Left X = Y, Y = Z
		case	T3D_CAMERA_LEFT:
		x = aVertex.Y();
		y = aVertex.Z();
		break;

	}

	aX = (dword_t)(x * itsFactor + aWc);
	aY = (dword_t)(y * itsFactor + aHc);
}

// --------------------------------------------------------------------
// EOF: CPictureSourceT3D.cxx
// --------------------------------------------------------------------
