// --------------------------------------------------------------------
// CTcpSocket.hxx
// Whatis:  The socket class
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 01-FEB-2003     Created this source
// --------------------------------------------------------------------
#include    "CError.hxx"
#include    "CTcpSocket.hxx"

// ----------------------------------------------------------------
// The windows platform needs to know if sockets are needed anymore
// or if this was the first time in.
// ----------------------------------------------------------------
#ifdef WIN32
static  int         statWinSockCount   = 0;
#endif

// ----------------------------------------------------------------
// public:      Initializing an entirely new socket
// ----------------------------------------------------------------
CTcpSocket::CTcpSocket( socket_type_t aType ) {
#ifndef WIN32
    errno = 0;
#endif

    itsSocket   = SOCKET_NONE;
    itsState    = SOCKET_STATE_IDLE;
#ifdef  WIN32
    itsWsaFound = 0;
#endif
    itsHostName     = NULL;
    itsHostAddr     = NULL;
    itsPeerName     = NULL;
    itsPeerAddr     = NULL;
    itsPeerPort     = 0;
    itsHostPort     = 0;
    itsHostService  = NULL;
	itsWriteOffs	= 0;
	itsReadSize		= 0;
	itsReadOffs		= 0;
	itsConnect      = false;
    itsType			= aType;

    // Some windows stuff
#ifdef  WIN32
    EnterWinSock();
#endif

    // Create the socket. The windows world could have set the error
    itsSocket = socket( AF_INET, (int)aType, 0 );
    CTcpSocket::ChkError( 0 );
    SetHost();

}

// ----------------------------------------------------------------
// public:      Initialize for an existing socket (accepted socket)
// ----------------------------------------------------------------
CTcpSocket::CTcpSocket( SOCKET_t aSocket ) {
#ifndef WIN32
    errno = 0;
#endif

    itsSocket   = SOCKET_NONE;
    itsState    = SOCKET_STATE_IDLE;
#ifdef  WIN32
    itsWsaFound = 0;
#endif
    itsHostName     = NULL;
    itsHostAddr     = NULL;
    itsPeerName     = NULL;
    itsPeerAddr     = NULL;
    itsPeerPort     = 0;
    itsHostPort     = 0;
    itsHostService  = NULL;
	itsWriteOffs	= 0;
	itsReadSize		= 0;
	itsReadOffs		= 0;
	itsConnect      = true;
    itsType			= socket_type_stream;

    // Some windows stuff
#ifdef  WIN32
    EnterWinSock();
#endif

    // Create the socket. The windows world could have set the error
    itsSocket = aSocket;
    SetHost();
    SetPeer();
}

// ----------------------------------------------------------------
// public:      Dispose the socket
// ----------------------------------------------------------------
CTcpSocket::~CTcpSocket() {           // Does a brute force close
#ifndef WIN32
    errno = 0;
#endif

    if  ( itsSocket != SOCKET_NONE ) {
        shutdown( itsSocket, 2 );   // Brute force close
#ifdef  WIN32
        closesocket( itsSocket );
#else
        close( itsSocket );
#endif
    }

    // Release the DLL in case we don't need it anymore (in windows world)
#ifdef  WIN32
    ExitWinSock();
#endif

    // Dispose some data
    if  ( itsHostName )     delete [] itsHostName;
    if  ( itsHostAddr )     delete [] itsHostAddr;
    if  ( itsPeerName )     delete [] itsPeerName;
    if  ( itsPeerAddr )     delete [] itsPeerAddr;
    if  ( itsHostService )  delete [] itsHostService;
}

// ----------------------------------------------------------------
// public:      Bind automatically
// ----------------------------------------------------------------
void    CTcpSocket::Bind       ( void ) {
    Bind( (unsigned short)0 );
}

// ----------------------------------------------------------------
// public:      Bind to a service
// ----------------------------------------------------------------
void    CTcpSocket::Bind       ( const char * aName ) {
#ifndef WIN32
    errno = 0;
#endif
    struct servent * serv = getservbyname( aName, NULL );
    if  ( serv )    Bind( ntohs( (unsigned short)(serv->s_port) ) );
    else            CTcpSocket::ChkError( 0 );
}

// ----------------------------------------------------------------
void    CTcpSocket::Bind       ( unsigned short aPort, CTcpSocket * aSock ) {
#ifndef WIN32
    errno = 0;
#endif
	int					retry, bres;
    struct servent *    serv;

    // ------------------------------------------------------------
	// Bind socket using some retries
	// Also, turn on the address reuse in case the socket
	// whines about this matter
    // ------------------------------------------------------------
	for	( retry = 0; retry < 10; retry++ ) {
        struct  sockaddr_in	sin	= { 0 };

        // --------------------------------------------------------
		// If we have a socket, we can fill in something beforehand
        // --------------------------------------------------------
        if	( aSock ) {
#ifdef	WIN32
			int			sln = sizeof( sin );
#else
			socklen_t	sln = sizeof( sin );
#endif
			::getsockname( aSock->Socket(), (struct sockaddr *)(&sin), &sln );
		}
        else {
		    sin.sin_family		= AF_INET;
		    sin.sin_addr.s_addr = htonl(INADDR_ANY);
        }
	    sin.sin_port = htons( aPort );

        // --------------------------------------------------------
        // Now, do the bind
        // --------------------------------------------------------
        bres = bind( itsSocket, (struct sockaddr *)(&sin), sizeof( sin ) );

        // --------------------------------------------------------
        // If the bind was ok, we can break this loop
        // --------------------------------------------------------
        if  ( ! bres )  break;

        // --------------------------------------------------------
        // Otherwice, check out the error code
        // --------------------------------------------------------
#ifdef  WIN32
        if  ( WSAGetLastError() != WSAEADDRINUSE ) {
#else
		if	( errno != EADDRINUSE ) {
#endif
            CTcpSocket::ChkError( 0 );
            break;
        }

        // --------------------------------------------------------
        // Trying to reuse address, set sock option
        // --------------------------------------------------------
        bres = 1;
    	setsockopt(	itsSocket,
					SOL_SOCKET,
					SO_REUSEADDR,
#ifdef WIN32
                    (char *)&bres,
#else
                    &bres,
#endif
	    			sizeof( int ) );

        ::SLEEP( 1000 );
	}

    // ------------------------------------------------------------
    // Now, if the result was allright, save the port and get possible
    // service name for that port
    // ------------------------------------------------------------
    SetHost();
    serv = getservbyport( htons( itsHostPort ), NULL );
    if  ( serv ) {
        itsHostService = new char [strlen( serv->s_name ) + 1];
        (void)strcpy( itsHostService, serv->s_name );
    }
}

// ----------------------------------------------------------------
// public:      Set listen mode
// ----------------------------------------------------------------
void    CTcpSocket::Listen     ( void ) {
#ifndef WIN32
    errno = 0;
#endif
    if  ( listen( itsSocket, 5 ) )  CTcpSocket::ChkError( 0 );
    SetHost();
}

// ----------------------------------------------------------------
// public:      Accept connections
// The method returns NULL in case of an error or if there was no
// connection waiting.
// Either case, the function returns immediately
// ----------------------------------------------------------------
CTcpSocket *	CTcpSocket::Accept     ( void ) {
CTcpSocket *	result  = NULL;
#ifndef WIN32
    errno = 0;
#endif
    int             acpt;
    fd_set          rtmp;
	struct timeval  wait;

    // ------------------------------------------------------------
    // Because the accept is blocking by it's nature, we go around
    // this problem by first checking if somebody is ringing
    // the doorbell (polling)
    // ------------------------------------------------------------
	wait.tv_sec     = 0;
	wait.tv_usec    = 0;

    FD_ZERO( &rtmp );
	FD_SET( itsSocket, &rtmp );

    acpt = select( FD_SETSIZE, &rtmp, NULL, NULL, &wait );
    if      ( acpt > 0 ) {
        if  ( FD_ISSET( itsSocket, &rtmp ) )    acpt = 1;
        else                                    acpt = 0;
    }
    else if ( acpt < 0 ) {
        CTcpSocket::ChkError( 0 );
    }

    // ------------------------------------------------------------
    // Create a new socket from the listening one
    // ------------------------------------------------------------
    if  ( acpt > 0 ) {
        SOCKET_t    sock = accept( itsSocket, NULL, NULL );
        if  ( sock == SOCKET_NONE ) CTcpSocket::ChkError( 0 );
        else                        result = new CTcpSocket( sock );
    }
	return  result;
}

// ----------------------------------------------------------------
// public:      Connect to a server
// ----------------------------------------------------------------
bool        CTcpSocket::Connect(   const char *    aHost,
                                   unsigned short  aPort ) {
	bool                result = false;
#ifndef WIN32
    errno = 0;
#endif

    int                 acpt;
    fd_set              rtmp;
	struct timeval      wait;
    int                 tcpresult;
    struct  sockaddr_in sin;
    struct hostent *    hp;
    socklen_t           size;
    int                 ores = 0;

#ifdef WIN32
	unsigned long		blocking_non = 1;
#endif

    if  ( itsSocket == SOCKET_NONE )    result = false;

    else switch ( itsState ) {
        case    SOCKET_STATE_IDLE:  // Initial state, make the socket non-blocking
                                    // and begin connect process
#ifdef WIN32
        (void)ioctlsocket( itsSocket, FIONBIO, &blocking_non );
#else
        ::fcntl( itsSocket, F_SETFL, O_NONBLOCK );
#endif

        // --------------------------------------------------------
		// We assume that the caller uses dot-notation
        // --------------------------------------------------------
        hp = gethostbyname( aHost );
        if  ( ! hp ) {
            itsState    = SOCKET_STATE_ERROR;
            CTcpSocket::ChkError( 0 );
            break;
        }

        // --------------------------------------------------------
		// Make the initial connection here
        // --------------------------------------------------------
        memset( &sin, 0, sizeof( sin ) );
        sin.sin_family  = AF_INET;
        sin.sin_port    = htons( aPort );
        memcpy( &(sin.sin_addr), hp->h_addr, hp->h_length );

        // --------------------------------------------------------
		// The connection succeded, put the socket back to blocking mode
        // --------------------------------------------------------
        if  ( ! connect( itsSocket, (struct sockaddr *)&sin, sizeof( sin ) ) ) {
#ifdef WIN32
			blocking_non = 0;
			(void)ioctlsocket( itsSocket, FIONBIO, &blocking_non );
#else
			fcntl( itsSocket, F_SETFL, 0 );
#endif
            itsState = SOCKET_STATE_CONNECT;
            SetPeer();
            SetHost();
            break;
		}

        // --------------------------------------------------------
		// Then, check if the connection is under progress
        // --------------------------------------------------------
#ifdef WIN32
		if  ( WSAGetLastError() == WSAEWOULDBLOCK ) {
#else
		if  ( errno == EINPROGRESS ) {
#endif
            itsState = SOCKET_STATE_PROGRESS;
            break;
        }

        // --------------------------------------------------------
		// The connect has failed
        // --------------------------------------------------------
        itsState = SOCKET_STATE_ERROR;
        CTcpSocket::ChkError( 0 );
        break;

        // --------------------------------------------------------
		// Connection was under progress at previous call.
		// Let's continue the good work......
        // --------------------------------------------------------
        case    SOCKET_STATE_PROGRESS:
        size = sizeof( int );
        tcpresult = ::getsockopt( itsSocket, SOL_SOCKET, SO_ERROR, (char *)&ores, &size );

        // --------------------------------------------------------
        // Still working on it ?
        // --------------------------------------------------------
#ifdef WIN32
		if  ( tcpresult == WSAEWOULDBLOCK ) {
#else
		if  ( tcpresult == EINPROGRESS ) {
#endif
            itsState = SOCKET_STATE_PROGRESS;
            break;
        }

        // --------------------------------------------------------
        // Some other error ?
        // --------------------------------------------------------
        if  ( result ) {
            itsState = SOCKET_STATE_ERROR;
            CTcpSocket::ChkError( 0 );
            break;
        }

        // --------------------------------------------------------
        // Still working on it ?
        // --------------------------------------------------------
#ifdef WIN32
		if  ( ores == WSAEWOULDBLOCK ) {
#else
		if  ( ores == EINPROGRESS ) {
#endif
            itsState = SOCKET_STATE_PROGRESS;
            break;
        }

        // --------------------------------------------------------
        // Some other error ?
        // --------------------------------------------------------
        if  ( ores ) {
            itsState = SOCKET_STATE_ERROR;
            CTcpSocket::ChkError( ores );
            break;
        }

        // --------------------------------------------------------
        // Can the socket be written into (then it is connected)
        // --------------------------------------------------------
    	wait.tv_sec     = 0;
	    wait.tv_usec    = 0;

        FD_ZERO( &rtmp );
	    FD_SET( itsSocket, &rtmp );

        acpt = select( FD_SETSIZE, NULL, &rtmp, NULL, &wait );
        if  ( acpt < 0 ) {
            itsState = SOCKET_STATE_ERROR;
            CTcpSocket::ChkError( 0 );
            break;
        }
        if  ( acpt == 0 )                           break;
        if  ( FD_ISSET( itsSocket, &rtmp ) == 0 )   break;

        // --------------------------------------------------------
        // Seems like the connection is ready
        // --------------------------------------------------------
#ifdef WIN32
		blocking_non = 0;
		(void)ioctlsocket( itsSocket, FIONBIO, &blocking_non );
#else
		fcntl( itsSocket, F_SETFL, 0 );
#endif
        itsState = SOCKET_STATE_CONNECT;
        SetPeer();
        if	( ! itsPeerName ) {
            itsPeerName = new char[ strlen( aHost ) + 1 ];
            (void)strcpy( itsPeerName, aHost );
        }
        SetHost();
        break;

        // --------------------------------------------------------
        // The connection is working or it is in error
        // --------------------------------------------------------
        case    SOCKET_STATE_CONNECT:
        case    SOCKET_STATE_ERROR:
        break;
    }

    switch  ( itsState ) {
        case    SOCKET_STATE_PROGRESS:
        case    SOCKET_STATE_IDLE:      result = false;     break;
        case    SOCKET_STATE_CONNECT:   result = true;      break;
        case    SOCKET_STATE_ERROR:     throw CError( "Connection failed" );
    }
    return  result;
}

// ----------------------------------------------------------------
// public:      Read a byte from the socket
// ----------------------------------------------------------------
int     CTcpSocket::Read       ( void ) {
    int             result;

#ifndef WIN32
    errno = 0;
#endif

    // ------------------------------------------------------------
	// There migh be some data left from previous reads ...
    // ------------------------------------------------------------
	if	( itsReadOffs < itsReadSize ) {
		result = itsReadBuff[itsReadOffs++];
		result &= 0xff;
	}	

    // ------------------------------------------------------------
    // Because read is blocking by it's nature we have to use
    // the select call to ensure there's at least one byte in there
    // ------------------------------------------------------------
	else {
	    fd_set          rtmp;
		struct timeval  wait;
		int				rl;


		itsReadOffs		= 0;
		itsReadSize		= 0;
		wait.tv_sec     = 0;
		wait.tv_usec    = 100;

		FD_ZERO( &rtmp );
		FD_SET( itsSocket, &rtmp );

		result = select( FD_SETSIZE, &rtmp, NULL, NULL, &wait );
		if      ( result > 0 ) {
			if  ( FD_ISSET( itsSocket, &rtmp ) ) {
#ifdef	WIN32
	            rl = recv( itsSocket, itsReadBuff, sizeof( itsReadBuff ), 0 );
#else
		        rl = recv( itsSocket, (unsigned char *)itsReadBuff, sizeof( itsReadBuff ), 0 );
#endif

				// ----------------------------------------------------
				// The rl will be zero in case the connection was closed
				// ----------------------------------------------------
				if      ( rl == 0 ) {
					result = SOCKET_READ_DISCONNECT;
				}

				// ----------------------------------------------------
				// Return value of > 1 indicates a good read
				// The pretty ugly looking typecast is because the
				// char would expand to negative integers ....
				// ----------------------------------------------------
				else if ( rl >= 1 ) {
					itsReadSize = rl;
					result = itsReadBuff[itsReadOffs++];
					result &= 0xff;
				}

				// ----------------------------------------------------
				// Other values are errors
				// ----------------------------------------------------
				else {
					CTcpSocket::ChkError( 0 );
					result = SOCKET_READ_ERROR;
				}
			}

			// --------------------------------------------------------
			// Nothing in the read buffer
			// --------------------------------------------------------
			else {
				result = SOCKET_READ_IDLE;
			}
		}

		// ------------------------------------------------------------
		// Error occurred while polling
		// ------------------------------------------------------------
		else if ( result < 0 ) {
			CTcpSocket::ChkError( 0 );
			result = SOCKET_READ_ERROR;
		}

		// ------------------------------------------------------------
		// Nothing was set
		// ------------------------------------------------------------
		else {
			result = SOCKET_READ_IDLE;
		}
	}
	return  result;
}

// ----------------------------------------------------------------
// public:      Write long integer to the socket (asciiz)
// ----------------------------------------------------------------
void    CTcpSocket::Write      (   long         aLong ) {
    char    lbf[64];
    sprintf( lbf, "%li", aLong );
    Write( lbf );
}

// ----------------------------------------------------------------
// public:      Write one character to the socket
// ----------------------------------------------------------------
void    CTcpSocket::Write      (   char            aChr ) {
    Write( &aChr, 1 );
}

// ----------------------------------------------------------------
// public:      Write asciiz string to the socket
// ----------------------------------------------------------------
void    CTcpSocket::Write      (   const char *    aStr ) {
    Write( aStr, (int)strlen( aStr ) );
}

// ----------------------------------------------------------------
// public:      Write data to the socket
// ----------------------------------------------------------------
void    CTcpSocket::Write      (	const char *    aData,
									int             aDlen ) {
#ifndef WIN32
    errno = 0;
#endif
	while   ( aDlen > 0 ) {
		itsWriteBuff[itsWriteOffs++] = *(aData++);
		aDlen--;
		if	( itsWriteOffs >= (int)(sizeof( itsWriteBuff )) ) {
			Flush();
		}
	}
}

// ----------------------------------------------------------------
// public:      Flush contents on write buffer into line
// ----------------------------------------------------------------
void    CTcpSocket::Flush      ( void ) {
#ifndef WIN32
    errno = 0;
#endif
	char *	p = itsWriteBuff;
	int		l = itsWriteOffs;
	while	( l > 0 ) {
		int written = send( itsSocket, p, l, 0 );
        if  ( written < 0 ) {
			CTcpSocket::ChkError( written );
            break;
        }
        p = p + written;
        l = l - written;
    }
	itsWriteOffs = 0;
}

// ----------------------------------------------------------------
// Set host variables. Hostname, address and possible port
// ----------------------------------------------------------------
void    CTcpSocket::SetHost    ( void ) {
    char              	hn[150];
    struct hostent *    hp;
    struct  sockaddr_in sin;
    socklen_t           sln = sizeof( sin );

    // ------------------------------------------------------------
	// Do not do this if the port and names are already there
    // ------------------------------------------------------------
    if	( ( itsHostPort ) && ( itsHostName ) && ( itsHostAddr ) ) {
    	return;
    }

    // ------------------------------------------------------------
    // Dispose current settings
    // ------------------------------------------------------------
    if  ( itsHostName ) delete [] itsHostName;
    if  ( itsHostAddr ) delete [] itsHostAddr;
    itsHostPort = 0;

    // ------------------------------------------------------------
    // The host name
    // ------------------------------------------------------------
    if  ( ::gethostname( hn, sizeof(hn)-1 ) ) {
        CTcpSocket::ChkError( 0 );
        return;
    }

    // ------------------------------------------------------------
    // The host address
    // ------------------------------------------------------------
    hp = ::gethostbyname( hn );
    if  ( ! hp ) {
        CTcpSocket::ChkError( 0 );
        return;
    }

    // ------------------------------------------------------------
    // Set up the host name, address and port
    // ------------------------------------------------------------
    itsHostName = new char [strlen( hp->h_name ) + 1];
    strcpy( itsHostName, hp->h_name );

    itsHostAddr = new char [strlen( inet_ntoa(*((struct in_addr *)hp->h_addr)) ) + 1];
    strcpy( itsHostAddr, inet_ntoa(*((struct in_addr *)hp->h_addr)) );

    // ------------------------------------------------------------
    // The host port
    // ------------------------------------------------------------
    if  ( ! getsockname( itsSocket, (struct sockaddr *)(&sin), &sln ) ) {
        itsHostPort = ntohs( sin.sin_port );
    }
}

// ----------------------------------------------------------------
// Set peer variables. Peername and address
// ----------------------------------------------------------------
void    CTcpSocket::SetPeer    ( void ) {
    struct  hostent *   hp;
    struct  sockaddr_in sin;
    socklen_t           sln = sizeof( sin );

    // ------------------------------------------------------------
    // Dispose current values
    // ------------------------------------------------------------
    if  ( itsPeerName ) delete [] itsPeerName;
    if  ( itsPeerAddr ) delete [] itsPeerAddr;
    itsPeerPort = 0;

    // ------------------------------------------------------------
    // This should go allright
    // ------------------------------------------------------------
    if  ( getpeername( itsSocket, (struct sockaddr *)&sin, &sln ) ) {
        CTcpSocket::ChkError( 0 );
        return;
    }

    // ------------------------------------------------------------
	// Sometimes we don't get the peer name - this is actually not an error
    // ------------------------------------------------------------
    hp = gethostbyaddr( (char *)(&sin.sin_addr), sizeof( sin.sin_addr ), AF_INET );
    if  ( hp ) {
	    itsPeerName = new char [strlen( hp->h_name ) + 1];
	    (void)strcpy( itsPeerName, hp->h_name );
    }

    // ------------------------------------------------------------
    // Raw ip address
    // ------------------------------------------------------------
    itsPeerAddr = new char [strlen( inet_ntoa( sin.sin_addr ) ) + 1];
    (void)strcpy( itsPeerAddr, inet_ntoa( sin.sin_addr ) );

    // ------------------------------------------------------------
    // Port number
    // ------------------------------------------------------------
    itsPeerPort = ntohs( sin.sin_port );
}

// ----------------------------------------------------------------
// private:     Check socket layer errors
// ----------------------------------------------------------------
void        CTcpSocket::ChkError   ( int   aError ) {
    int     errv;

#ifdef  WIN32
    errv = WSAGetLastError();
#else
    errv = errno;
#endif
    if  ( ! errv ) errv = aError;
#ifdef  WIN32
    switch  ( errv ) {
        case    0:                  break;
        case    WSAEINTR:           throw CError( "Interrupted system call." );
        case    WSAEBADF:           throw CError( "Bad file number." );
        case    WSAEFAULT:          throw CError( "Bad address." );
        case    WSAEINVAL:          throw CError( "Invalid argument." );
        case    WSAEMFILE:          throw CError( "Too many open files." );
        case    WSAEWOULDBLOCK:     throw CError( "Operation would block." );
        case    WSAEINPROGRESS:     throw CError( "Operation now in progress." );
        case    WSAEALREADY:        throw CError( "Operation already in progress." );
        case    WSAENOTSOCK:        throw CError( "Socket operation on nonsocket." );
        case    WSAEDESTADDRREQ:    throw CError( "Destination address required." );
        case    WSAEMSGSIZE:        throw CError( "Message too long." );
        case    WSAEPROTOTYPE:      throw CError( "Protocol wrong type for socket." );
        case    WSAENOPROTOOPT:     throw CError( "Protocol not available." );
        case    WSAEPROTONOSUPPORT: throw CError( "Protocol not supported." );
        case    WSAESOCKTNOSUPPORT: throw CError( "Socket type not supported." );
        case    WSAEOPNOTSUPP:      throw CError( "Operation not supported on socket." );
        case    WSAEPFNOSUPPORT:    throw CError( "Protocol family not supported." );
        case    WSAEAFNOSUPPORT:    throw CError( "Address family not supported by protocol family." );
        case    WSAEADDRINUSE:      throw CError( "Address already in use." );
        case    WSAEADDRNOTAVAIL:   throw CError( "Cannot assign requested address." );
        case    WSAENETDOWN:        throw CError( "Network is down." );
        case    WSAENETUNREACH:     throw CError( "Network is unreachable." );
        case    WSAENETRESET:       throw CError( "Network dropped connection on reset." );
        case    WSAECONNABORTED:    throw CError( "Software caused connection abort." );
        case    WSAECONNRESET:      throw CError( "Connection reset by peer." );
        case    WSAENOBUFS:         throw CError( "No buffer space available." );
        case    WSAEISCONN:         throw CError( "Socket is already connected." );
        case    WSAENOTCONN:        throw CError( "Socket is not connected." );
        case    WSAESHUTDOWN:       throw CError( "Cannot send after socket shutdown." );
        case    WSAETOOMANYREFS:    throw CError( "Too many references: cannot splice." );
        case    WSAETIMEDOUT:       throw CError( "Connection timed out." );
        case    WSAECONNREFUSED:    throw CError( "Connection refused." );
        case    WSAELOOP:           throw CError( "Too many levels of symbolic links." );
        case    WSAENAMETOOLONG:    throw CError( "File name too long." );
        case    WSAEHOSTDOWN:       throw CError( "Host is down." );
        case    WSAEHOSTUNREACH:    throw CError( "No route to host." );
        case    WSASYSNOTREADY:     throw CError( "Network subsystem is unusable." );
        case    WSAVERNOTSUPPORTED: throw CError( "DLL cannot support this application." );
        case    WSANOTINITIALISED:  throw CError( "Winsock not initialized." );
        case    WSAEDISCON:         throw CError( "Disconnect." );
        case    WSAHOST_NOT_FOUND:  throw CError( "Host not found." );
        case    WSATRY_AGAIN:       throw CError( "Nonauthoritative host not found." );
        case    WSANO_RECOVERY:     throw CError( "Nonrecoverable error." );
        case    WSANO_DATA:         throw CError( "Valid name, no data record." );
        default:        			throw CError( "Unknown error." );
    }
#else
    switch  ( errv ) {
        case    0:                  break;
        case    EACCES:             throw CError( "No permission for that address" );
        case    EADDRINUSE:         throw CError( "Address already in use" );
        case    EADDRNOTAVAIL:      throw CError( "Address not available" );
        case    EAFNOSUPPORT:       throw CError( "Address family not supported" );
        case    EBADF:              throw CError( "Bad function parameter" );
        case    EFAULT:             throw CError( "Invalid argument on function call" );
        case    EINVAL:             throw CError( "Invalid arguments" );
        case    EMFILE:             throw CError( "The descriptor table is full" );
        case    ENOBUFS:            throw CError( "Insufficient resources available" );
        case    ENOTSOCK:           throw CError( "Invalid socket" );
        case    ENOMEM:             throw CError( "Out of memory" );
        case    EPERM:              throw CError( "Permission denied" );
        case    EPROTONOSUPPORT:    throw CError( "Protocol not supported" );
        case    EINPROGRESS:        throw CError( "Operation now in progress" );
        case    ECONNREFUSED:       throw CError( "Connection refused" );

        default:
        throw CError( "Unknown error." );
    }
#endif
}

// ----------------------------------------------------------------
// Windows only: Entering the sockets. Load DLL as needed
// ----------------------------------------------------------------
#ifdef  WIN32
void    CTcpSocket::EnterWinSock( void ) {
    if  ( statWinSockCount == 0 ) {
        WORD    wVersionRequested;
        WSADATA wsaData;
        int     err;

        wVersionRequested = 2;
        // MAKEWORD( 2, 0 );
        if  ( ( err = WSAStartup( wVersionRequested, &wsaData ) ) != 0 ) {
            switch ( err ) {
                case WSASYSNOTREADY:        throw CError( "Not ready for network communication" );
                case WSAVERNOTSUPPORTED:    throw CError( "Incompatible socket version" );
                default:                    throw CError( "Can't find a usable WinSock DLL" );
            }
        }

        // Confirm that the WinSock DLL supports 2.0.
        // Note that if the DLL supports versions greater
        // than 2.0 in addition to 2.0, it will still return
        // 2.0 in wVersion since that is the version we
        // requested.
        else if ( ( LOBYTE( wsaData.wVersion ) != 2 ) ||
                  ( HIBYTE( wsaData.wVersion ) != 0 ) ) {
            // Tell the user that we couldn't find a usable
            // WinSock DLL.
            throw CError( "Can't find a usable WinSock DLL (version 2.0)" );
        }

        else {
            statWinSockCount++;
            itsWsaFound = 1;
        }
    }
}
#endif

// ----------------------------------------------------------------
// Windows only: Exiting the sockets. Unload DLL as needed
// ----------------------------------------------------------------
#ifdef  WIN32
void    CTcpSocket::ExitWinSock( void ) {
    if  ( itsWsaFound ) {
        statWinSockCount--;
        if  ( statWinSockCount <= 0 ) {
            WSACleanup();
            statWinSockCount = 0;
        }
    }
}
#endif

// ----------------------------------------------------------------
void            CTcpSocket::Write       ( short aS )            { Write( (long)aS ); }
void            CTcpSocket::Write       ( dword_t aUs )  		{ Write( (long)aUs ); }
void            CTcpSocket::Write       ( unsigned short aUs )  { Write( (long)aUs ); }
void            CTcpSocket::Write       ( unsigned long aUl )   { Write( (long)aUl ); }
const char *    CTcpSocket::HostName    ( void )    { return (const char *)( itsHostName    ? itsHostName    : "" ); }
const char *    CTcpSocket::HostAddr    ( void )    { return (const char *)( itsHostAddr    ? itsHostAddr    : "" ); }
const char *    CTcpSocket::PeerName    ( void )    { return (const char *)( itsPeerName    ? itsPeerName    : "" ); }
const char *    CTcpSocket::PeerAddr    ( void )    { return (const char *)( itsPeerAddr    ? itsPeerAddr    : "" ); }
unsigned short  CTcpSocket::HostPort    ( void )    { return itsHostPort; }
unsigned short  CTcpSocket::PeerPort    ( void )    { return itsPeerPort; }
const char *    CTcpSocket::HostService ( void )    { return (const char *)( itsHostService ? itsHostService : "" ); }
SOCKET_t		CTcpSocket::Socket		( void )	{ return itsSocket; }
socket_type_t	CTcpSocket::Type		( void )	{ return itsType; }

// ---------------------------------------------------------------- //
//                              E O F                               //
// ---------------------------------------------------------------- //
