// --------------------------------------------------------------------
// HTTP.cxx
// Whatis:  HTTP utility functions
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 24-NOV-2001     Created this source
// --------------------------------------------------------------------
#include    "Platform.hxx"

// --------------------------------------------------------------------
// Days and months
// --------------------------------------------------------------------
static const char *    __weekday[] = {
    "Sun", "Mon", "Tue", "Wed", "Thu",
    "Fri", "Sat"
};
static const char *    __month[] = {
    "Jan", "Feb", "Mar", "Apr",
    "May", "Jun", "Jul", "Aug",
    "Sep", "Oct", "Nov", "Dec"
};

// --------------------------------------------------------------------
// Table to output HTML data on character by character basis
// --------------------------------------------------------------------
static  const char *    __html_char_encoding[256] = {
    "\x00", "\x01", "\x02", "\x03", "\x04", "\x05", "\x06", "\x07",	"\x08", "\x09", "<br>", "\x0b", "\x0c", "\x0d", "\x0e", "\x0f",
	"\x10", "\x11", "\x12", "\x13", "\x14", "\x15", "\x16", "\x17",	"\x18", "\x19", "\x1a", "\x1b", "\x1c", "\x1d", "\x1e", "\x1f",
	" ", "!", "&quot;", "#", "$", "%", "&amp;", "'", "(", ")", "*", "+", ",", "-", ".", "/",
	"0", "1", "2", "3", "4", "5", "6", "7", "8", "9", ":", ";", "&lt;", "=", "&gt;", "?",
	"@", "A", "B", "C", "D", "E", "F", "G", "H", "I", "J", "K", "L", "M", "N", "O",
	"P", "Q", "R", "S", "T", "U", "V", "W", "X", "Y", "Z", "[", "\\", "]", "^", "_",
	"`", "a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o",
	"p", "q", "r", "s", "t", "u", "v", "w", "x", "y", "z", "{", "|", "}", "~", " ",
    "&Auml;",   "&Aring;",  "&Ccedil;", "&Eacute;", "&Ntilde;", "&Ouml;",   "&Uuml;",   "&aacute;", "&agrave;", "&acirc;",  "&auml;",   "&atilde;", "&aring;",  "&ccedil;", "&eacute;", "&egrave;",
    "&ecirc;",  "&euml;",   "&iacute;", "&igrave;", "&icirc;",  "&iuml;",   "&ntilde;", "&oacute;", "&ograve;", "&ocirc;",  "&ouml;",   "&otilde;", "&uacute;", "&ugrave;", "&ucirc;",  "&uuml;",
    "&#160;",   "&deg;",    "&cent;",   "&pound;",  "&sect;",   "&#165;",   "&para;",   "&szlig;",  "&reg;",    "&copy;",   "&#170;",   "&acute;",  "&uml;",    "&#173;",   "&AElig;",  "&Oslash;",
    "&#176;",   "&plusmn;", "&#178;",   "&#179;",   "&yen;",    "&micro;",  "&#182;",   "&#183;",   "&#184;",   "&#185;",   "&#186;",   "&ordf;",   "&ordm;",   "&#189;",   "&aelig;",  "&oslash;",
    "&iquest;", "&iexcl;",  "&not;",    "&#195;",   "&#196;",   "&#197;",   "&#198;",   "&laquo;",  "&raquo;",  "&#201;",   "&nbsp;",   "&Agrave;", "&Atilde;", "&Otilde;", "&#206;",   "&#207;",
    "&#208;",   "&#209;",   "&#210;",   "&#211;",   "&#212;",   "&#213;",   "&divide;", "&#215;",   "&yuml;",   "&#217;",   "&#218;",   "&curren;", "&#220;",   "&#221;",   "&#222;",   "&#223;",
    "&#224;",   "&#225;",   "&#226;",   "&#227;",   "&#228;",   "&Acirc;",  "&Ecirc;",  "&Aacute;", "&Euml;",   "&Egrave;", "&Iacute;", "&Icirc;",  "&Iuml;",   "&Igrave;", "&Oacute;", "&Ocirc;",
    "&#240;",   "&Ograve;", "&Uacute;", "&Ucirc;",  "&Ugrave;", "&#245;",   "&#246;",   "&#247;",   "&#248;",   "&#249;",   "&#250;",   "&#251;",   "&#252;",  "&#253;",   "&#254;",   "&#255;"
};

static  const char *  __encode_hex_nibs = "0123456789ABCDEF";

// --------------------------------------------------------------------
// local:   Time output in MIME compliant form
// --------------------------------------------------------------------
static  void    __mime_time ( time_t aTime ) {
    struct tm * tms = ::gmtime( &aTime );
    int         year= (tms->tm_year) % 100;

    ::printf( "%s, ", __weekday[tms->tm_wday] );
    ::printf( "%d ",  tms->tm_mday );
    ::printf( "%s ",  __month[tms->tm_mon] );
    ::printf( "%s%d ", year < 10 ? "0" : "", year );
    ::printf( "%s%d:", tms->tm_hour < 10 ? "0" : "", tms->tm_hour );
    ::printf( "%s%d:", tms->tm_min  < 10 ? "0" : "", tms->tm_min );
    ::printf( "%s%d ", tms->tm_sec  < 10 ? "0" : "", tms->tm_sec );
#ifdef	WIN32
    ::printf( "GMT\n" );
#else
    ::printf( "%s\n", tms->tm_zone );
#endif
}

// --------------------------------------------------------------------
// local:   Time output in cookie-MIME compliant form
// --------------------------------------------------------------------
static  void    __cookie_mime_time ( time_t aTime ) {
    struct tm * tms = ::gmtime( &aTime );
    int         year= (tms->tm_year) % 100;

    ::printf( "%s, ", __weekday[tms->tm_wday] );
    ::printf( "%d-",  tms->tm_mday );
    ::printf( "%s-",  __month[tms->tm_mon] );
    ::printf( "%s%d ", year < 10 ? "0" : "", year );
    ::printf( "%s%d:", tms->tm_hour < 10 ? "0" : "", tms->tm_hour );
    ::printf( "%s%d:", tms->tm_min  < 10 ? "0" : "", tms->tm_min );
    ::printf( "%s%d ", tms->tm_sec  < 10 ? "0" : "", tms->tm_sec );
    ::printf( "GMT" );
}

// --------------------------------------------------------------------
// local: Encode a link
// --------------------------------------------------------------------
static	void	__Encode_nib ( char aNib ) {
    ::putchar( __encode_hex_nibs[ aNib & 0x0f ] );
}

// --------------------------------------------------------------------
static	void	__Encode_hex ( char aHex ) {
    ::putchar( '%' );
    __Encode_nib ( aHex >> 4 );
    __Encode_nib ( aHex );
}

// --------------------------------------------------------------------
// public:  General HTTP 1.0 - 1.1 header, Set-Cookie
// --------------------------------------------------------------------
extern  void                HTTP_cookie(	const char *	aName,
											const char *	aValue,
											const char *	aPath,
											const char *	aComment ) {
	dword_t	lifetime = 24 * 14 * 3600;	// Two weeks

	::printf( "Set-Cookie: %s=%s; expires=", aName, aValue );
	__cookie_mime_time( ::time( NULL ) + lifetime );
	if	( aPath )		::printf( "; path=%s", aPath );
	if	( aComment ) {
		::printf( "; comment=" );
		::CGI_quote( aComment );
	}
	::printf( "\n" );
}

// --------------------------------------------------------------------
// public:  General HTTP 1.0 - 1.1 header, Kill a cookie
// --------------------------------------------------------------------
extern  void                HTTP_kill_cookie(	const char *	aName,
												const char *	aValue,
												const char *	aPath ) {

	::printf( "Set-Cookie: %s=%s; expires=", aName, aValue );
	__cookie_mime_time( ::time( NULL ) - 24 * 3600 * 14 );	// Time in the past
	if	( aPath )		::printf( "; path=%s", aPath );
	::printf( "\n" );
}

// --------------------------------------------------------------------
// public:  General HTTP 1.0 - 1.1 header, HTML
// --------------------------------------------------------------------
extern  void                HTTP_html( const char * aLanguage ) {
	::printf( "Date: " );   		__mime_time( ::time( NULL ) );
	::printf( "Last-Modified: ");	__mime_time( ::time( NULL ) - 600 );
	::printf( "Expires: ");			__mime_time( ::time( NULL ) - 300 );
	::printf( "Pragma: no-cache\n" );
    ::printf( "Content-Language: %s\n", aLanguage );
    ::printf( "Content-type: text/html\n" );
    ::printf( "\n" );
}

// --------------------------------------------------------------------
// public:  General HTTP 1.0 - 1.1 header, text
// --------------------------------------------------------------------
extern  void                HTTP_text( const char * aLanguage ) {
	::printf( "Date: " );   		__mime_time( ::time( NULL ) );
	::printf( "Last-Modified: ");	__mime_time( ::time( NULL ) - 600 );
	::printf( "Expires: ");			__mime_time( ::time( NULL ) - 300 );
	::printf( "Pragma: no-cache\n" );
    ::printf( "Content-Language: %s\n", aLanguage );
    ::printf( "Content-type: text/plain\n" );
    ::printf( "\n" );
}

// --------------------------------------------------------------------
// public:  General HTTP 1.0 - 1.1 header, image
// --------------------------------------------------------------------
extern  void                HTTP_image( const char * aImageType,
										dword_t		 aImageLength ) {
	::printf( "Date: " );   		__mime_time( ::time( NULL ) );
	::printf( "Last-Modified: ");	__mime_time( ::time( NULL ) - 600 );
	::printf( "Expires: ");			__mime_time( ::time( NULL ) - 300 );
	::printf( "Pragma: no-cache\n" );
	if	( aImageLength > 0 ) {
	    ::printf( "Content-Length: %d\n", aImageLength );
	}
    ::printf( "Content-type: image/%s\n", aImageType );
    ::printf( "\n" );
}

// --------------------------------------------------------------------
// public:  General HTTP 1.0 - 1.1 header, general mime
// --------------------------------------------------------------------
extern  void                HTTP_mime( 	const char * aMime,
										dword_t		 aImageLength ) {
	::printf( "Date: " );   		__mime_time( ::time( NULL ) );
	::printf( "Last-Modified: ");	__mime_time( ::time( NULL ) - 600 );
	::printf( "Expires: ");			__mime_time( ::time( NULL ) - 300 );
	::printf( "Pragma: no-cache\n" );
	if	( aImageLength > 0 ) {
	    ::printf( "Content-Length: %d\n", aImageLength );
	}
    ::printf( "Content-type: %s\n", aMime );
    ::printf( "\n" );
}

// --------------------------------------------------------------------
// public:  General HTTP 1.0 - 1.1 header, spesific for zip files
// --------------------------------------------------------------------
extern  void                HTTP_ziphead(	dword_t		aStartOffset,
											dword_t		aEndOffset,
											dword_t		aTotalLength ) {

	::printf( "Date: " );   		__mime_time( ::time( NULL ) );
	::printf( "Last-Modified: ");	__mime_time( ::time( NULL ) - 600 );
	::printf( "Expires: ");			__mime_time( ::time( NULL ) - 300 );
	::printf( "Pragma: no-cache\n" );
	::printf( "Accept-Ranges: bytes\n" );
    ::printf( "Content-Length: %d\n", aEndOffset - aStartOffset + 1 );
    ::printf( "Content-Range: bytes %d-%d/%d\n", aStartOffset, aEndOffset, aTotalLength );
	::printf( "Connection: close\n" );
    ::printf( "Content-type: application/zip\n" );
    ::printf( "\n" );
}

// --------------------------------------------------------------------
// public:  Output a quoted string suitable for HTML processing
// --------------------------------------------------------------------
extern  void    HTML_quotec ( char aChar ) {
	::printf( "%s", __html_char_encoding[ aChar & 0xff ] );
}

// --------------------------------------------------------------------
// public:  Output a quoted string suitable for HTML processing
// --------------------------------------------------------------------
extern  void    HTML_quote ( const char * aText ) {
    while   ( *aText ) {
        ::printf( "%s", __html_char_encoding[ *(aText++) & 0xff ] );
    }
}

// --------------------------------------------------------------------
// public:  Output a CGI quoted string
// --------------------------------------------------------------------
extern  void    CGI_quote ( const char * aVal ) {
    while   ( *aVal ) {
        if      ( ( *aVal >= 'A' ) && ( *aVal <= 'Z' ) )	::putchar( *(aVal++) );
        else if ( ( *aVal >= 'a' ) && ( *aVal <= 'z' ) )	::putchar( *(aVal++) );
        else if ( ( *aVal >= '0' ) && ( *aVal <= '9' ) )	::putchar( *(aVal++) );
        else if ( *aVal == ' ' )							{ ::putchar( '+' ); aVal++; }
        else												__Encode_hex ( *(aVal++) );
    }
}

// --------------------------------------------------------------------
// EOF: HTTP.cxx
// --------------------------------------------------------------------
