// --------------------------------------------------------------------
// CUnFile.cxx
// Whatis:  Unreal file reader
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 24-NOV-2001     Created this source
// --------------------------------------------------------------------
#include    "CError.hxx"
#include    "CUnFile.hxx"
#include    "CIOConvert.hxx"

// --------------------------------------------------------------------
// public:      Constructor #1
// --------------------------------------------------------------------
CUnFile::CUnFile ( const char * aFileName ) {
	CUnFile::Cleanup();

	itsFileName = ::my_private_strdup( aFileName );

	// ----------------------------------------------------------------
	// Get the file statisctics
	// ----------------------------------------------------------------
	if  ( ::stat( itsFileName, &itsFileStat ) ) {
		throw CError( itsFileName, ::strerror( errno ) );
	}

	itsStream = ::fopen( itsFileName, "rb" );
	if  ( ! itsStream )     throw CError( itsFileName, ::strerror( errno ) );

	try {
		CUnFile::Setup();
	}

	catch ( CError e ) {
		if  ( itsFileName )         delete [] itsFileName;
		if  ( itsGenExportCount )   delete [] itsGenExportCount;
		if  ( itsGenNameCount )     delete [] itsGenNameCount;
		if  ( itsStream )           ::fclose( itsStream );
		CUnFile::Cleanup();
		throw CError( "CUnFile::CUnFile", e.Error() );
	}

	catch ( ... ) {
		if  ( itsFileName )         delete [] itsFileName;
		if  ( itsGenExportCount )   delete [] itsGenExportCount;
		if  ( itsGenNameCount )     delete [] itsGenNameCount;
		if  ( itsStream )           ::fclose( itsStream );
		CUnFile::Cleanup();
		throw;
	}
}

// --------------------------------------------------------------------
// public:      Destructor
// --------------------------------------------------------------------
CUnFile::~CUnFile() {
	if  ( itsFileName )         delete [] itsFileName;
	if  ( itsGenExportCount )   delete [] itsGenExportCount;
	if  ( itsGenNameCount )     delete [] itsGenNameCount;
	if  ( itsStream )           ::fclose( itsStream );
	CUnFile::Cleanup();
}

// --------------------------------------------------------------------
// private:     Read in global header
// --------------------------------------------------------------------
void    CUnFile::Setup ( void ) {
	try {
		dword_t     tempdword;
		itsSignature = CUnFile::ReadDword();
		if  ( itsSignature != UN_FILE_SIGNATURE ) {
			throw CError( itsFileName, "Not an Unreal package" );
		}
	
		tempdword = CUnFile::ReadDword();
		itsPackageVersion   = (word_t)( tempdword & 0xffff);
		itsLicenseeMode     = (word_t)( (tempdword  >> 16 ) & 0xffff );
		if  ( ( itsPackageVersion < UN_MIN_PACKAGE_VERSION ) ||
			  ( itsPackageVersion > UN_MAX_PACKAGE_VERSION ) ) {
			char	versbuf[32];
			::sprintf( versbuf, "%d", itsPackageVersion );
			throw CError( itsFileName, "Version out of range", versbuf );
		}
	
		itsPackageFlags     = CUnFile::ReadDword();
		itsNameCount        = CUnFile::ReadDword();
		itsNameOffset       = CUnFile::ReadDword();
		itsExportCount      = CUnFile::ReadDword();
		itsExportOffset     = CUnFile::ReadDword();
		itsImportCount      = CUnFile::ReadDword();
		itsImportOffset     = CUnFile::ReadDword();
		if  ( itsPackageVersion < 68 ) {
			itsHeritageCount    = CUnFile::ReadDword();
			itsHeritageOffset   = CUnFile::ReadDword();
			CUnFile::Seek( itsHeritageOffset );
			CUnFile::ReadGuid();
		}
		else {
			CUnFile::ReadGuid();
			itsGenerationCount  = CUnFile::ReadDword();
			if  ( itsGenerationCount > 0 ) {
				itsGenExportCount   = new dword_t[ itsGenerationCount ];
				itsGenNameCount     = new dword_t[ itsGenerationCount ];
				for ( tempdword = 0; tempdword < itsGenerationCount; tempdword++ ) {
					itsGenExportCount[tempdword]    = CUnFile::ReadDword();
					itsGenNameCount[tempdword]      = CUnFile::ReadDword();
				}
			}
		}
	}

	catch ( CError e ) {
		throw CError( "CUnFile::Setup", e.Error() );
	}
}

// --------------------------------------------------------------------
// private:     Instance variable cleanup
// --------------------------------------------------------------------
void    CUnFile::Cleanup ( void ) {
	itsFileName         = NULL;
	itsStream           = NULL;
	itsSignature        = 0;
	itsPackageVersion   = 0;
	itsLicenseeMode     = 0;
	itsPackageFlags     = 0;
	itsNameCount        = 0;
	itsNameOffset       = 0;
	itsExportCount      = 0;
	itsExportOffset     = 0;
	itsImportCount      = 0;
	itsImportOffset     = 0;
	itsHeritageCount    = 0;
	itsHeritageOffset   = 0;
	itsGenerationCount  = 0;
	itsGenExportCount   = NULL;
	itsGenNameCount     = NULL;
	::memset( itsGUID, 0, sizeof( itsGUID ) );
	::memset( &itsFileStat, 0, sizeof( itsFileStat ) );
}

// --------------------------------------------------------------------
// public:      Read raw data
// --------------------------------------------------------------------
void        CUnFile::ReadRawData    ( void * aBuf, size_t aLen ) {
	try	{
		dword_t bfp = 0;
		size_t  gbl;
		::memset( aBuf, 0, aLen );
		while   ( bfp < aLen ) {
			if  ( bfp + 8192 <= aLen ) {
				gbl = ::fread( ((char *)aBuf) + bfp, 8192, 1, itsStream );
				bfp += 8192;
			}
			else {
				gbl = ::fread( ((char *)aBuf) + bfp, aLen - bfp, 1, itsStream );
				bfp = aLen;
			}
			if      ( ferror( itsStream ) ) throw CError( itsFileName, ::strerror( errno ) );
			else if ( gbl != 1 ) {
				char	ermsg[128];
				::sprintf( ermsg, "Not enough data (wanted %d bytes)", (dword_t)aLen );
				throw CError( itsFileName, ermsg );
			}
		}
	}

	catch ( CError e ) {
		throw CError( "CUnFile::ReadRawData", e.Error() );
	}
}

// --------------------------------------------------------------------
// public:      Read a byte
// --------------------------------------------------------------------
byte_t      CUnFile::ReadByte   ( void ) {
	try	{
		byte_t          b = 0;
		CUnFile::ReadRawData( &b, sizeof(b) );
		return  b;
	}

	catch ( CError e ) {
		throw CError( "CUnFile::ReadByte", e.Error() );
	}
}

// --------------------------------------------------------------------
// public:      Read a word
// --------------------------------------------------------------------
word_t      CUnFile::ReadWord   ( void ) {
	try {
		CIOConvert_t    c;    
		word_t          w = 0;
		CUnFile::ReadRawData( &w, sizeof(w) );
		return  c.Import( w );
	}

	catch ( CError e ) {
		throw CError( "CUnFile::ReadWord", e.Error() );
	}
}

// --------------------------------------------------------------------
// public:      Read a dword
// --------------------------------------------------------------------
dword_t     CUnFile::ReadDword   ( void ) {
	try {
		CIOConvert_t    c;    
		dword_t         dw = 0;
		CUnFile::ReadRawData( &dw, sizeof(dw) );
		return  c.Import( dw );
	}

	catch ( CError e ) {
		throw CError( "CUnFile::ReadDword", e.Error() );
	}
}

// --------------------------------------------------------------------
// public:      Read a one byte integer
// --------------------------------------------------------------------
int1_t      CUnFile::ReadInt1   ( void ) {
	try	{
		int1_t          i1 = 0;
		CUnFile::ReadRawData( &i1, sizeof(i1) );
		return  i1;
	}

	catch ( CError e ) {
		throw CError( "CUnFile::ReadInt1", e.Error() );
	}
}

// --------------------------------------------------------------------
// public:      Read a two byte integer
// --------------------------------------------------------------------
int2_t      CUnFile::ReadInt2   ( void ) {
	try	{
		CIOConvert_t    c;    
		int2_t          i2 = 0;
		CUnFile::ReadRawData( &i2, sizeof(i2) );
		return  c.Import( i2 );
	}

	catch ( CError e ) {
		throw CError( "CUnFile::ReadInt2", e.Error() );
	}
}

// --------------------------------------------------------------------
// public:      Read a four byte integer
// --------------------------------------------------------------------
int4_t      CUnFile::ReadInt4   ( void ) {
	try	{
		CIOConvert_t    c;    
		int4_t          i4 = 0;
		CUnFile::ReadRawData( &i4, sizeof(i4) );
		return  c.Import( i4 );
	}

	catch ( CError e ) {
		throw CError( "CUnFile::ReadInt4", e.Error() );
	}
}

// --------------------------------------------------------------------
// public:      Read a float
// --------------------------------------------------------------------
float       CUnFile::ReadFloat   ( void ) {
	try {
		CIOConvert_t    c;    
		float           ff = 0.0;
		CUnFile::ReadRawData( &ff, sizeof(ff) );
		return  c.Import( ff );
	}

	catch ( CError e ) {
		throw CError( "CUnFile::ReadFloat", e.Error() );
	}
}

// --------------------------------------------------------------------
// public:      Read an index - special compression used in Un files
// --------------------------------------------------------------------
#define INDEX_SIGN_BIT  0x80        // Sign bit on first byte
#define INDEX_CONT1_BIT 0x40        // Continuation bit on first byte
#define INDEX_BYT1_MASK 0x3f        // Data mast on first byte
#define INDEX_CONT_BIT  0x80        // Continuation bit on other bytes
#define INDEX_BYTE_MASK 0x7f        // Data mask on other bytes
// --------------------------------------------------------------------
dword_t     CUnFile::ReadIndex  ( void ) {
	try {
		byte_t          byte;
		dword_t         data;
		dword_t         result;
		word_t          shift   = 6;
		bool            sign;

		// ------------------------------------------------------------
		// Read in the first byte separately
		// ------------------------------------------------------------
		byte = CUnFile::ReadByte();
		sign = (byte & INDEX_SIGN_BIT) != 0;
		result = byte & INDEX_BYT1_MASK;

		// ------------------------------------------------------------
		// Read in the following bytes as needed
		// ------------------------------------------------------------
		if  ( byte & INDEX_CONT1_BIT ) {
			do  {
				byte = CUnFile::ReadByte();
				data = ((dword_t)byte) & INDEX_BYTE_MASK;
				data = data << shift;
				result = result | data;
				shift = (word_t)(shift + 7);
			}   while   ( ( byte & INDEX_CONT_BIT ) && ( shift < 32 ) );
		}

		if  ( sign ) {
			result = (dword_t)( -((int4_t)result) );
		}
		return  result;
	}

	catch ( CError e ) {
		throw CError( "CUnFile::ReadIndex", e.Error() );
	}
}

// --------------------------------------------------------------------
// public:      Read an array index - special compression used in Un files
// --------------------------------------------------------------------
dword_t CUnFile::ReadArrIdx ( void ) {
	try	{
		dword_t index;
		byte_t  ib;
		word_t  iw;
		ib = CUnFile::ReadByte();
		if  ( ib < 128 ) {
			index = (dword_t)ib;
		}
		else {
			CUnFile::Seek( CUnFile::Offset() - 1 );
			iw = (word_t)(CUnFile::ReadWord() & 0x7fff);
			if  ( iw < 16384 ) {
				index = (dword_t)iw;
			}
			else {
				CUnFile::Seek( CUnFile::Offset() - 2 );
				index = CUnFile::ReadDword() & 0x3fffffff;
			}
		}
		return  index;
	}
	catch ( CError e ) {
		throw CError( "CUnFile::ReadArrIdx", e.Error() );
	}
}

// --------------------------------------------------------------------
// public:      Read some text
// --------------------------------------------------------------------
void    CUnFile::ReadText   ( char * aBuf, size_t aLen ) {
	try {
		::memset( aBuf, 0, aLen );
		if  	( itsPackageVersion < 64 ) {
			size_t  size;
			size = 0;
			while   ( size < aLen ) {
				CUnFile::ReadRawData( aBuf + size, 1 );
				if  ( aBuf[size] == 0 ) break;
				size++;
				if  ( size == aLen ) {
					throw CError( "Invalid data (no text terminator found)" );
				}
			}
		}
		else {
        	dword_t	dwsize;
			dwsize = (size_t)CUnFile::ReadIndex();

			if  ( ( dwsize > aLen ) || ( dwsize & 0x80000000 ) ) {
				throw CError( "Invalid data (text size too big or negative)" );
			}
			CUnFile::ReadRawData( aBuf, dwsize );
		}
	}
	catch ( CError e ) {
		throw CError( "CUnFile::ReadText", e.Error() );
	}
}

// --------------------------------------------------------------------
// public:      File management - Seek
// --------------------------------------------------------------------
void    CUnFile::Seek   ( long aOffset ) {
	try {
		::fseek( itsStream, aOffset, SEEK_SET );
		if  ( ferror( itsStream ) ) {
			throw CError( itsFileName, ::strerror( errno ) );
		}
	}
	catch ( CError e ) {
		throw CError( "CUnFile::Seek", e.Error() );
	}
}

// --------------------------------------------------------------------
// public:      File management - Offset
// --------------------------------------------------------------------
long    CUnFile::Offset  ( void ) const {
	return	::ftell( itsStream );
}

// --------------------------------------------------------------------
// private:     Generate one nibble for guid
// --------------------------------------------------------------------
void    CUnFile::GuidNibble ( char * aB, char aN, const char * aHx ) {
	*aB = aHx[ aN & 0x0f ];
}

// --------------------------------------------------------------------
// private:     Generate one byte for guid
// --------------------------------------------------------------------
void    CUnFile::GuidByte   ( char * aB, char aN, const char * aHx ) {
	CUnFile::GuidNibble( aB + 0, (char)(aN >> 4), aHx );
	CUnFile::GuidNibble( aB + 1, aN,      aHx );
}

// --------------------------------------------------------------------
// private:     Generate the entire guid string
// --------------------------------------------------------------------
void    CUnFile::GuidString ( const char * aGuid, size_t aLen, const char * aHx ) {
	char *  w = itsGUID;
	for ( size_t r = 0; r < aLen; r++ ) {
		CUnFile::GuidByte( w, aGuid[r], aHx );
		w += 2;
	}
	*w = 0;
}

// --------------------------------------------------------------------
// private:     Read the guid from the file
// --------------------------------------------------------------------
void    CUnFile::ReadGuid   ( void ) {
	char    myguid  [16];
	CUnFile::ReadRawData( myguid, 16 );
	CUnFile::GuidString ( myguid, 16, "0123456789ABCDEF" );
}

// --------------------------------------------------------------------
const char *	CUnFile::FileName			( void ) const { return	itsFileName; }
time_t			CUnFile::FileTime			( void ) const { return	itsFileStat.st_mtime; }
size_t			CUnFile::FileSize			( void ) const { return	itsFileStat.st_size; }
dword_t			CUnFile::Signature			( void ) const { return	itsSignature; }
word_t			CUnFile::PackageVersion		( void ) const { return	itsPackageVersion; }
word_t			CUnFile::LicenseeMode		( void ) const { return	itsLicenseeMode; }
dword_t			CUnFile::PackageFlags		( void ) const { return	itsPackageFlags; }
bool			CUnFile::AllowDownload		( void ) const { return	(itsPackageFlags & PKG_AllowDownload) != 0;	}
bool			CUnFile::ClientOptional		( void ) const { return	(itsPackageFlags & PKG_ClientOptional) != 0; }
bool			CUnFile::ServerSideOnly		( void ) const { return	(itsPackageFlags & PKG_ServerSideOnly) != 0; }
bool			CUnFile::BrokenLinks		( void ) const { return	(itsPackageFlags & PKG_BrokenLinks)	!= 0; }
bool			CUnFile::Unsecure			( void ) const { return	(itsPackageFlags & PKG_Unsecure) !=	0; }
bool			CUnFile::Need				( void ) const { return	(itsPackageFlags & PKG_Need) !=	0; }
dword_t			CUnFile::NameCount			( void ) const { return	itsNameCount; }
dword_t			CUnFile::NameOffset			( void ) const { return	itsNameOffset; }
dword_t			CUnFile::ExportCount		( void ) const { return	itsExportCount;	}
dword_t			CUnFile::ExportOffset		( void ) const { return	itsExportOffset; }
dword_t			CUnFile::ImportCount		( void ) const { return	itsImportCount;	}
dword_t			CUnFile::ImportOffset		( void ) const { return	itsImportOffset; }
dword_t			CUnFile::HeritageCount		( void ) const { return	itsHeritageCount; }
dword_t			CUnFile::HeritageOffset		( void ) const { return	itsHeritageOffset; }
const char *	CUnFile::GUID				( void ) const { return	itsGUID; }
dword_t			CUnFile::GenerationCount	( void ) const { return	itsGenerationCount;	}
const dword_tp	CUnFile::GenExportCount		( void ) const { return	itsGenExportCount; }
const dword_tp	CUnFile::GenNameCount		( void ) const { return	itsGenNameCount; }
FILE *			CUnFile::Stream				( void )	   { return	itsStream; }

// --------------------------------------------------------------------
// EOF: CUnFile.cxx
// --------------------------------------------------------------------
