// --------------------------------------------------------------------
// CUnLevelInfo.cxx
// Whatis:  Level info
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 24-NOV-2001     Created this source
// --------------------------------------------------------------------
#include    "CError.hxx"
#include    "CUnLevelInfo.hxx"
#include    "CUnMaterialSequence.hxx"

// --------------------------------------------------------------------
// public:      Constructor
// --------------------------------------------------------------------
CUnLevelInfo::CUnLevelInfo (        CUnFile_t &               aUf,
                                    const CUnNameTable_t &    aUn,
                                    const CUnExportTable_t &  aUe,
                                    const CUnImportTable_t &  aUi ) {
    CUnProperty		property;
    dword_t     	n_author		= aUn.Find( "author" );
    dword_t     	n_audio			= aUn.Find( "song" );
    dword_t     	n_title			= aUn.Find( "title" );
    dword_t			n_description	= aUn.Find( "description" );
    dword_t     	n_entertext		= aUn.Find( "levelentertext" );
    dword_t     	n_idealcount	= aUn.Find( "idealplayercount" );
    dword_t     	n_screenshot	= aUn.Find( "screenshot" );
    dword_t     	n_aiprofile		= aUn.Find( "aiprofile" );
    dword_t     	n_level			= aUn.Find( "level" );
    dword_t			n_playermin		= aUn.Find( "idealplayercountmin" );
    dword_t			n_playermax		= aUn.Find( "idealplayercountmax" );
	dword_t			playermin		= 0;
	dword_t			playermax		= 0;
    const char *	propname;


    CUnLevelInfo::Cleanup();

	try {
	    CUnLevelInfo::SetFileReader( aUf, aUn, aUe, aUi );

		do	{
	        property = CUnProperty( aUf, aUn, aUi, aUe );
            propname = aUn.LcName( property.Name() );
	        property.Rewind();

	        if      ( property.Name() == n_author )     	CUnLevelInfo::SetMyAuthor         ( property );
	        else if ( property.Name() == n_audio )      	CUnLevelInfo::SetMyAudio          ( property );
	        else if ( property.Name() == n_title )      	CUnLevelInfo::SetMyTitle          ( property );
	        else if ( property.Name() == n_entertext )  	CUnLevelInfo::SetMyEnterText      ( property );
	        else if ( property.Name() == n_description )  	CUnLevelInfo::SetMyDescription    ( property );
	        else if ( property.Name() == n_idealcount ) 	CUnLevelInfo::SetMyIdealPlayers   ( property );
	        else if ( property.Name() == n_screenshot )		CUnLevelInfo::SetMyScreenshot     ( property, aUf, aUn, aUe, aUi );
	        else if ( property.Name() == n_playermin )		playermin = property.GetWord();
	        else if ( property.Name() == n_playermax )		playermax = property.GetWord();

	        if  ( property.Name() == n_level )      break;
	        if  ( property.Name() == n_aiprofile )	break;

	        if  ( ( property.Name() == n_screenshot ) && ( aUf.PackageVersion() < 100 ) )	break;

	    }   while   ( ::strcmp( "none", propname ) );

		if	( ( ! itsIdealPlayers ) && ( playermin < playermax ) ) {
			char mystrbuf[32];
			::sprintf( mystrbuf, "%d - %d", playermin, playermax );
			itsIdealPlayers = ::my_private_strdup( mystrbuf );
		}

	}

	catch ( CError e ) {
	    CUnLevelInfo::Free();
		throw	CError( "CUnLevelInfo::CUnLevelInfo", e.Error() );
	}

	catch ( ... ) {
	    CUnLevelInfo::Free();
		throw	CError( "CUnLevelInfo::CUnLevelInfo", "unknown" );
	}
}

// --------------------------------------------------------------------
// public:      Destructor
// --------------------------------------------------------------------
CUnLevelInfo::~CUnLevelInfo () {
    CUnLevelInfo::Free();
}

// --------------------------------------------------------------------
// private:     Set diverse data
// --------------------------------------------------------------------
void    CUnLevelInfo::SetMyAuthor         ( CUnProperty_t & aP ) {
	try	{
	    itsAuthor = ::my_private_strdup( (const char *)aP.Data() );
	}
	catch	( ... ) {
		throw CError( "CUnLevelInfo::SetMyAuthor", "unknown" );
	}
}

// --------------------------------------------------------------------
void    CUnLevelInfo::SetMyAudio         ( CUnProperty_t & aP ) {
	try	{
	    itsAudio = ::my_private_strdup( (const char *)aP.Data() );
	}
	catch	( ... ) {
		throw CError( "CUnLevelInfo::SetMyAudio", "unknown" );
	}
}

// --------------------------------------------------------------------
void    CUnLevelInfo::SetMyTitle          ( CUnProperty_t & aP ) {
	try	{
	    itsTitle  = ::my_private_strdup( (const char *)aP.Data() );
	}
	catch	( ... ) {
		throw CError( "CUnLevelInfo::SetMyTitle", "unknown" );
	}
}

// --------------------------------------------------------------------
void    CUnLevelInfo::SetMyEnterText      ( CUnProperty_t & aP ) {
	try	{
    	itsEnterText = ::my_private_strdup( (const char *)aP.Data() );
	}
	catch	( ... ) {
		throw CError( "CUnLevelInfo::SetMyEnterText", "unknown" );
	}
}

// --------------------------------------------------------------------
void    CUnLevelInfo::SetMyDescription    ( CUnProperty_t & aP ) {
	try	{
    	if	( ! itsEnterText ) {
	    	itsEnterText = ::my_private_strdup( (const char *)aP.Data() );
            if	( itsEnterText ) {
            	while	( ::strchr( itsEnterText, '|' ) ) {
                	*::strchr( itsEnterText, '|' ) = '\n';
                }
            }
		}
	}
	catch	( ... ) {
		throw CError( "CUnLevelInfo::SetMyDescription", "unknown" );
	}
}

// --------------------------------------------------------------------
void    CUnLevelInfo::SetMyIdealPlayers   ( CUnProperty_t & aP ) {
	try	{
	    itsIdealPlayers = ::my_private_strdup( (const char *)aP.Data() );
	}
	catch	( ... ) {
		throw CError( "CUnLevelInfo::SetMyIdealPlayers", "unknown" );
	}
}

// --------------------------------------------------------------------
void    CUnLevelInfo::SetMyScreenshot     ( CUnProperty_t &           aP,
                                            CUnFile_t &               aUf,
                                            const CUnNameTable_t &    aUn,
                                            const CUnExportTable_t &  aUe,
                                            const CUnImportTable_t &  aUi ) {
    long		rmboffset	= aUf.Offset();

	try {
		aP.Rewind();
		CUnExport   myexport	= aUe.Export( NAMEEXPORT( aP.GetWord() ) );
		dword_t		classname	= aUi.Import( NAMEIMPORT( myexport.Class() ) ).ObjectName();

		// ------------------------------------------------------------
		// Some maps have class "material" instead of "texture" as screenshot
		// ------------------------------------------------------------
		if		( classname == aUn.Find( "materialsequence" ) ) {
			aUf.Seek( myexport.SerialOffset() );
			CUnLevelInfo::SetMyScreenshotSequence( aUf, aUn, aUe, aUi );
		}
		else if	( classname == aUn.Find( "texture" ) ) {
			aUf.Seek( myexport.SerialOffset() );
			CUnLevelInfo::SetMyScreenshotPicture( aUf, aUn, aUe, aUi );
		}

		else {
			CUnLevelInfo::FreeScreenies();
		}
	}

	catch	( ... ) {
		CUnLevelInfo::FreeScreenies();
	}

	aUf.Seek( rmboffset );
}

// --------------------------------------------------------------------
void    CUnLevelInfo::SetMyScreenshotSequence (	CUnFile_t &               aUf,
	                                            const CUnNameTable_t &    aUn,
	                                            const CUnExportTable_t &  aUe,
	                                            const CUnImportTable_t &  aUi ) {
	dword_t					texname	= aUn.Find( "texture" );
	CUnMaterialSequence		sequence( aUf, aUn, aUe, aUi );
	material_sequence_tlci	seqloop;
	dword_tl				objlist;
	dword_tli				objloop;

	// ----------------------------------------------------------------
	// Get all different texture objects into a list
	// ----------------------------------------------------------------
	for	( 	seqloop =  sequence.Sequence().begin();
			seqloop != sequence.Sequence().end();
			seqloop++ ) {
		CUnExport	myexport = aUe.Export( NAMEEXPORT( (*seqloop).msa_object ) );

		// Only textures get processed
		if	( aUi.Import( NAMEIMPORT( myexport.Class() ) ).ObjectName() == texname ) {
			for	( objloop = objlist.begin(); objloop != objlist.end(); objloop++ ) {
				if	( *objloop == (*seqloop).msa_object )	break;
			}
			if	( objloop == objlist.end() ) {
				objlist.push_back( (*seqloop).msa_object );
			}
		}
	}

	// ----------------------------------------------------------------
	// Set screenies using them
	// ----------------------------------------------------------------
	for	( objloop = objlist.begin(); objloop != objlist.end(); objloop++ ) {
		CUnExport	myexport = aUe.Export( NAMEEXPORT( *objloop ) );
		aUf.Seek( myexport.SerialOffset() );
		CUnLevelInfo::SetMyScreenshotPicture( aUf, aUn, aUe, aUi );
	}
}

// --------------------------------------------------------------------
void    CUnLevelInfo::SetMyScreenshotPicture  (	CUnFile_t &               aUf,
	                                            const CUnNameTable_t &    aUn,
	                                            const CUnExportTable_t &  aUe,
	                                            const CUnImportTable_t &  aUi ) {
	CUnTexture *	texture = new CUnTexture( aUf, aUn, aUe, aUi );

	if	( ! itsScreenshot ) {
		itsScreenshot = new CUnTexture * [1];
		itsScreenshot[0] = texture;
		itsShotCount = 1;
	}
	else {
		CUnTexture ** 	mycopy = new CUnTexture * [itsShotCount + 1];
		word_t			mycount;

		for	( mycount = 0; mycount < itsShotCount; mycount++ ) {
			mycopy[mycount] = itsScreenshot[mycount];
			itsScreenshot[mycount] = NULL;
		}
		CUnLevelInfo::FreeScreenies();
		mycopy[mycount++]	= texture;
		itsScreenshot 		= mycopy;
		itsShotCount		= mycount;
	}
}

// --------------------------------------------------------------------
// private:     Find the LevelInfo object and place the file there
// --------------------------------------------------------------------
void    CUnLevelInfo::SetFileReader(    CUnFile_t &              aUf,
                                        const CUnNameTable_t &   aUn,
                                        const CUnExportTable_t & aUe,
                                        const CUnImportTable_t & aUi ) {
	try {

	    dword_t i_cname, i_lname, o_levelinfo;

	    // ----------------------------------------------------------------
	    // Find the leveinfo record
	    // ----------------------------------------------------------------
	    o_levelinfo = 0;
	    i_lname = aUn.Find( "LevelInfo" );
	    for ( dword_t e = 0; e < aUe.Count(); e++ ) {
	        CUnExport_t     exp = aUe.Export( e );

	        // ------------------------------------------------------------
	        // Extract the class name of the item
	        // ------------------------------------------------------------
	        i_cname = exp.Class();
	        i_cname = aUi.Import( NAMEIMPORT( i_cname ) ).ObjectName();

	        if  ( i_cname == i_lname ) {
	            o_levelinfo = exp.SerialOffset();
	            break;
	        }
	    }

	    // ----------------------------------------------------------------
	    // No LevelInfo - Propably not a map !
	    // ----------------------------------------------------------------
	    if  ( o_levelinfo == 0 ) {
	        throw   CError( aUf.FileName(), "Missing level info" );
	    }

	    aUf.Seek( o_levelinfo );
	    dword_t fr_node = aUf.ReadIndex();  // Node
	    aUf.ReadIndex();                    // StateNode
	    aUf.Seek( aUf.Offset() + 12 );      // ProbeMask and LatentAction
	    if  ( fr_node ) aUf.ReadIndex();    // Offset
	}

	catch	( CError e ) {
		throw CError( "CUnLevelInfo::SetFileReader", e.Error() );
	}

	catch	( ... ) {
		throw CError( "CUnLevelInfo::SetFileReader", "unknown" );
	}
}

// --------------------------------------------------------------------
// private:     Free up data
// --------------------------------------------------------------------
void    CUnLevelInfo::Free ( void ) {
    if  ( itsAuthor )       delete [] itsAuthor;
    if  ( itsTitle )        delete [] itsTitle;
    if  ( itsEnterText )    delete [] itsEnterText;
    if  ( itsIdealPlayers ) delete [] itsIdealPlayers;
    if  ( itsAudio ) 		delete [] itsAudio;
	CUnLevelInfo::FreeScreenies();
    CUnLevelInfo::Cleanup();
}

// --------------------------------------------------------------------
// private:     Free up data
// --------------------------------------------------------------------
void    CUnLevelInfo::FreeScreenies ( void ) {
    if  ( itsScreenshot ) {
		for	( word_t i = 0; i < itsShotCount; i++ ) {
			if	( itsScreenshot[i] ) {
				delete itsScreenshot[i];
				itsScreenshot[i] = NULL;
			}
		}
	   	delete [] itsScreenshot;
	}
	itsScreenshot	= NULL;
	itsShotCount	= 0;
}

// --------------------------------------------------------------------
// private:     Cleenex
// --------------------------------------------------------------------
void    CUnLevelInfo::Cleanup ( void ) {
    itsAuthor       = NULL;
    itsTitle        = NULL;
    itsEnterText    = NULL;
    itsIdealPlayers = NULL;
    itsScreenshot   = NULL;
    itsAudio		= NULL;
	itsShotCount	= 0;
}

// --------------------------------------------------------------------
const char *    CUnLevelInfo::Author      ( void ) const { return itsAuthor ? itsAuthor : ""; }
const char *    CUnLevelInfo::Title       ( void ) const { return itsTitle ? itsTitle : ""; }
const char *    CUnLevelInfo::EnterText   ( void ) const { return itsEnterText ? itsEnterText : ""; }
const char *    CUnLevelInfo::IdealPlayers( void ) const { return itsIdealPlayers ? itsIdealPlayers : ""; }
const char *    CUnLevelInfo::Audio		  ( void ) const { return itsAudio ? itsAudio : ""; }
word_t			CUnLevelInfo::ShotCount	  ( void ) const { return itsShotCount; }
CUnTexture_tp   CUnLevelInfo::Screenshot  ( word_t aIx ) const { return itsScreenshot[aIx]; }

// --------------------------------------------------------------------
// EOF: CUnLevelInfo.cxx
// --------------------------------------------------------------------
