// --------------------------------------------------------------------
// CUnMasterServer.cxx
// Whatis:	Unreal server info browser
// Authors:	Esko 'Varpu' Ilola	EIL
// History:	EIL	21-APR-2003		Created	this source
// --------------------------------------------------------------------
#include    "CError.hxx"
#include    "CUnMasterServer.hxx"


// --------------------------------------------------------------------
// HTTP header for querying HTTP data
// --------------------------------------------------------------------
static	const char *	http_query[] = {
	"GET /serverlist/full-all.txt HTTP/1.1\x0d\x0a",
	"Accept: */*\x0d\x0a",
	"User-Agent: Mozilla/4.0 (compatible;)\x0d\x0a",
	"Host: ut2003master.epicgames.com\x0d\x0a",
	"Connection: Close\x0d\x0a",
	"\x0d\x0a",
	NULL
};

// --------------------------------------------------------------------
// ##### CUnMasterServerItem
// --------------------------------------------------------------------

// --------------------------------------------------------------------
// public:      Constructor
// --------------------------------------------------------------------
CUnMasterServerItem::CUnMasterServerItem(	const char *	aIpAddress,
											word_t			aQueryPort,
											unreal_server_t	aServerType )
										:	itsIpAddress( NULL ),
											itsQueryPort( 0 ),
											itsServerType( unreal_server_unknown ) {
	itsIpAddress	= ::my_private_strdup( aIpAddress );
	itsQueryPort	= aQueryPort;
	itsServerType	= aServerType;
}

// --------------------------------------------------------------------
CUnMasterServerItem::CUnMasterServerItem(	const CUnMasterServerItem & aC )
										:	itsIpAddress( NULL ),
											itsQueryPort( 0 ),
											itsServerType( unreal_server_unknown ) {
	itsIpAddress	= ::my_private_strdup( aC.IpAddress() );
	itsQueryPort	= aC.QueryPort();
	itsServerType	= aC.ServerType();
}

// --------------------------------------------------------------------
CUnMasterServerItem	& CUnMasterServerItem::operator =	( const CUnMasterServerItem & aC ) {
	if	( itsIpAddress )	delete [] itsIpAddress;
	itsIpAddress	= NULL;
	itsQueryPort	= 0;
	itsServerType	= unreal_server_unknown;
	itsIpAddress	= ::my_private_strdup( aC.IpAddress() );
	itsQueryPort	= aC.QueryPort();
	itsServerType	= aC.ServerType();
	return *this;
}

// --------------------------------------------------------------------
// public:      Destructor
// --------------------------------------------------------------------
CUnMasterServerItem::~CUnMasterServerItem	( ) {
	if	( itsIpAddress )	delete [] itsIpAddress;
	itsIpAddress	= NULL;
	itsQueryPort	= 0;
	itsServerType	= unreal_server_unknown;
}

// --------------------------------------------------------------------
const char *	CUnMasterServerItem::IpAddress	( void ) const { return itsIpAddress ? itsIpAddress : ""; }
word_t			CUnMasterServerItem::QueryPort	( void ) const { return itsQueryPort; }
unreal_server_t	CUnMasterServerItem::ServerType	( void ) const { return itsServerType; }

// --------------------------------------------------------------------
// ##### CUnMasterServer
// --------------------------------------------------------------------

// --------------------------------------------------------------------
// public:      Constructor
// --------------------------------------------------------------------
CUnMasterServer::CUnMasterServer(	const char *	aIpAddress,
									word_t			aQueryPort,
									unreal_server_t	aServerType ) {
	CTcpSocket	sock( socket_type_stream );

	try {
		sock.Bind();

		// Go and connect to the site
		time_t	timenow = ::time( NULL ) + 10;
		while	( timenow > ::time( NULL ) ) {
			if	( sock.Connect( aIpAddress, aQueryPort ) )	break;
		}
		if	( ! sock.Connect( aIpAddress, aQueryPort ) ) {
			throw	0;
		}

		CUnMasterServer::WaitForServer( sock, aServerType );
		CUnMasterServer::SendQuery( sock, aServerType );
		if		( aServerType == unreal_server_ut )		CUnMasterServer::ParseInput( sock, aServerType );
		else if	( aServerType == unreal_server_2003 )	CUnMasterServer::ParseHtml( sock, aServerType );
	}

	catch	( ... ) {
		itsServerList.clear();
	}
}

// --------------------------------------------------------------------
// public:      Destructor
// --------------------------------------------------------------------
CUnMasterServer::~CUnMasterServer() {
	itsServerList.clear();
}

// --------------------------------------------------------------------
// public:      Copy constructor and assignment operator
// --------------------------------------------------------------------
CUnMasterServer::CUnMasterServer				( const CUnMasterServer & aC ) {
	itsServerList = aC.ServerList();
}

// --------------------------------------------------------------------
CUnMasterServer	& CUnMasterServer::operator =	( const CUnMasterServer & aC ) {
	itsServerList = aC.ServerList();
	return *this;
}

// --------------------------------------------------------------------
const CUnMasterServerItem_tl &	CUnMasterServer::ServerList	( void ) const { return itsServerList; }

// --------------------------------------------------------------------
// private:		Private helpers
// --------------------------------------------------------------------
void	CUnMasterServer::WaitForServer	( CTcpSocket & aS, unreal_server_t aT ) {
	const char *	r = CUnMasterServer::ServerBanner( aT );

	while	( *r ) {
		int		rch = aS.Read();

		if		( rch == -1 )				;
		else if	( rch == -2 )				throw 	0;
		else if	( rch != (int)(*(r++)) )	throw	0;
	}
}

// --------------------------------------------------------------------
void	CUnMasterServer::SendQuery		( CTcpSocket & aS, unreal_server_t aT ) {
	if		( aT == unreal_server_ut ) {
		aS.Write( CUnMasterServer::ServerQuery( aT ) );
	}
	else if	( aT == unreal_server_2003 ) {
		const char ** p = http_query;
		while	( *p )	aS.Write( *(p++) );
	}
	aS.Flush();
}

// --------------------------------------------------------------------
void	CUnMasterServer::ParseInput		( CTcpSocket & aS, unreal_server_t aT ) {
	char	ipbuf	[512];
	int		ipoff	= 0;
	char	pnbuf	[512];
	int		pnoff	= 0;
	int		mode	= 0;
	int		rch;

	while	( mode < 20 ) {

		rch = aS.Read();
		if	( rch == -1 )	continue;
		if	( rch == -2 )	break;

		switch	( mode ) {
			case	0:	// Expect to have a '\\'
			case	3:
			if	( rch != '\\' )	throw 0;
			mode++;
			ipoff	= 0;
			pnoff	= 0;
			break;

			case	1:	// Expect to have a 'i'
			if	( rch == 102 ) {
            	mode = 20;
            }
			else if	( rch != 'i' ) {
            	throw 0;
            }
			mode++;
			break;

			case	2:	// Expect to have a 'p'
			if	( rch != 'p' )	throw 0;
			mode++;
			break;

			case	4:	// Store Ip address untill we have a ':' or a terminating '\\'
			if		( rch == ':' ) {
				mode++;
			}
			else if	( rch == '\\' ) {
				mode = 6;
			}
			else {
				ipbuf[ipoff++] = (char)rch;
				if	( ipoff >= (int)sizeof( ipbuf ) )	throw 0;
			}
			break;

			case	5:	// Store port number untill we have a '\\'
			if	( rch == '\\' ) {
				mode++;
			}
			else {
				pnbuf[pnoff++] = (char)rch;
				if	( pnoff >= (int)sizeof( pnbuf ) )	throw 0;
			}
			break;
		}

		// If we reached mode == 6 we store one entry
		if	( mode == 6 ) {
			ipbuf[ipoff]	= 0;	ipoff = 0;
			pnbuf[pnoff]	= 0;	pnoff = 0;


			CUnMasterServerItem	myitem( ipbuf, (word_t)::atol( pnbuf ), aT );
			itsServerList.push_back( myitem );

			// We continue with reading the leading 'i'
			mode = 1;
		}
	}
}

// --------------------------------------------------------------------
const char *	CUnMasterServer::ServerBanner	( unreal_server_t aT ) const {
	const char *	r = "";
	switch	( aT ) {
		case	unreal_server_ut:	r = "\\basic\\\\secure\\wookie";	break;
		case	unreal_server_2003:	r = "";								break;
		default:					throw	0;	// Abort
	}
	return	r;
}


// --------------------------------------------------------------------
const char *	CUnMasterServer::ServerQuery	( unreal_server_t aT ) const {
	const char *	r = "";
	switch	( aT ) {
		case	unreal_server_ut:	r = "\\gamename\\ut\\location\\0\\validate\\0\\final\\\\list\\\\gamename\\ut\\final\\";	break;
		case	unreal_server_2003:	r = "";	break;
		default:					throw	0;	// Abort
	}
	return	r;
}


// --------------------------------------------------------------------
void	CUnMasterServer::ParseHtml		( CTcpSocket & aS, unreal_server_t aT ) {
	char	myline	[1024];
	char *	ip;
	char *	gpn;
	char *	qpn;

	// First get data until we have an empty line (got past http-headers)
	do {
		CUnMasterServer::ReadSockLine( myline, sizeof( myline ), aS );
	}	while	( *myline != 0 );

	// Get the first line and continue as long as we get them
	CUnMasterServer::ReadSockLine( myline, sizeof( myline ), aS );
	while	( *myline ) {
		// Point to the ip-port
		gpn = ip = myline;

		// Advance game port until we have whitespace
		while	( *gpn > ' ' ) gpn++;
		*gpn = 0;	gpn++;

		// Advance query port until we have whitespace
		qpn = gpn;
		while	( *qpn > ' ' ) qpn++;
		*qpn = 0;

		if	( (*ip >= '0') && ( *ip <= '9' ) && ( ::atol( gpn ) > 0 ) ) {
			CUnMasterServerItem	myitem( ip, (word_t)(::atol( gpn ) + 1), aT );
			itsServerList.push_back( myitem );
		}

		CUnMasterServer::ReadSockLine( myline, sizeof( myline ), aS );
	}
}

// --------------------------------------------------------------------
void	CUnMasterServer::ReadSockLine	( char * aBuf, size_t aSize, CTcpSocket & aS ) {
	int		rch = 0;
	size_t	off	= 0;

	::memset( aBuf, 0, aSize );

	while	( rch >= 0 ) {
		rch = aS.Read();
		if	( rch == -1 ) {
			rch = 0;
			continue;
		}
		if	( rch == -2 ) {
			break;
		}
		if	( rch == 0x0d ) {
			continue;
		}
		if	( rch == 0x0a ) {
			break;
		}
		aBuf[off++] = (char)rch;
		if	( off >= aSize ) {
			break;
		}
	}
}

// --------------------------------------------------------------------
// EOF: CUnMasterServer.cpp
// --------------------------------------------------------------------
