// --------------------------------------------------------------------
// CUnProperty.cxx
// Whatis:  Unreal Property
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 24-NOV-2001     Created this source
// --------------------------------------------------------------------
#include    "CError.hxx"
#include    "CUnProperty.hxx"

#define PROPERTY_TYPE_MASK  0x0f
#define PROPERTY_SIZE_MASK  0x70
#define PROPERTY_ARRAY_MASK 0x80


// #define	_DEBUGPROP	1

// --------------------------------------------------------------------
// public:      Constructor #1
// --------------------------------------------------------------------
CUnProperty::CUnProperty (	CUnFile & 				aUf,
							const CUnNameTable &	aUn,
							const CUnImportTable & 	aUi,
							const CUnExportTable & 	aUe ) {
    CUnProperty::Cleanup();

	try {
	    itsName = aUf.ReadIndex();
	
	    // --------------------------------------------------------
	    // In case it is a none entry, we skip rest of the reading
	    // as this is just a terminator for properties
	    // --------------------------------------------------------
	    if  ( ::strcmp( "none", aUn.LcName( itsName ) ) ) {
	
	        // ----------------------------------------------------
	        // Structure of the info byte
	        // 7 6 5 4 3 2 1 0
	        // A S S S T T T T
	        // A = Array bit
	        // S = Size
	        // T = Type
	        // ----------------------------------------------------
	        byte_t  info = aUf.ReadByte();
	
	        itsArray    = (info & PROPERTY_ARRAY_MASK) != 0;
	        itsType     = (byte_t)(info & PROPERTY_TYPE_MASK);
	
	        switch  ( info & PROPERTY_SIZE_MASK ) {
	            case    0x00:  itsSize = 1;                        break;
	            case    0x10:  itsSize = 2;                        break;
	            case    0x20:  itsSize = 4;                        break;
	            case    0x30:  itsSize = 12;                       break;
	            case    0x40:  itsSize = 16;                       break;
	            case    0x50:  itsSize = (dword_t)aUf.ReadByte();  break;
	            case    0x60:  itsSize = (dword_t)aUf.ReadWord();  break;
	            case    0x70:  itsSize = aUf.ReadDword();          break;
	        }
	
	        // ----------------------------------------------------
	        // Read in the data according to the type
	        // ----------------------------------------------------
	        switch  ( itsType ) {
	
	            case    UNP_ByteProperty:	SetByteProperty( aUf );		break;
	            case    UNP_IntProperty:	SetIntProperty( aUf );		break;
	            case    UNP_BoolProperty:	SetBoolProperty();			break;
	            case    UNP_FloatProperty:	SetFloatProperty( aUf );	break;
	            case    UNP_ObjectProperty:
	            case    UNP_NameProperty:	SetObjectProperty( aUf );			break;
	            case    UNP_StrProperty:	SetStringProperty( aUf );			break;
	            case    UNP_StringProperty:	SetAsciizProperty( aUf );			break;
	            case    UNP_StructProperty:	SetStructProperty( aUf, aUn );		break;
	            case    UNP_ArrayProperty:	SetArrayProperty( aUf, aUn, aUi, aUe );	break;

		        // ------------------------------------------------
				// These are considered as unknown and will throw
		        // ------------------------------------------------
	            case    UNP_ClassProperty:
	            case    UNP_VectorProperty:
	            case    UNP_RotatorProperty:
	            case    UNP_MapProperty:
	            case    UNP_FixedArrayProperty:	SetUnknownProperty( aUn, aUf );	break;

	        }
	    }
	}
	catch ( CError e ) {
		dword_t	n = itsName;
	    CUnProperty::Free();
		throw CError( "CUnProperty::CUnProperty", aUn.Name( n ), e.Error() );
	}
	catch ( ... ) {
		dword_t	n = itsName;
	    CUnProperty::Free();
		throw CError( "CUnProperty::CUnProperty", aUn.Name( n ), "unknown" );
	}
}

// --------------------------------------------------------------------
// public:      Constructor #2
// --------------------------------------------------------------------
CUnProperty::CUnProperty () {
    CUnProperty::Cleanup();
}

// --------------------------------------------------------------------
// public:      Copy constructor
// --------------------------------------------------------------------
CUnProperty::CUnProperty ( const CUnProperty & aC ) {
    CUnProperty::Cleanup();
    CUnProperty::Copy( aC );
}

// --------------------------------------------------------------------
// public:      Assignment operator
CUnProperty & CUnProperty::operator = ( const CUnProperty & aC ) {
    CUnProperty::Copy( aC );
	return *this;
}

// --------------------------------------------------------------------
// public:      Destructor
// --------------------------------------------------------------------
CUnProperty::~CUnProperty () {
    CUnProperty::Free();
}

// --------------------------------------------------------------------
// private:     Store data into local data space (updates itsDataLen)
//              and allocates more data as needed
// --------------------------------------------------------------------
void        CUnProperty::Store   ( CUnFile & aUf, dword_t aLen) {
	try {
		if	( ( aLen & 0x80000000 ) == 0 ) {
		    CUnProperty::AddSpace( aLen );
		    aUf.ReadRawData( itsData + itsDataLen, aLen );
		    itsDataLen += aLen;
		}
		else {
			CUnProperty::Store( aLen );
		}
	}
	catch	( CError e ) {
		throw CError( "CUnProperty::Store(CunFile, aLen)", e.Error() );
	}
	catch	( ... ) {
		throw CError( "CUnProperty::Store(CunFile, aLen)", "unknown" );
	}
}

// --------------------------------------------------------------------
void        CUnProperty::Store   ( bool aValue ) {
	try {
	    CUnProperty::AddSpace( sizeof( bool ) );
	    *((bool *)itsData) = aValue;
	    itsDataLen += sizeof( bool );
	}
	catch	( CError e ) {
		throw CError( "CUnProperty::Store(bool)", e.Error() );
	}
	catch	( ... ) {
		throw CError( "CUnProperty::Store(bool)", "unknown" );
	}
}

// --------------------------------------------------------------------
void        CUnProperty::Store   ( byte_t aValue ) {
	try {
	    CUnProperty::AddSpace( sizeof( byte_t ) );
	    *((byte_t *)itsData) = aValue;
	    itsDataLen += sizeof( byte_t );
	}
	catch	( CError e ) {
		throw CError( "CUnProperty::Store(byte_t)", e.Error() );
	}
	catch	( ... ) {
		throw CError( "CUnProperty::Store(byte_t)", "unknown" );
	}
}

// --------------------------------------------------------------------
void        CUnProperty::Store   ( dword_t aValue ) {
	try {
	    CUnProperty::AddSpace( sizeof( dword_t ) );
	    *((dword_t *)itsData) = aValue;
	    itsDataLen += sizeof( dword_t );
	}
	catch	( CError e ) {
		throw CError( "CUnProperty::Store(dword_t)", e.Error() );
	}
	catch	( ... ) {
		throw CError( "CUnProperty::Store(dword_t)", "unknown" );
	}
}

// --------------------------------------------------------------------
void        CUnProperty::Store   ( float aValue ) {
	try {
	    CUnProperty::AddSpace( sizeof( float ) );
	    *((float *)itsData) = aValue;
	    itsDataLen += sizeof( float );
	}
	catch	( CError e ) {
		throw CError( "CUnProperty::Store(float)", e.Error() );
	}
	catch	( ... ) {
		throw CError( "CUnProperty::Store(float)", "unknown" );
	}
}

// --------------------------------------------------------------------
void        CUnProperty::AddSpace( dword_t aLen ) {
    byte_tp original = NULL;
    try {
        original = itsData;
        itsData = new byte_t [ itsDataLen + aLen ];
        if  ( ! itsData ) {
            throw CError( "Out of memory" );
        }
        if	( original ) {
	        ::memcpy( itsData, original, itsDataLen );
	        delete [] original;
		}
        original = NULL;
    }
    catch   ( CError e ) {
        if  ( original ) {
            itsData = original;
        }
        char	numbuf[32];
        ::sprintf( numbuf, "len=%d", aLen );
        throw CError( "CUnProperty::AddSpace", numbuf, e.Error() );
    }
    catch   ( ... ) {
        if  ( original ) {
            itsData = original;
        }
        char	numbuf[32];
        ::sprintf( numbuf, "len=%d", aLen );
        throw CError( "CUnProperty::AddSpace", numbuf, "unknown" );
    }
}

// --------------------------------------------------------------------
// public:      Read different values
// --------------------------------------------------------------------
bool        CUnProperty::GetBool     ( void ) {
    bool    result = false;
    if  ( itsDataLen - itsReadOffs >= sizeof( result ) ) {
        result = *((bool *)(itsData + itsReadOffs));
        itsReadOffs += sizeof( result );
    }
    else {
        itsReadOffs = itsDataLen;
    }
    return  result;
}

// --------------------------------------------------------------------
byte_t      CUnProperty::GetByte     ( void ) {
    byte_t  result = 0;
    if  ( itsDataLen - itsReadOffs >= sizeof( result ) ) {
        result = *((byte_t *)(itsData + itsReadOffs));
        itsReadOffs += sizeof( result );
    }
    else {
        itsReadOffs = itsDataLen;
    }
    return  result;
}

// --------------------------------------------------------------------
word_t      CUnProperty::GetWord     ( void ) {

    word_t  result = 0;

    if  ( itsDataLen - itsReadOffs >= sizeof( result ) ) {
        result = *((word_t *)(itsData + itsReadOffs));
        itsReadOffs += sizeof( result );
    }
    else {
        itsReadOffs = itsDataLen;
    }
    return  result;
}



// --------------------------------------------------------------------
dword_t     CUnProperty::GetDword    ( void ) {
    dword_t result = 0;
    if  ( itsDataLen - itsReadOffs >= sizeof( result ) ) {
        result = *((dword_t *)(itsData + itsReadOffs));
        itsReadOffs += sizeof( result );
    }
    else {
        itsReadOffs = itsDataLen;
    }
    return  result;
}

// --------------------------------------------------------------------
float       CUnProperty::GetFloat    ( void ) {
    float   result = 0;
    if  ( itsDataLen - itsReadOffs >= sizeof( result ) ) {
        result = *((float *)(itsData + itsReadOffs));
        itsReadOffs += sizeof( result );
    }
    else {
        itsReadOffs = itsDataLen;
    }
    return  result;
}

// --------------------------------------------------------------------
// private:     Copy data
// --------------------------------------------------------------------
void        CUnProperty::Copy   ( const CUnProperty & aC ) {
    CUnProperty::Free();
    itsName     = aC.Name();
    itsSize     = aC.Size();
    itsType     = aC.Type();
    itsArray    = aC.Array();
    itsArrIdx   = aC.ArrIdx();
    itsDataLen  = aC.DataLen();
    itsReadOffs = 0;
    itsData     = new byte_t [itsDataLen];
    if  ( ! itsData ) {
        throw CError( "Out of memory" );
    }
    ::memcpy( itsData, aC.Data(), itsDataLen );
}

// --------------------------------------------------------------------
// private:		Set Byte property
// --------------------------------------------------------------------
void	CUnProperty::SetByteProperty	( CUnFile & aUf ) {
	try	{
	    if  ( itsArray ) {
	        itsArrIdx = (byte_t)(aUf.ReadArrIdx());
	    }
	    CUnProperty::Store( aUf.ReadByte() );
	}

	catch	( CError e ) {
		throw CError( "CUnProperty::SetByteProperty", e.Error() );
	}
	catch	( ... ) {
		throw CError( "CUnProperty::SetByteProperty", "unknown" );
	}
}

// --------------------------------------------------------------------
// private:		Set Int property
// --------------------------------------------------------------------
void	CUnProperty::SetIntProperty		( CUnFile & aUf ) {
	try	{
        if  ( itsArray ) {
            itsArrIdx = (byte_t)(aUf.ReadArrIdx());
        }
        CUnProperty::Store( aUf.ReadDword() );
	}

	catch	( CError e ) {
		throw CError( "CUnProperty::SetIntProperty", e.Error() );
	}
	catch	( ... ) {
		throw CError( "CUnProperty::SetIntProperty", "unknown" );
	}
}

// --------------------------------------------------------------------
// private:		Set Bool property
// --------------------------------------------------------------------
void	CUnProperty::SetBoolProperty	( void ) {
	try {
		CUnProperty::Store( itsArray );
		itsArray = false;
	}

	catch	( CError e ) {
		throw CError( "CUnProperty::SetBoolProperty", e.Error() );
	}
	catch	( ... ) {
		throw CError( "CUnProperty::SetBoolProperty", "unknown" );
	}
}
	
// --------------------------------------------------------------------
// private:		Set float property
// --------------------------------------------------------------------
void	CUnProperty::SetFloatProperty	( CUnFile & aUf ) {
	try	{
        if  ( itsArray ) {
            itsArrIdx = (byte_t)(aUf.ReadArrIdx());
        }
        CUnProperty::Store( aUf.ReadFloat() );
	}

	catch	( CError e ) {
		throw CError( "CUnProperty::SetFloatProperty", e.Error() );
	}
	catch	( ... ) {
		throw CError( "CUnProperty::SetFloatProperty", "unknown" );
	}
}

// --------------------------------------------------------------------
// private:		Set object property
// --------------------------------------------------------------------
void	CUnProperty::SetObjectProperty	( CUnFile & aUf ) {
	try	{
        if  ( itsArray ) {
            itsArrIdx = (byte_t)(aUf.ReadArrIdx());
        }
        CUnProperty::Store( aUf.ReadIndex() );
	}

	catch	( CError e ) {
		throw CError( "CUnProperty::SetObjectProperty", e.Error() );
	}
	catch	( ... ) {
		throw CError( "CUnProperty::SetObjectProperty", "unknown" );
	}
}

// --------------------------------------------------------------------
// private:		Set string property
// --------------------------------------------------------------------
void	CUnProperty::SetStringProperty	( CUnFile & aUf ) {
	try	{
        if  ( itsArray ) {
            itsArrIdx = (byte_t)(aUf.ReadArrIdx());
        }
        CUnProperty::Store( aUf, aUf.ReadIndex() );
	}

	catch	( CError e ) {
		throw CError( "CUnProperty::SetStringProperty", itsArray ? "Array" : "Plain",  e.Error() );
	}
	catch	( ... ) {
		throw CError( "CUnProperty::SetStringProperty", itsArray ? "Array" : "Plain", "unknown" );
	}
}

// --------------------------------------------------------------------
// private:		Set asciiz string
// --------------------------------------------------------------------
void	CUnProperty::SetAsciizProperty	( CUnFile & aUf ) {
	try	{
        if  ( itsArray ) {
            itsArrIdx = (byte_t)(aUf.ReadArrIdx());
        }
		// Memorize the place for actual reading
		dword_t	offsetnow = aUf.Offset();
		dword_t	readlen	  = 1;

		// Get bytes until we meet a zero
		while	( aUf.ReadByte() != 0 )	readlen++;

		// Seek it back and go read the string
		aUf.Seek( offsetnow );
        CUnProperty::Store( aUf, readlen );
	}

	catch	( CError e ) {
		throw CError( "CUnProperty::SetStringProperty", itsArray ? "Array" : "Plain",  e.Error() );
	}
	catch	( ... ) {
		throw CError( "CUnProperty::SetStringProperty", itsArray ? "Array" : "Plain", "unknown" );
	}
}

// --------------------------------------------------------------------
// private:		Set struct property
// --------------------------------------------------------------------
void	CUnProperty::SetStructProperty	( CUnFile & aUf, const CUnNameTable & aUn ) {
	try	{
        itsStruc = aUf.ReadIndex();
        if  ( itsArray ) {
            itsArrIdx = (byte_t)(aUf.ReadArrIdx());
        }

        if      ( itsStruc == aUn.Find( "color" ) ) {
            CUnProperty::Store( aUf, 4 );   // R, G, B, A
        }
        else if ( itsStruc == aUn.Find( "vector" ) ) {
            CUnProperty::Store( aUf.ReadFloat() );  // X
            CUnProperty::Store( aUf.ReadFloat() );  // Y
            CUnProperty::Store( aUf.ReadFloat() );  // Z
        }

        else if ( itsStruc == aUn.Find( "rotator" ) ) {
            CUnProperty::Store( aUf.ReadDword() );  // Pitch
            CUnProperty::Store( aUf.ReadDword() );  // Yaw
            CUnProperty::Store( aUf.ReadDword() );  // Roll
        }
        else if ( itsStruc == aUn.Find( "scale" ) ) {
            CUnProperty::Store( aUf.ReadFloat() );  // X
            CUnProperty::Store( aUf.ReadFloat() );  // Y
            CUnProperty::Store( aUf.ReadFloat() );  // Z
            CUnProperty::Store( aUf.ReadDword() );  // sheer rate
            CUnProperty::Store( aUf.ReadByte() );   // sheer axis
        }
        else if ( itsStruc == aUn.Find( "pointregion" ) ) {
            CUnProperty::Store( aUf.ReadIndex() );  // Zone
            CUnProperty::Store( aUf.ReadDword() );  // I Leaf
            CUnProperty::Store( aUf.ReadByte() );   // Zone #
        }

        else {
            throw   CError( "Unsupported property structure", aUn.Name( itsStruc ) );
        }
	}

	catch	( CError e ) {
		throw CError( "CUnProperty::SetStructProperty", e.Error() );
	}
	catch	( ... ) {
		throw CError( "CUnProperty::SetStructProperty", "unknown" );
	}
}

// --------------------------------------------------------------------
// private:		Set array property
// --------------------------------------------------------------------
void	CUnProperty::SetArrayProperty	(	CUnFile & 				aUf,
											const CUnNameTable & 	aUn,
											const CUnImportTable &	aUi,
											const CUnExportTable &	aUe ) {
	try	{
		CUnProperty::Store( aUf.ReadIndex() );	// Number of items in list
	}

	catch	( CError e ) {
		throw CError( "CUnProperty::SetArrayProperty", e.Error() );
	}
	catch	( ... ) {
		throw CError( "CUnProperty::SetArrayProperty", "unknown" );
	}
}

// --------------------------------------------------------------------
// private:		Set unknown property
// --------------------------------------------------------------------
void	CUnProperty::SetUnknownProperty	( 	const CUnNameTable &	aUn,
											CUnFile &				aUf ) {
	char	errstr [1024];
	::sprintf( errstr, "%s [%d] type %d\nFollowing 16 bytes:", aUn.Name( itsName ), itsName, itsType );
	for	( int i = 0; i < 16; i++ ) {
		::sprintf( errstr + ::strlen( errstr ), "%x ", aUf.ReadByte() );
	}
	::strcat( errstr, "\n" );
	
	throw   CError( "Unsupported property", errstr );
}

// --------------------------------------------------------------------
// private:     Free data
// --------------------------------------------------------------------
void        CUnProperty::Free   ( void ) {
    if  ( itsData ) delete [] itsData;
    CUnProperty::Cleanup();
}

// --------------------------------------------------------------------
// private:     Cleenex
// --------------------------------------------------------------------
void        CUnProperty::Cleanup( void ) {
    itsName     = 0;
    itsStruc    = 0;

    itsType     = 0;
    itsSize     = 0;

    itsArray    = false;
    itsArrIdx   = 0;
    itsData     = NULL;
    itsDataLen  = 0;
    itsReadOffs = 0;
}

// --------------------------------------------------------------------
dword_t                 CUnProperty::Name    ( void ) const { return itsName; }
dword_t                 CUnProperty::Struc   ( void ) const { return itsStruc; }
byte_t                  CUnProperty::Type    ( void ) const { return itsType; }
dword_t                 CUnProperty::Size    ( void ) const { return itsSize; }
bool                    CUnProperty::Array   ( void ) const { return itsArray; }
byte_t                  CUnProperty::ArrIdx  ( void ) const { return itsArrIdx; }
const byte_tp           CUnProperty::Data    ( void ) const { return itsData; }
dword_t                 CUnProperty::DataLen ( void ) const { return itsDataLen; }

// --------------------------------------------------------------------
// EOF: CUnProperty.cxx
// --------------------------------------------------------------------
