// --------------------------------------------------------------------
// CUnServerInfo.cxx
// Whatis:	Unreal server info browser
// Authors:	Esko 'Varpu' Ilola	EIL
// History:	EIL	21-APR-2003		Created	this source
// --------------------------------------------------------------------
#include    "CError.hxx"
#include    "CUnServerInfo.hxx"
#include    "CTcpSocket.hxx"

// --------------------------------------------------------------------
// local:		UT2k3 server requests
// --------------------------------------------------------------------
static	const char	myUT2k3_request[3][5] = {
	{ 0x79, 0x00, 0x00, 0x00, 0x00 },
    { 0x79, 0x00, 0x00, 0x00, 0x01 },
	{ 0x79, 0x00, 0x00, 0x00, 0x02 }
};

// --------------------------------------------------------------------
// ##### CUnServerGameInfo
// --------------------------------------------------------------------

// --------------------------------------------------------------------
// public:      Constructor
// --------------------------------------------------------------------
CUnServerGameInfo::CUnServerGameInfo	(	const char *	aOption,
											const char *	aValue )
										:	itsOption( NULL ),
											itsValue( NULL ) {
	itsOption	= ::my_private_strdup( aOption );
	itsValue	= ::my_private_strdup( aValue );
}

// --------------------------------------------------------------------
// public:      Copy constructor and assignment operator
// --------------------------------------------------------------------
CUnServerGameInfo::CUnServerGameInfo				(	const CUnServerGameInfo & aC )
													:	itsOption( NULL ),
														itsValue( NULL ) {
	itsOption	= ::my_private_strdup( aC.Option() );
	itsValue	= ::my_private_strdup( aC.Value() );
}

// --------------------------------------------------------------------
CUnServerGameInfo	& CUnServerGameInfo::operator =	( const CUnServerGameInfo & aC ) {
	if	( itsOption )	delete [] itsOption;
	if	( itsValue )	delete [] itsValue;
	itsOption = itsValue = NULL;
	itsOption	= ::my_private_strdup( aC.Option() );
	itsValue	= ::my_private_strdup( aC.Value() );
	return *this;
}

// --------------------------------------------------------------------
// public:      Destructor
// --------------------------------------------------------------------
CUnServerGameInfo::~CUnServerGameInfo	( ) {
	if	( itsOption )	delete [] itsOption;
	if	( itsValue )	delete [] itsValue;
	itsOption = itsValue = NULL;
}


// --------------------------------------------------------------------
// ##### CUnServerPlayerInfo
// --------------------------------------------------------------------

// --------------------------------------------------------------------
// public:      Constructor
// --------------------------------------------------------------------
CUnServerPlayerInfo::CUnServerPlayerInfo	(	dword_t			aPlayerId,
												const char *	aPlayerName,
												const char *	aPlayerMesh,
												const char *	aPlayerSkin,
												const char *	aPlayerFace,
												dword_t			aPing,
												int				aScore,
												dword_t			aStatID )
											:	itsPlayerId( 0 ),
												itsPlayerName( NULL ),
												itsPlayerMesh( NULL ),
												itsPlayerSkin( NULL ),
												itsPlayerFace( NULL ),
												itsPing( 0 ),
												itsScore( 0 ),
												itsStatID( 0 ) {
	itsPlayerId		= aPlayerId;
	itsPlayerName	= ::my_private_strdup( aPlayerName );
	itsPlayerMesh	= ::my_private_strdup( aPlayerMesh );
	itsPlayerSkin	= ::my_private_strdup( aPlayerSkin );
	itsPlayerFace	= ::my_private_strdup( aPlayerFace );
	itsPing			= aPing;
	itsScore		= aScore;
	itsStatID		= aStatID;
}

// --------------------------------------------------------------------
// public:      Copy constructor and assignment operator
// --------------------------------------------------------------------
CUnServerPlayerInfo::CUnServerPlayerInfo	(	const CUnServerPlayerInfo & aC )
											:	itsPlayerId( 0 ),
												itsPlayerName( NULL ),
												itsPlayerMesh( NULL ),
												itsPlayerSkin( NULL ),
												itsPlayerFace( NULL ),
												itsPing( 0 ),
												itsScore( 0 ),
												itsStatID( 0 ) {
	itsPlayerId		= aC.PlayerId();
	itsPlayerName	= ::my_private_strdup( aC.PlayerName() );
	itsPlayerMesh	= ::my_private_strdup( aC.PlayerMesh() );
	itsPlayerSkin	= ::my_private_strdup( aC.PlayerSkin() );
	itsPlayerFace	= ::my_private_strdup( aC.PlayerFace() );
	itsPing			= aC.Ping();
	itsScore		= aC.Score();
	itsStatID		= aC.StatID();
}

// --------------------------------------------------------------------
CUnServerPlayerInfo	& CUnServerPlayerInfo::operator =	( const CUnServerPlayerInfo & aC ) {
	if	( itsPlayerName )	delete [] itsPlayerName;
	if	( itsPlayerMesh )	delete [] itsPlayerMesh;
	if	( itsPlayerSkin )	delete [] itsPlayerSkin;
	if	( itsPlayerFace )	delete [] itsPlayerFace;
	itsPlayerId		= 0;
	itsPlayerName	= NULL;
	itsPlayerMesh	= NULL;
	itsPlayerSkin	= NULL;
	itsPlayerFace	= NULL;
	itsPing			= 0;
	itsScore		= 0;
	itsStatID		= 0;

	itsPlayerId		= aC.PlayerId();
	itsPlayerName	= ::my_private_strdup( aC.PlayerName() );
	itsPlayerMesh	= ::my_private_strdup( aC.PlayerMesh() );
	itsPlayerSkin	= ::my_private_strdup( aC.PlayerSkin() );
	itsPlayerFace	= ::my_private_strdup( aC.PlayerFace() );
	itsPing			= aC.Ping();
	itsScore		= aC.Score();
	itsStatID		= aC.StatID();

	return	*this;
}

// --------------------------------------------------------------------
// public:      Destructor
// --------------------------------------------------------------------
CUnServerPlayerInfo::~CUnServerPlayerInfo	( ) {
	if	( itsPlayerName )	delete [] itsPlayerName;
	if	( itsPlayerMesh )	delete [] itsPlayerMesh;
	if	( itsPlayerSkin )	delete [] itsPlayerSkin;
	if	( itsPlayerFace )	delete [] itsPlayerFace;
	itsPlayerId		= 0;
	itsPlayerName	= NULL;
	itsPlayerMesh	= NULL;
	itsPlayerSkin	= NULL;
	itsPlayerFace	= NULL;
	itsPing			= 0;
	itsScore		= 0;
	itsStatID		= 0;
}

// --------------------------------------------------------------------
// ##### CUnServerInfo
// --------------------------------------------------------------------

// --------------------------------------------------------------------
// public:      Constructor
// --------------------------------------------------------------------
CUnServerInfo::CUnServerInfo	(	const char *	aIpAddress,
									word_t			aQueryPort,
                                    unreal_server_t	aType ) {
	CUnServerInfo::Cleanup();

	itsIpAddress	= ::my_private_strdup( aIpAddress );
	itsQueryPort	= aQueryPort;
    itsType			= aType;

	CUnServerInfo::Poll();
}

// --------------------------------------------------------------------
// public:      Copy constructor and assignment operator
// --------------------------------------------------------------------
CUnServerInfo::CUnServerInfo				( const CUnServerInfo & aC ) {
	CUnServerInfo::Cleanup();

    itsType				= aC.Type();
	itsIpAddress		= ::my_private_strdup( aC.IpAddress() );
	itsGamePort			= aC.GamePort();
	itsQueryPort		= aC.QueryPort();
	itsServerName		= ::my_private_strdup( aC.ServerName() );
	itsGameMap			= ::my_private_strdup( aC.GameMap() );
	itsMapTitle			= ::my_private_strdup( aC.MapTitle() );
	itsGameType			= ::my_private_strdup( aC.GameType() );
	itsCurrentPlayers	= aC.CurrentPlayers();
	itsMaximumPlayers	= aC.MaximumPlayers();
	itsGameInfo			= aC.GameInfo();
	itsPlayerInfo		= aC.PlayerInfo();
}

// --------------------------------------------------------------------
CUnServerInfo & CUnServerInfo::operator =	( const CUnServerInfo & aC ) {
	CUnServerInfo::Free();

    itsType				= aC.Type();
	itsIpAddress		= ::my_private_strdup( aC.IpAddress() );
	itsGamePort			= aC.GamePort();
	itsQueryPort		= aC.QueryPort();
	itsServerName		= ::my_private_strdup( aC.ServerName() );
	itsGameMap			= ::my_private_strdup( aC.GameMap() );
	itsMapTitle			= ::my_private_strdup( aC.MapTitle() );
	itsGameType			= ::my_private_strdup( aC.GameType() );
	itsCurrentPlayers	= aC.CurrentPlayers();
	itsMaximumPlayers	= aC.MaximumPlayers();
	itsGameInfo			= aC.GameInfo();
	itsPlayerInfo		= aC.PlayerInfo();

	return	*this;
}

// --------------------------------------------------------------------
// public:      Destructor
// --------------------------------------------------------------------
CUnServerInfo::~CUnServerInfo	( ) {
	CUnServerInfo::Free();
}

// --------------------------------------------------------------------
// public:		Refresh server information
// --------------------------------------------------------------------
void	CUnServerInfo::Refresh	( void ) {

	CUnServerInfo::CleanServerInfo		();

    switch	( itsType ) {
		case	unreal_server_2003:
	   	CUnServerInfo::UT2k3BasicServerInfo	();
		CUnServerInfo::UT2k3GameInfo		();
		CUnServerInfo::UT2k3PlayerInfo		();
        break;

		case	unreal_server_ut:
	   	CUnServerInfo::UTBasicServerInfo	( "\\status\\" );
        break;
    }
}

// --------------------------------------------------------------------
// public:		Poll server information
// --------------------------------------------------------------------
void	CUnServerInfo::Poll		( void ) {

	CUnServerInfo::CleanServerInfo		();

    switch	( itsType ) {
		case	unreal_server_2003:
	   	CUnServerInfo::UT2k3BasicServerInfo	();
        break;

		case	unreal_server_ut:
	   	CUnServerInfo::UTBasicServerInfo	( "\\info\\" );
        break;
    }
}

// --------------------------------------------------------------------
// private:		Private helpers
// --------------------------------------------------------------------
void	CUnServerInfo::Free					( void ) {
	if	( itsIpAddress )	delete []	itsIpAddress;
	if	( itsServerName )	delete []	itsServerName;
	if	( itsGameMap )		delete []	itsGameMap;
	if	( itsMapTitle )		delete []	itsMapTitle;
	if	( itsGameType )		delete []	itsGameType;
	CUnServerInfo::Cleanup();
}

// --------------------------------------------------------------------
void	CUnServerInfo::Cleanup				( void ) {
	itsType				= unreal_server_unknown;
	itsIpAddress		= NULL;
	itsQueryPort		= 0;
	itsGamePort			= 0;
	itsServerName		= NULL;
	itsGameMap			= NULL;
	itsMapTitle			= NULL;
	itsGameType			= NULL;
	itsCurrentPlayers	= 0;
	itsMaximumPlayers	= 0;
	itsGameInfo.clear();
	itsPlayerInfo.clear();
}

// --------------------------------------------------------------------
void	CUnServerInfo::CleanServerInfo		( void ) {
	if	( itsServerName )	delete []	itsServerName;
	if	( itsGameMap )		delete []	itsGameMap;
	if	( itsMapTitle )		delete []	itsMapTitle;
	if	( itsGameType )		delete []	itsGameType;

	itsGamePort			= 0;
	itsServerName		= NULL;
	itsGameMap			= NULL;
	itsMapTitle			= NULL;
	itsGameType			= NULL;
	itsCurrentPlayers	= 0;
	itsMaximumPlayers	= 0;
	itsGameInfo.clear();
	itsPlayerInfo.clear();
}

// --------------------------------------------------------------------
void	CUnServerInfo::UT2k3BasicServerInfo	( void ) {
	char	mybuf[2048];

	try	{
		if	( CUnServerInfo::SendRequest( mybuf, sizeof( mybuf ), itsQueryPort,  myUT2k3_request[0], 5 ) ||
			  CUnServerInfo::SendRequest( mybuf, sizeof( mybuf ), (word_t)(itsQueryPort + 9),  myUT2k3_request[0], 5 ) ) {
			// Parse the input data
			char *		p = mybuf;

    	    // First byte must be 0x79
        	if	( *(p++) != 0x79 )	throw 0;
			if	( CUnServerInfo::ParseDword( &p ) != 0 )	throw 0;
			p += 5;
			itsGamePort = (word_t)CUnServerInfo::ParseDword( &p );
			p += 4;
            itsServerName		= CUnServerInfo::ParseString( &p );
            itsGameMap			= CUnServerInfo::ParseString( &p );
            itsGameType			= CUnServerInfo::ParseString( &p );
       		itsCurrentPlayers	= (word_t)CUnServerInfo::ParseDword( &p );
       		itsMaximumPlayers	= (word_t)CUnServerInfo::ParseDword( &p );
    	}
    }
    catch	( ... ) {
		CUnServerInfo::CleanServerInfo();
    }
}

// --------------------------------------------------------------------
void	CUnServerInfo::UTBasicServerInfo	( const char * aQuery ) {
	char	mybuf[2048];
    char *	option	= NULL;
    char *	value	= NULL;

	try	{
		if	( CUnServerInfo::SendRequest( mybuf, sizeof( mybuf ), itsQueryPort, aQuery, ::strlen( aQuery ) ) ) {

			// Parse the input data
			char *		p = mybuf;

			while	( *p ) {
				if	( option ) {
                    if	( ! ::strcmp( option, "final" ) )	break;
                   	delete [] option;
                }
				if	( value )	delete [] value;
                value = option = NULL;

                option	= CUnServerInfo::ParseUtString( &p );
                if	( ! option )	break;

                // If the option is 'final' we are done through
                if	( ! ::strcmp( option, "final" ) )	break;

               	value	= CUnServerInfo::ParseUtString( &p );
                if	( ! value )		break;

                // Parse the different server options here
                if		( ! ::strcmp( option, "hostname" ) )	CUnServerInfo::AllocNewString( &itsServerName, &value );
                else if	( ! ::strcmp( option, "maptitle" ) )	CUnServerInfo::AllocNewString( &itsMapTitle, &value );
                else if	( ! ::strcmp( option, "mapname" ) )		CUnServerInfo::AllocNewString( &itsGameMap, &value );
                else if	( ! ::strcmp( option, "gametype" ) )	CUnServerInfo::AllocNewString( &itsGameType, &value );
                else if	( ! ::strcmp( option, "numplayers" ) )	itsCurrentPlayers	= (word_t)::atol( value );
                else if	( ! ::strcmp( option, "maxplayers" ) )	itsMaximumPlayers	= (word_t)::atol( value );
                else if	( ! ::strcmp( option, "queryid" ) )		;
                else if	( ! ::strcmp( option, "gamename" ) )	;
                else if	( ! ::strcmp( option, "location" ) )	;
                else if	( ! ::strcmp( option, "hostport" ) )	itsGamePort			= (word_t)::atol( value );
                else if	( ! ::strcmp( option, "gamever" ) )		;
                else if	( ! ::strcmp( option, "gamemode" ) )	;
                else if	( ! ::strcmp( option, "minnetver" ) )	;
                else if	( ! ::strcmp( option, "worldlog" ) ) 	;
                else if	( ! ::strcmp( option, "wantworldlog" ) );
                else if	( ! ::strcmp( option, "listenserver" ) );

				// Are we getting a player entry ?
                else if	( ! ::strncmp( option, "player_", 7 ) ) {
                	while	( ! ::strncmp( option, "player_", 7 ) ) {
						dword_t	playerid	= (dword_t)::atol( option + 7 );
                		char *	playername	= value;
                        char *	playermesh	= NULL;
                        char *	playerskin	= NULL;
                        char *	playerface	= NULL;
                        dword_t	ping		= 0;
                        int		score		= 0;
                        dword_t	statid		= 0;

						value = NULL;
						while	( *p ) {
 			                if	( option )	delete [] option;
                            if	( value )	delete [] value;
	        	            value = option = NULL;

		           	        option	= CUnServerInfo::ParseUtString( &p );
							if	( ! option )	break;
                            if	( ! ::strcmp( option, "final" ) )	break;
            		   		value	= CUnServerInfo::ParseUtString( &p );
	                		if	( ! value )		break;

		                    if		( ! ::strncmp( option, "player_", 7 ) )		break;
                            else if	( ! ::strcmp( option, "queryid" ) )			;
                            else if	( ! ::strncmp( option, "frags_", 6 ) )  	score	= (int)::atol( value );
                            else if	( ! ::strncmp( option, "ping_", 5 ) )  		ping	= (dword_t)::atol( value );
							else if	( ! ::strncmp( option, "team_", 5 ) )		;
							else if	( ! ::strncmp( option, "mesh_", 5 ) )		CUnServerInfo::AllocNewString( &playermesh, &value );
							else if	( ! ::strncmp( option, "skin_", 5 ) )		CUnServerInfo::AllocNewString( &playerskin, &value );
							else if	( ! ::strncmp( option, "face_", 5 ) )		CUnServerInfo::AllocNewString( &playerface, &value );
							else if	( ! ::strncmp( option, "ngsecret_", 9 ) )	;

						}

						if	( playername ) {
							CUnServerPlayerInfo	pinfo( playerid, playername, playermesh, playerskin, playerface, ping, score, statid );
    	                    itsPlayerInfo.push_back( pinfo );
        	            }
						if	( playername )	delete [] playername;
						if	( playermesh )	delete [] playermesh;
						if	( playerskin )	delete [] playerskin;
						if	( playerface )	delete [] playerface;
                        if	( ! option )	break;
                    }
				}

				// Other options NOT queryID or similar go to game options
                else {
                    CUnServerGameInfo	ginfo( option, value );
					itsGameInfo.push_back( ginfo );
                }
            }
    	}
    }
    catch	( ... ) {
		CUnServerInfo::CleanServerInfo();
    }
	if	( option )	delete [] option;
	if	( value )	delete [] value;
}

// --------------------------------------------------------------------
void	CUnServerInfo::UT2k3GameInfo		( void ) {
	char	mybuf[2048];
    char *	option	= NULL;
    char *	value	= NULL;
	try	{
	    if	( itsServerName ) {
			if	( CUnServerInfo::SendRequest( mybuf, sizeof( mybuf ), itsQueryPort,  myUT2k3_request[1], 5 ) ) {
				// Parse the input data
				char *		p = mybuf;

            	if	( *(p++) != 0x79 )	throw 0;
				if	( CUnServerInfo::ParseDword( &p ) != 0x01000000 )	throw 0;

                while	( *p ) {
					if	( option )	delete [] option;
					if	( value )	delete [] value;
                    value = option = NULL;
                    option	= CUnServerInfo::ParseString( &p );
                    value	= CUnServerInfo::ParseString( &p );

                    if	( value ) {
                        CUnServerGameInfo	ginfo( option, value );
						itsGameInfo.push_back( ginfo );
                    }
                }
		    }
	    }
    }
    catch ( ... ) {
		itsGameInfo.clear();
    }
	if	( option )	delete [] option;
	if	( value )	delete [] value;
}

// --------------------------------------------------------------------
void	CUnServerInfo::UT2k3PlayerInfo		( void ) {
	char	mybuf[2048];
	dword_t	playerid;
    char *	playername	= NULL;
    dword_t	ping;
	int		score;
    dword_t	statid;

    try {
	    if	( itsGameInfo.size() > 0 ) {
			if	( CUnServerInfo::SendRequest( mybuf, sizeof( mybuf ), itsQueryPort,  myUT2k3_request[2], 5 ) ) {
				// Parse the input data
				char *		p = mybuf;

                if	( *p ) {
	            	if	( *(p++) != 0x79 )	throw 0;
					if	( CUnServerInfo::ParseDword( &p ) != 0x02000000 )	throw 0;

	                playerid = CUnServerInfo::ParseDword( &p );
    	            while	( *p ) {
        	        	if	( playername )	delete [] playername;
            	        playername	= NULL;
                	    playername	= CUnServerInfo::ParseString( &p );
     					ping		= CUnServerInfo::ParseDword( &p );
	                    score		= (int)CUnServerInfo::ParseDword( &p );
    	                statid		= CUnServerInfo::ParseDword( &p );
						if	( playername ) {
							CUnServerPlayerInfo	pinfo( playerid, playername, "", "", "", ping, score, statid );
                	        itsPlayerInfo.push_back( pinfo );
                    	}
						playerid = CUnServerInfo::ParseDword( &p );
                    }
                }
    	    }
	    }
    }
    catch ( ... ) {
		itsPlayerInfo.clear();
    }
	if	( playername )	delete [] playername;
}

// --------------------------------------------------------------------
bool	CUnServerInfo::SendRequest	(   char *			aBuffer,
										size_t			aBufSize,
										word_t			aPort,
										const char *	aMessage,
                                        size_t			aMsgLen ) {
	bool		result = true;
	CTcpSocket	sock( socket_type_datagram );
    size_t		offs = 0;
    int			recv;
    time_t		mytime;

	::memset( aBuffer, 0, aBufSize );

    try	{
        mytime	= ::time( NULL ) + 3;
        sock.Bind( aPort );
        while	( ! sock.Connect( itsIpAddress, aPort ) ) {
        	if	( mytime < ::time( NULL ) ) {
            	throw	CError( "Unable to connect" );
            }
        }

		sock.Write( aMessage, aMsgLen );
        sock.Flush();

        offs	= 0;
        mytime	= ::time( NULL ) + 3;
		do	{
        	recv = sock.Read();
            if		( recv >= 0 ) {
                if	( offs >= aBufSize ) {
	            	throw	CError( "Data overflow" );
                }
		        mytime	= ::time( NULL ) + 1;
                aBuffer[offs++] = (char)recv;
            }
            else if	( recv == - 1 ) {
            }
            else if	( recv == - 2 ) {
            	break;
            }
            else {
            	throw	CError( "Unknown error" );
            }
		}	while	( mytime >= ::time( NULL ) );
    }
    catch	( ... ) {
		result = false;
    }
	if	( offs == 0 )	result = false;
	if	( result ) {
		if	( sock.PeerAddr() ) {
			if	( (sock.PeerAddr())[0] ) {
				if	( ::strcmp( sock.PeerAddr(), itsIpAddress ) ) {
					delete [] itsIpAddress;
					itsIpAddress = ::my_private_strdup( sock.PeerAddr() );
				}
			}
		}
		itsQueryPort = aPort;
	}

    return	result;
}

// --------------------------------------------------------------------
dword_t	CUnServerInfo::ParseDword 	( char ** aS ) {
	char *	s		= *aS;
	dword_t	result	=	((dword_t)s[3] & 0xff) * 0x1000000
    				+	((dword_t)s[2] & 0xff) * 0x10000
    				+	((dword_t)s[1] & 0xff) * 0x100
    				+	((dword_t)s[0] & 0xff);
	*aS = s + 4;
    return	result;
}

// --------------------------------------------------------------------
char *	CUnServerInfo::ParseString	( char ** aS ) {
	char *	s		= *aS;
	char *	result;
    dword_t	sl		= ((dword_t)(*(s++))) & 0xff;

    if	( sl != ::strlen( s ) + 1 )	throw 0;

	result = ::my_private_strdup( s );
	*aS = s + sl;
	return	result;
}

// --------------------------------------------------------------------
char *	CUnServerInfo::ParseUtString	( char ** aS ) {
	char	buf		[1024];
    int		ptr		= 0;
	char *	s		= *aS;

    if	( *s != '\\' )	return NULL;
    s++;
	while	( ( *s != '\\' ) && ( *s ) ) {
    	buf[ptr++] = *(s++);
        if	( ptr >= (int)sizeof( buf ) )	return NULL;
    }
    if	( *s != '\\' )	return NULL;
	buf[ptr] = 0;
    *aS = s;
    return	::my_private_strdup( buf );
}

// --------------------------------------------------------------------
void	CUnServerInfo::AllocNewString	( char ** aT, char ** aS ) {
	if	( *aT )	delete [] (*aT);
	*aT = *aS;
	*aS = NULL;
}

// --------------------------------------------------------------------
const char *	CUnServerGameInfo::Option		( void ) const { return itsOption ? itsOption : ""; }
const char *	CUnServerGameInfo::Value		( void ) const { return itsValue ? itsValue : ""; }

// --------------------------------------------------------------------
dword_t			CUnServerPlayerInfo::PlayerId	( void ) const { return itsPlayerId; }
const char *	CUnServerPlayerInfo::PlayerName	( void ) const { return itsPlayerName ? itsPlayerName : ""; }
const char *	CUnServerPlayerInfo::PlayerMesh	( void ) const { return itsPlayerMesh ? itsPlayerMesh : ""; }
const char *	CUnServerPlayerInfo::PlayerSkin	( void ) const { return itsPlayerSkin ? itsPlayerSkin : ""; }
const char *	CUnServerPlayerInfo::PlayerFace	( void ) const { return itsPlayerFace ? itsPlayerFace : ""; }
dword_t			CUnServerPlayerInfo::Ping		( void ) const { return itsPing; }
int				CUnServerPlayerInfo::Score		( void ) const { return itsScore; }
dword_t			CUnServerPlayerInfo::StatID		( void ) const { return itsStatID; }

// --------------------------------------------------------------------
unreal_server_t					CUnServerInfo::Type				( void ) const { return itsType; }
const char *					CUnServerInfo::IpAddress		( void ) const { return itsIpAddress ? itsIpAddress : ""; }
word_t							CUnServerInfo::GamePort			( void ) const { return itsGamePort; }
word_t							CUnServerInfo::QueryPort		( void ) const { return itsQueryPort; }
const char *					CUnServerInfo::ServerName		( void ) const { return itsServerName ? itsServerName : ""; }
const char *					CUnServerInfo::GameMap			( void ) const { return itsGameMap ? itsGameMap : ""; }
const char *					CUnServerInfo::MapTitle			( void ) const { return itsMapTitle ? itsMapTitle : ""; }
const char *					CUnServerInfo::GameType			( void ) const { return itsGameType ? itsGameType : ""; }
word_t							CUnServerInfo::CurrentPlayers	( void ) const { return itsCurrentPlayers; }
word_t							CUnServerInfo::MaximumPlayers	( void ) const { return itsMaximumPlayers; }
const CUnServerGameInfo_tl &	CUnServerInfo::GameInfo			( void ) const { return itsGameInfo; }
const CUnServerPlayerInfo_tl &	CUnServerInfo::PlayerInfo		( void ) const { return itsPlayerInfo; }

// --------------------------------------------------------------------
// EOF: CUnServerInfo.cpp
// --------------------------------------------------------------------
