// --------------------------------------------------------------------
// CUnUmod.cxx
// Whatis:  UMOD file manipulator
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 24-NOV-2001     Created this source
// --------------------------------------------------------------------
#include    "CError.hxx"
#include    "CIOConvert.hxx"
#include    "CUnUmod.hxx"

// --------------------------------------------------------------------
// Some constants needed here
// --------------------------------------------------------------------
#define	UMOD_MAGIC_NUMBER	0x9FE3C5A3
#define INDEX_SIGN_BIT  	0x80        // Sign bit on first byte
#define INDEX_CONT1_BIT 	0x40        // Continuation bit on first byte
#define INDEX_BYT1_MASK 	0x3f        // Data mast on first byte
#define INDEX_CONT_BIT  	0x80        // Continuation bit on other bytes
#define INDEX_BYTE_MASK 	0x7f        // Data mask on other bytes

// --------------------------------------------------------------------
// public:      Constructor #1
// --------------------------------------------------------------------
CUnUmod::CUnUmod ( const char * aFileName ) {
	itsFileName	= NULL;
	itsStream	= NULL;

	if	( ::stat( aFileName, &itsStat ) ) {
		throw CError( aFileName, ::strerror( errno ) );
	}

#ifdef	WIN32

#ifdef	__BORLANDC__
	if	( ( itsStat.st_mode & S_IFREG ) == 0 ) {
#else
	if	( ( itsStat.st_mode & _S_IFREG ) == 0 ) {
#endif

#else
	if	( ( itsStat.st_mode & __S_IFMT ) != __S_IFREG ) {
#endif
		throw	CError( aFileName, "Not a regular file" );
	}

	itsFileName = ::my_private_strdup( aFileName );
	itsStream	= ::fopen( itsFileName, "rb" );
	if	( ! itsStream ) {
		throw CError( itsFileName, ::strerror( errno ) );
	}
	try	{
		CUnUmod::ReadHeader();
		CUnUmod::ReadDir();
	}
	catch ( ... ) {
		if	( itsStream )	::fclose( itsStream );
		if  ( itsFileName )	delete [] itsFileName;
		itsFileName	= NULL;
		itsStream	= NULL;
		throw;
	}
}

// --------------------------------------------------------------------
// public:      Destructor
// --------------------------------------------------------------------
CUnUmod::~CUnUmod() {
	if	( itsStream )	::fclose( itsStream );
	if  ( itsFileName )	delete [] itsFileName;
}

// --------------------------------------------------------------------
// public:      Extract a file
// --------------------------------------------------------------------
void		CUnUmod::Extract	( 	const char *		aTargetDir,
									const umod_dir_t &	aEntry,
									bool				aFlat ) {
	char	target	[1024];
	char	buff	[8192];
	FILE *	ostrm	= NULL;
	dword_t	offs	= 0;
	dword_t	size;
#ifdef	WIN32

#ifdef	__BORLANDC__
	struct  utimbuf utb;
#else
	struct  _utimbuf utb;
#endif

#else
	struct  utimbuf utb;
#endif

	// ----------------------------------------------------------------
	// Set up the target file name
	// ----------------------------------------------------------------
	::strcpy( target, aTargetDir );
	if	( *target ) {
		if	( target[ ::strlen( target ) - 1 ] != '/' ) {
			::strcat( target, "/" );
		}
	}
	else {
		::strcpy( target, "./" );
	}
	if	( aFlat ) {
		if	( ::strrchr( aEntry.file, '/' ) ) {
			::strcat( target, ::strrchr( aEntry.file, '/' ) + 1 );
		}
		else {
			::strcat( target, aEntry.file );
		}
	}
	else {
		::strcat( target, aEntry.file );
	}

	// ----------------------------------------------------------------
	// Position file and let go
	// ----------------------------------------------------------------
	try	{
		ostrm = ::fopen( target, "wb" );
		if	( ! ostrm ) {
			throw CError( target, ::strerror( errno ) );
		}

		CUnUmod::Seek( aEntry.offs );

		while	( offs < aEntry.size ) {
			if	( aEntry.size - offs > sizeof( buff ) ) {
				size = sizeof( buff );
			}
			else {
				size = aEntry.size - offs;
			}
			CUnUmod::ReadRawData( buff, size );
			::fwrite( buff, size, 1, ostrm );
			if	( ferror( ostrm ) ) {
				throw CError( target, ::strerror( errno ) );
			}
			offs = offs + size;
		}

		if	( ::fclose( ostrm ) ) {
			ostrm = NULL;
			throw CError( target, ::strerror( errno ) );
		}
		else {
			utb.actime  = itsStat.st_mtime;
			utb.modtime = itsStat.st_mtime;
#ifdef	WIN32
	    	::_utime( target, &utb );
#else
	    	::utime( target, &utb );
#endif
	    }
	}

	catch ( ... ) {
		if	( ostrm ) ::fclose( ostrm );
	}
}

// --------------------------------------------------------------------
// private:     Read raw data
// --------------------------------------------------------------------
void        CUnUmod::ReadRawData    ( void * aBuf, size_t aLen ) {
	dword_t bfp = 0;
	size_t  gbl;
	::memset( aBuf, 0, aLen );
	while   ( bfp < aLen ) {
		if  ( bfp + 8192 <= aLen ) {
			gbl = ::fread( ((char *)aBuf) + bfp, 8192, 1, itsStream );
			bfp += 8192;
		}
		else {
			gbl = ::fread( ((char *)aBuf) + bfp, aLen - bfp, 1, itsStream );
			bfp = aLen;
		}
		if      ( ferror( itsStream ) ) throw CError( itsFileName, ::strerror( errno ) );
		else if ( gbl != 1 )            throw CError( itsFileName, "Not enough data" );
	}
}

// --------------------------------------------------------------------
// private:     Read a byte
// --------------------------------------------------------------------
byte_t      CUnUmod::ReadByte   ( void ) {
	byte_t          b = 0;
	CUnUmod::ReadRawData( &b, sizeof(b) );
	return  b;
}

// --------------------------------------------------------------------
// private:     Read a dword
// --------------------------------------------------------------------
dword_t     CUnUmod::ReadDword   ( void ) {
	CIOConvert_t    c;
	dword_t         dw = 0;
	CUnUmod::ReadRawData( &dw, sizeof(dw) );
	return  c.Import( dw );
}

// --------------------------------------------------------------------
// private:     Read an index - special compression used in Un files
// --------------------------------------------------------------------
dword_t     CUnUmod::ReadIndex  ( void ) {
	byte_t          byte;
	dword_t         data;
	dword_t         result;
	word_t          shift   = 6;
	bool            sign;

	// ------------------------------------------------------------
	// Read in the first byte separately
	// ------------------------------------------------------------
	byte = CUnUmod::ReadByte();
	sign = (byte & INDEX_SIGN_BIT) != 0;
	result = byte & INDEX_BYT1_MASK;

	// ------------------------------------------------------------
	// Read in the following bytes as needed
	// ------------------------------------------------------------
	if  ( byte & INDEX_CONT1_BIT ) {
		do  {
			byte = CUnUmod::ReadByte();
			data = ((dword_t)byte) & INDEX_BYTE_MASK;
			data = data << shift;
			result = result | data;
			shift = (word_t)(shift + 7);
		}   while   ( ( byte & INDEX_CONT_BIT ) && ( shift < 32 ) );
	}

	if  ( sign ) {
		result = (dword_t)( -((int4_t)result) );
	}

	return  result;
}

// --------------------------------------------------------------------
// private:		Read a text entry
// --------------------------------------------------------------------
void	CUnUmod::ReadText	(	char *	aBuf,
								size_t	aBufLen ) {
	dword_t	texlen;
	::memset( aBuf, 0, aBufLen );

	texlen = CUnUmod::ReadIndex();

	if	( texlen > aBufLen ) {
		throw CError( itsFileName, "Insane file name length" );
	}

	CUnUmod::ReadRawData( aBuf, texlen );
}

// --------------------------------------------------------------------
// private:     File management - Seek
// --------------------------------------------------------------------
void    CUnUmod::Seek   ( long aOffset ) {
	::fseek( itsStream, aOffset, SEEK_SET );
	if  ( ferror( itsStream ) ) {
		throw CError( itsFileName, ::strerror( errno ) );
	}
}

// --------------------------------------------------------------------
// private:    File management - Offset
// --------------------------------------------------------------------
long    CUnUmod::Offset  ( void ) const {
	return  (long)::ftell( itsStream );
}

// --------------------------------------------------------------------
// private:    Read in the UMOD file header - actually tailer
// --------------------------------------------------------------------
void	CUnUmod::ReadHeader	( void ) {

	// ----------------------------------------------------------------
	// There is a minimum length for an UMOD file - at least one file
	// Using hardwired constants because of platform issues
	// ----------------------------------------------------------------
	if	( itsStat.st_size < ( 20 + 14 ) ) {
		throw CError( itsFileName, "Not enough data" );
	}

	// ----------------------------------------------------------------
	// Set location at the point 20 bytes before the end of the file
	// ----------------------------------------------------------------
	::fseek( itsStream, -20, SEEK_END );
	if  ( ferror( itsStream ) ) {
		throw CError( itsFileName, ::strerror( errno ) );
	}

	// ----------------------------------------------------------------
	// Read each field into the structure - done in this way to make
	// the code platform insensitive
	// ----------------------------------------------------------------
	itsHead.sign	= CUnUmod::ReadDword();
	itsHead.diro	= CUnUmod::ReadDword();
	itsHead.size	= CUnUmod::ReadDword();
	itsHead.vers	= CUnUmod::ReadDword();
	itsHead.csum	= CUnUmod::ReadDword();

	// ----------------------------------------------------------------
	// Sanity check
	// ----------------------------------------------------------------
	if	( itsHead.sign != UMOD_MAGIC_NUMBER ) {
		throw CError( itsFileName, "Not an UMOD file" );
	}
	if	( itsHead.size != (dword_t)(itsStat.st_size) ) {
		throw CError( itsFileName, "Insane file size reported" );
	}
}

// --------------------------------------------------------------------
// private:    Read in the UMOD directory
// --------------------------------------------------------------------
void	CUnUmod::ReadDir	( void ) {
	umod_dir_t	dir;
	dword_t		count, i;

	CUnUmod::Seek( itsHead.diro );

	// First, get the number of directory entries in the UMOD file
	count = CUnUmod::ReadIndex();

	// Then just read them in ...
	for	( i = 0; i < count; i++ ) {
		::memset( &dir, 0, sizeof( dir ) );
		CUnUmod::ReadText( dir.file, sizeof( dir.file ) );
		dir.offs = CUnUmod::ReadDword();
		dir.size = CUnUmod::ReadDword();
		dir.flag = CUnUmod::ReadDword();

		// Sometimes the pathname of the file contains
		// MsDos backslashes .... converting to UNIX stuff
		while	( ::strchr( dir.file, '\\' ) ) {
			*::strchr( dir.file, '\\' ) = '/';
		}

		// Store to our local list
		itsDir.push_back( dir );
	}
}

// --------------------------------------------------------------------
const char *		CUnUmod::FileName	( void ) const { return itsFileName; }
const umod_dir_tl &	CUnUmod::Dir		( void ) const { return itsDir; }
umod_head_t			CUnUmod::Head		( void ) const { return itsHead; }

// --------------------------------------------------------------------
// EOF: CUnUmod.cxx
// --------------------------------------------------------------------
