// --------------------------------------------------------------------
// CUpDoc.cpp
// Whatis:  UnrPack Document class
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 10-FEB-2003     Created this source
// --------------------------------------------------------------------
#include	"CUpDoc.h"
#include	<Registry.hpp>

// --------------------------------------------------------------------
// public:	Constructor and destructor
// --------------------------------------------------------------------
CUpDoc::CUpDoc		() {
	CUpDoc::Cleanup();
}

// --------------------------------------------------------------------
CUpDoc::~CUpDoc		() {
	CUpDoc::Free();
}

// --------------------------------------------------------------------
// public:	Locking + progress for threads
// --------------------------------------------------------------------
void				CUpDoc::Lock			( void ) 		{ itsMutex.Lock(); }
void				CUpDoc::Unlock			( void ) 		{ itsMutex.Unlock(); }
const char *		CUpDoc::ProgressMessage	( void ) const	{ return itsProgressMessage; }
dword_t				CUpDoc::ProgressOffs	( void ) const	{ return itsProgressOffs; }
dword_t				CUpDoc::ProgressSize	( void ) const	{ return itsProgressSize; }
bool				CUpDoc::ProgressDone	( void ) const	{ return itsProgressDone; }
CZipNotification &	CUpDoc::ZipProgress		( void )		{ return itsZipProgress; }

// --------------------------------------------------------------------
// public:	Streaming
// --------------------------------------------------------------------
void	CUpDoc::NewDoc	( const char * aExePath ) {
	CUpDoc::Free();

	// From registry check out for copies of UT and UT2003
    TRegistry *	reg;

	reg = new TRegistry;
	try {
	    reg->RootKey = HKEY_LOCAL_MACHINE;

		if	( reg->OpenKeyReadOnly( "\\SOFTWARE\\Unreal Technology\\Installed Apps\\UT2003" ) ) {
			AnsiString	mypath;
            mypath = reg->ReadString( "Folder" );
            reg->CloseKey();
            CUpDoc::DefaultPaths( mypath.c_str() );

			// Check for existence of UCC.exe
            CUpDoc::RegisterUcc( itsUcc2Path, itsUcc2Exe, mypath.c_str() );
        }

		if	( reg->OpenKeyReadOnly( "\\SOFTWARE\\Unreal Technology\\Installed Apps\\UnrealTournament" ) ) {
			AnsiString	mypath;
            mypath = reg->ReadString( "Folder" );
            reg->CloseKey();
            CUpDoc::DefaultPaths( mypath.c_str() );

			// Check for existence of UCC.exe
            CUpDoc::RegisterUcc( itsUccPath, itsUccExe, mypath.c_str() );
        }
    }

	catch	( ... ) {
    }
	delete reg;

	// Load in the default set (executable name + upp extension)
	// Note that this is not complete set but it allows for quick start
    if	( aExePath ) {
    	struct stat	mystat;
    	char	config	[512];
    	char	drive	[16];
        char	path	[256];
        char	fname	[256];
		::fnsplit( aExePath, drive, path, fname, NULL );
        ::fnmerge( config, drive, path, fname, ".upp" );
		if	( ::stat( config, &mystat ) == 0 ) {
	        CUpDoc::Read( config );
        }
    }

	itsDirty = false;
}

// --------------------------------------------------------------------
// These subdirectories will be automatically tried with paths
// --------------------------------------------------------------------
static const char *	default_path[] = {
    "\\Textures",
    "\\System",
    "\\Maps",
    "\\Music",
    "\\Sounds",
    "\\StaticMeshes",
	"\\Animations",
    NULL
};

// --------------------------------------------------------------------
void	CUpDoc::DefaultPaths( const char * aUtRoot ) {
	char		mypath	[512];
    struct stat	mystat;
    int			tryix;

	for	( tryix = 0; default_path[tryix] != NULL; tryix++ ) {
		::strcpy( mypath, aUtRoot );
        ::strcat( mypath, default_path[tryix] );
		if	( ::stat( mypath, &mystat ) == 0 ) {
			if	( mystat.st_mode & S_IFDIR ) {
	            CUpDoc::PathListInsert( mypath );
	            CUpDoc::PathListSelect( mypath, true );

				// Check out if we have any ini files here
				CUpDoc::RipIniFilePaths( mypath, "UT2003.ini" );
				CUpDoc::RipIniFilePaths( mypath, "Unreal2.ini" );
				CUpDoc::RipIniFilePaths( mypath, "UnrealTournament.ini" );
            }
        }
    }
}

// --------------------------------------------------------------------
void	CUpDoc::Write	( const char * aFile ) const {
	CWinIniFile		ini;
    CWinIniSection	sec;

	// General lists
	sec.Clear();
    sec.Name( "PathList" );
	CUpDoc::PathListWrite	( sec );
    ini.Insert( sec );

	sec.Clear();
    sec.Name( "DenyList" );
	CUpDoc::DenyListWrite	( sec );
    ini.Insert( sec );

	CUpDoc::FileTypeListWrite( ini );

	// Zip settings
	sec.Clear();
    sec.Name( "ZipSettings" );
	sec.Insert( "PathMode", 	itsZipPathMode );
	sec.Insert( "CompressMode",	itsZipCompressMode );
    itsZipComment.Write		( sec, "Comment" );
    ini.Insert( sec );

    // GZip stuff
	sec.Clear();
    sec.Name( "GZipSettings" );
	sec.Insert( "PathMode", 	itsGZipPathMode );
	sec.Insert( "CompressMode",	itsGZipCompressMode );
    itsGZipComment.Write	( sec, "Comment" );
    ini.Insert( sec );

	// UZ Stuff
	sec.Clear();
    sec.Name( "UzSettings" );
	sec.Insert					( "UzMode", itsUzMode );
	itsUzRepository.Write		( sec, "UzRepository" );
	itsUz2Repository.Write		( sec, "Uz2Repository" );
	sec.Insert					( "UzSkipNoPack", itsUzSkipNoPack );
	sec.Insert					( "UzIncremental", itsUzIncremental );
    ini.Insert( sec );

	// FTP Mirroring stuff
	sec.Clear();
    sec.Name( "FtpMirSettings" );
	itsFtpMirIp.Write			( sec, "FtpMirIp" );
    sec.Insert					( "FtpMirPort", itsFtpMirPort );
    sec.Insert					( "FtpMirPasv", itsFtpMirPasv );
	itsFtpMirLogin.Write		( sec, "FtpMirLogin" );
	itsFtpMirPassw.Write		( sec, "FtpMirPassw" );
	itsFtpMirServerRoot.Write	( sec, "FtpMirServerRoot" );
	itsFtpMirLocalRoot.Write	( sec, "FtpMirLocalRoot" );
    sec.Insert					( "FtpMirAdd", itsFtpMirAdd );
    sec.Insert					( "FtpMirDel", itsFtpMirDel );
    sec.Insert					( "FtpMirMod", itsFtpMirMod );
    ini.Insert( sec );

	// Readme
	sec.Clear();
    sec.Name( "ReadmeTemplate" );
    itsReadmeName.Write		( sec, "Name" );
    itsReadmeEmail.Write	( sec, "Email" );
    itsReadmeWebsite.Write	( sec, "Website" );
	sec.Insert( "Naming", itsReadmeNaming );
    itsReadmeFileName.Write	( sec, "FileName" );
	sec.Insert( "Html", itsReadmeHtml );
    itsReadmeText.Write		( sec, "Text" );
    ini.Insert( sec );

	// Input file list is the last one
	CUpDoc::InputListWrite( ini );

    // Finally store the data
    ini.Write( aFile );
}

// --------------------------------------------------------------------
void	CUpDoc::Read	( const char * aFile ) {
	CWinIniFile	ini;

    // Read the file
    ini.Read( aFile );

	// General lists
	CUpDoc::PathListRead	( ini );
	CUpDoc::DenyListRead	( ini );
	CUpDoc::FileTypeListRead( ini );

	// Zip and GZip stuff
	itsZipPathMode			= ini.IntValue( "ZipSettings", "PathMode" );
    itsZipCompressMode		= ini.IntValue( "ZipSettings", "CompressMode" );
    itsZipComment.Read		( ini, "ZipSettings", "Comment" );
	itsGZipPathMode			= ini.IntValue( "GZipSettings", "PathMode" );
    itsGZipCompressMode		= ini.IntValue( "GZipSettings", "CompressMode" );
    itsGZipComment.Read		( ini, "GZipSettings", "Comment" );

    // UZ Stuff
	itsUzMode           	= ini.IntValue( "UzSettings", "UzMode" );
	itsUzRepository.Read    ( ini, "UzSettings", "UzRepository" );
	itsUz2Repository.Read   ( ini, "UzSettings", "Uz2Repository" );
	itsUzSkipNoPack     	= ini.BoolValue( "UzSettings", "UzSkipNoPack" );
	itsUzIncremental    	= ini.BoolValue( "UzSettings", "UzIncremental" );

	// FTP Mirroing stuff
	itsFtpMirIp.Read		( ini, "FtpMirSettings", "FtpMirIp" );
	itsFtpMirPort           = (unsigned short)ini.IntValue( "FtpMirSettings", "FtpMirPort" );
    itsFtpMirPasv           = ini.BoolValue( "FtpMirSettings", "FtpMirPasv" );
	itsFtpMirLogin.Read		( ini, "FtpMirSettings", "FtpMirLogin" );
	itsFtpMirPassw.Read		( ini, "FtpMirSettings", "FtpMirPassw" );
	itsFtpMirServerRoot.Read( ini, "FtpMirSettings", "FtpMirServerRoot" );
	itsFtpMirLocalRoot.Read	( ini, "FtpMirSettings", "FtpMirLocalRoot" );
    itsFtpMirAdd			= ini.BoolValue( "FtpMirSettings", "FtpMirAdd" );
    itsFtpMirDel            = ini.BoolValue( "FtpMirSettings", "FtpMirDel" );
    itsFtpMirMod            = ini.BoolValue( "FtpMirSettings", "FtpMirMod" );

	// Readme
    itsReadmeName.Read		( ini, "ReadmeTemplate", "Name" );
    itsReadmeEmail.Read		( ini, "ReadmeTemplate", "Email" );
    itsReadmeWebsite.Read	( ini, "ReadmeTemplate", "Website" );
	itsReadmeNaming         = ini.IntValue( "ReadmeTemplate", "Naming" );
    itsReadmeFileName.Read	( ini, "ReadmeTemplate", "FileName" );
	itsReadmeHtml			= ini.BoolValue( "ReadmeTemplate", "Html" );
    itsReadmeText.Read		( ini, "ReadmeTemplate", "Text" );

	CUpDoc::InputListRead	( ini );

	// We are no more dirty
    itsDirty	= false;
}

// --------------------------------------------------------------------
// public:	Archiving with UCC
// --------------------------------------------------------------------
const char *		CUpDoc::UccPath				( int aMode ) const {
	if	( aMode )	return itsUcc2Path != "" ? itsUcc2Path.Value() : NULL;
	return itsUccPath != "" ? itsUccPath.Value() : NULL;
}

// --------------------------------------------------------------------
const char *		CUpDoc::UccExecutable		( int aMode ) const {
	if	( aMode )	return itsUcc2Exe != "" ? itsUcc2Exe.Value() : NULL;
	return itsUccExe != "" ? itsUccExe.Value() : NULL;
}

// --------------------------------------------------------------------
// public:	Accessing instance data
// --------------------------------------------------------------------
bool				CUpDoc::Dirty			( void ) const { return	itsDirty; }
CUpFile &     		CUpDoc::ProjectFile		( void )       { return itsProjectFile; }
int					CUpDoc::ZipPathMode		( void ) const { return itsZipPathMode; }
int					CUpDoc::ZipCompressMode	( void ) const { return itsZipCompressMode; }
const char *        CUpDoc::ZipComment		( void ) const { return itsZipComment.Value(); }
int					CUpDoc::GZipPathMode	( void ) const { return itsGZipPathMode; }
int					CUpDoc::GZipCompressMode( void ) const { return itsGZipCompressMode; }
const char *        CUpDoc::GZipComment		( void ) const { return itsGZipComment.Value(); }
const char *		CUpDoc::ReadmeName		( void ) const { return itsReadmeName.Value(); }
const char *		CUpDoc::ReadmeEmail		( void ) const { return itsReadmeEmail.Value(); }
const char *		CUpDoc::ReadmeWebsite	( void ) const { return itsReadmeWebsite.Value(); }
int					CUpDoc::ReadmeNaming	( void ) const { return itsReadmeNaming; }
const char *		CUpDoc::ReadmeFileName	( void ) const { return itsReadmeFileName.Value(); }
bool				CUpDoc::ReadmeHtml		( void ) const { return itsReadmeHtml; }
const char *		CUpDoc::ReadmeText		( void ) const { return itsReadmeText.Value(); }
const char *		CUpDoc::UzRepository	( void ) const { return itsUzRepository.Value(); }
const char *		CUpDoc::Uz2Repository	( void ) const { return itsUz2Repository.Value(); }
int					CUpDoc::UzMode			( void ) const { return itsUzMode; }
bool				CUpDoc::UzSkipNoPack	( void ) const { return itsUzSkipNoPack; }
bool				CUpDoc::UzIncremental	( void ) const { return itsUzIncremental; }
const char *		CUpDoc::FtpMirIp		( void ) const { return itsFtpMirIp.Value(); }
unsigned short		CUpDoc::FtpMirPort		( void ) const { return itsFtpMirPort; }
bool				CUpDoc::FtpMirPasv		( void ) const { return itsFtpMirPasv; }
const char *		CUpDoc::FtpMirLogin		( void ) const { return itsFtpMirLogin.Value(); }
const char *		CUpDoc::FtpMirPassw		( void ) const { return itsFtpMirPassw.Value(); }
const char *		CUpDoc::FtpMirServerRoot( void ) const { return itsFtpMirServerRoot.Value(); }
const char *		CUpDoc::FtpMirLocalRoot	( void ) const { return itsFtpMirLocalRoot.Value(); }
bool				CUpDoc::FtpMirAdd		( void ) const { return itsFtpMirAdd; }
bool				CUpDoc::FtpMirDel		( void ) const { return itsFtpMirDel; }
bool				CUpDoc::FtpMirMod		( void ) const { return itsFtpMirMod; }

// --------------------------------------------------------------------
void	CUpDoc::Dirty			( bool aV )			{ itsDirty = aV; }
void	CUpDoc::ZipPathMode		( int aV )			{ itsDirty = true; itsZipPathMode = aV; }
void	CUpDoc::ZipCompressMode	( int aV )			{ itsDirty = true; itsZipCompressMode = aV; }
void    CUpDoc::ZipComment		( const char * aV )	{ itsDirty = true; itsZipComment = aV; }
void	CUpDoc::GZipPathMode	( int aV )			{ itsDirty = true; itsGZipPathMode = aV; }
void	CUpDoc::GZipCompressMode( int aV )			{ itsDirty = true; itsGZipCompressMode = aV; }
void    CUpDoc::GZipComment		( const char * aV )	{ itsDirty = true; itsGZipComment = aV; }
void	CUpDoc::ReadmeName		( const char * aV )	{ itsDirty = true; itsReadmeName = aV; }
void	CUpDoc::ReadmeEmail		( const char * aV )	{ itsDirty = true; itsReadmeEmail = aV; }
void	CUpDoc::ReadmeWebsite	( const char * aV )	{ itsDirty = true; itsReadmeWebsite = aV; }
void	CUpDoc::ReadmeNaming	( int aV )			{ itsDirty = true; itsReadmeNaming = aV; }
void	CUpDoc::ReadmeFileName	( const char * aV )	{ itsDirty = true; itsReadmeFileName = aV; }
void	CUpDoc::ReadmeHtml		( bool aV )			{ itsDirty = true; itsReadmeHtml = aV; }
void	CUpDoc::ReadmeText		( const char * aV )	{ itsDirty = true; itsReadmeText = aV; }
void	CUpDoc::UzRepository	( const char * aV )	{ itsDirty = true; itsUzRepository = aV; }
void	CUpDoc::Uz2Repository	( const char * aV )	{ itsDirty = true; itsUz2Repository = aV; }
void	CUpDoc::UzMode			( int aV )			{ itsDirty = true; itsUzMode = aV; }
void	CUpDoc::UzSkipNoPack	( bool aV )			{ itsDirty = true; itsUzSkipNoPack = aV; }
void	CUpDoc::UzIncremental	( bool aV )			{ itsDirty = true; itsUzIncremental = aV; }
void	CUpDoc::FtpMirIp		( const char * aV )	{ itsDirty = true; itsFtpMirIp = aV; }
void	CUpDoc::FtpMirPort		( unsigned short aV ){ itsDirty = true; itsFtpMirPort = aV; }
void	CUpDoc::FtpMirPasv		( bool aV )			{ itsDirty = true; itsFtpMirPasv = aV; }
void	CUpDoc::FtpMirLogin		( const char * aV ) { itsDirty = true; itsFtpMirLogin = aV; }
void	CUpDoc::FtpMirPassw		( const char * aV ) { itsDirty = true; itsFtpMirPassw = aV; }
void	CUpDoc::FtpMirServerRoot( const char * aV ) { itsDirty = true; itsFtpMirServerRoot = aV; }
void	CUpDoc::FtpMirLocalRoot	( const char * aV ) { itsDirty = true; itsFtpMirLocalRoot = aV; }
void	CUpDoc::FtpMirAdd		( bool aV )         { itsDirty = true; itsFtpMirAdd = aV; }
void	CUpDoc::FtpMirDel		( bool aV )         { itsDirty = true; itsFtpMirDel = aV; }
void	CUpDoc::FtpMirMod		( bool aV )         { itsDirty = true; itsFtpMirMod = aV; }

// --------------------------------------------------------------------
// private:	Private helpers
// --------------------------------------------------------------------
void	CUpDoc::Free	( void ) {
	CUpDoc::Cleanup();
}

// --------------------------------------------------------------------
void	CUpDoc::Cleanup	( void ) {
	itsDirty			= false;
    itsUccPath			= (const char *)NULL;
    itsUcc2Path			= (const char *)NULL;
    itsUccExe			= (const char *)NULL;
    itsUcc2Exe			= (const char *)NULL;
    itsProjectFile.PathName( NULL );
    itsPathList.clear();
    itsDenyList.clear();
    itsFileTypeList.clear();
    itsInputList.clear();
    itsZipPathMode		= 0;
    itsZipCompressMode	= 0;
    itsZipComment		= (const char *)NULL;
    itsGZipPathMode		= 0;
    itsGZipCompressMode	= 0;
    itsGZipComment		= (const char *)NULL;
    itsUzRepository		= (const char *)NULL;
    itsUz2Repository	= (const char *)NULL;
    itsUzMode			= 0;
    itsUzSkipNoPack		= false;
    itsUzIncremental	= false;

	itsProgressMessage	= "";
	itsProgressOffs		= 0;
    itsProgressSize		= 0;
    itsProgressDone		= false;
	itsMutex.Reset();
    itsArchiveFile.PathName( NULL );

    itsReadmeName		= (const char *)NULL;
    itsReadmeEmail		= (const char *)NULL;
    itsReadmeWebsite	= (const char *)NULL;
    itsReadmeNaming		= 0;
    itsReadmeFileName	= (const char *)NULL;
    itsReadmeHtml		= false;
    itsReadmeText		= (const char *)NULL;

}

// --------------------------------------------------------------------
// private:	Check for ini-file existence and rip off the Paths=...
// --------------------------------------------------------------------
void	CUpDoc::RipIniFilePaths	( const char * aPath, const char * aIniFile ) {
	CWinIniFile		ini;
    CWinIniItem_l	list;
    CWinIniItem_li	loop;
	char			inifile[512];
    struct stat		mystat;

    ::sprintf( inifile, "%s\\%s", aPath, aIniFile );
	if	( ::stat( inifile, &mystat ) != 0 )	return;

	try	{
		ini.Read( inifile );
		list = ini.ItemList( "Core.System", "Paths" );
		for	( loop = list.begin(); loop != list.end(); loop++ ) {
			CUpDoc::RipIniPath( aPath, (*loop).Value() );
        }
    }

    catch	( ... ) {
    }
}

// --------------------------------------------------------------------
// private:	Rip one path
// --------------------------------------------------------------------
void	CUpDoc::RipIniPath	( const char * aPath, const char * aIniPath ) {
	char			inipath[512];
    struct stat		mystat;
    ::strcpy( inipath, aPath );

    while	( ::strncmp( aIniPath, "..", 2 ) == 0 ) {
    	aIniPath = aIniPath + 3;
        if	( ::strrchr( inipath, '\\' ) == NULL )	return;
        *::strrchr( inipath, '\\' ) = 0;
    }

    ::strcat( inipath, "\\" );
    ::strcat( inipath, aIniPath );
    while	( ::strchr( inipath, '/' ) ) {
    	*::strchr( inipath, '/' ) = '\\';
    }

	if	( ::strstr( inipath, "\\*." ) ) {
    	*::strstr( inipath, "\\*." ) = 0;
    }

	if	( ::stat( inipath, &mystat ) == 0 ) {
		if	( mystat.st_mode & S_IFDIR ) {
            CUpDoc::PathListInsert( inipath );
            CUpDoc::PathListSelect( inipath, true );
		}
    }
}

// --------------------------------------------------------------------
// private:	Register the UCC
// --------------------------------------------------------------------
void	CUpDoc::RegisterUcc	( CUpString & aPath, CUpString & aExe, const char * aRoot ) {
	struct stat	mystat;
    char		mypath[1024];

    // Does the root directory exist ?
    if	( ::stat( aRoot, &mystat ) )	return;

    // Make the System directory and chack if it exists
    ::sprintf( mypath, "%s\\System", aRoot );
    if	( ::stat( mypath, &mystat ) )	return;

	// Check if the UCC.exe is present in there
    ::strcat( mypath, "\\UCC.exe" );
    if	( ::stat( mypath, &mystat ) )	return;

    // So, we can register this one
    ::sprintf( mypath, "%s\\System", aRoot );
	aPath.Value( mypath );
    ::strcat( mypath, "\\UCC.exe" );
	aExe.Value( mypath );
}

// --------------------------------------------------------------------
// public:	Check if all files can use uz1 compression
// --------------------------------------------------------------------
bool	CUpDoc::CanUseUz1 ( void ) const {
	CUpInput_lci	fileloop;
    for	( fileloop = itsInputList.begin(); fileloop != itsInputList.end(); fileloop++ ) {
    	if	( (*fileloop).Select() ) {
	    	if	( (*fileloop).FileType().Uz2Only() ) {
            	return false;
            }
        }
	}
	return	true;
}

// --------------------------------------------------------------------
// public:	Check if Uz compression is available
// --------------------------------------------------------------------
bool	CUpDoc::UzAvailable ( void ) const {
	if	( itsUzMode == 0 ) {
		if	( itsUzRepository == "" )	return false;
		if	( itsUccExe == "" )			return false;
		return   CUpDoc::CanUseUz1();
    }
	else {
		if	( itsUz2Repository == "" )	return false;
		if	( itsUcc2Exe == "" )		return false;
		return	true;
    }
}

// --------------------------------------------------------------------
// EOF:	CUpDoc.cpp
// --------------------------------------------------------------------
