// --------------------------------------------------------------------
// CGzip.cxx
// Whatis:	Class for comressing/decompressing
// Authors:	Esko 'Varpu' Ilola	EIL
// History:	EIL	02-JUL-2002		Created	this source
// --------------------------------------------------------------------
#include	"CError.hxx"
#include	"CGzip.hxx"
#include	<zlib.h>

// --------------------------------------------------------------------
// public:		Constructor
// --------------------------------------------------------------------
CGzip::CGzip () {
}

// --------------------------------------------------------------------
// public:		Destructor
// --------------------------------------------------------------------
CGzip::~CGzip	() {
}

// --------------------------------------------------------------------
// public:		Make a GZip file
// --------------------------------------------------------------------
dword_t	CGzip::Gzip	( 	const char *	aGzFile,
						const char *	aDataFile ) {
	gzFile			zstrm	= NULL;
	FILE *			dstrm	= NULL;
	char			buff	[8192];
	int				err;
	struct stat		mystat;
	size_t			offs;
	size_t			size;

	try {

		// ------------------------------------------------------------
		// Get input file properties - especially the size of it
		// ------------------------------------------------------------
		if	( ::stat( aDataFile, &mystat ) ) {
			CGzip::Zerror( Z_ERRNO, aDataFile );
		}

		// ------------------------------------------------------------
		// Open the data file
		// ------------------------------------------------------------
		dstrm = ::fopen( aDataFile, "rb" );
		if	( ! dstrm ) {
			CGzip::Zerror( Z_ERRNO, aDataFile );
		}

		// ------------------------------------------------------------
		// Open the gzip file
		// ------------------------------------------------------------
		zstrm = ::gzopen( aGzFile, "wb9" );
		if	( ! zstrm ) {
			if	( errno )	CGzip::Zerror( Z_ERRNO, aGzFile );
			else			CGzip::Zerror( Z_MEM_ERROR, aGzFile );
		}

		// ------------------------------------------------------------
		// Loop through the input file
		// ------------------------------------------------------------
		offs = 0;
		while	( offs < (size_t)(mystat.st_size) ) {

			if	( mystat.st_size - offs > sizeof( buff ) ) {
				size = sizeof( buff );
			}
			else {
				size = mystat.st_size - offs;
			}

			::fread( buff, size, 1, dstrm );
			if	( ferror( dstrm ) ) {
				CGzip::Zerror( Z_ERRNO, aDataFile );
			}

			if	( ! ::gzwrite( zstrm, buff, size ) ) {
				::gzerror( zstrm, &err );
				CGzip::Zerror( err, aGzFile );
			}

			offs = offs + size;
		}

		// ------------------------------------------------------------
		// Closing the files
		// ------------------------------------------------------------
		err = ::gzclose( zstrm );
		zstrm = NULL;
		CGzip::Zerror( err, aGzFile );

		err = ::fclose( dstrm );
		dstrm = NULL;
		if	( err ) {
			throw CError( aDataFile, ::strerror( errno ) );
		}
	}

	catch ( ... ) {
		if	( zstrm ) {
			::gzclose( zstrm );
		}

		if	( dstrm ) {
			::fclose( dstrm );
		}

		::unlink( aGzFile );
		throw;
	}

	::stat( aGzFile, &mystat );
	return	mystat.st_size;
}

// --------------------------------------------------------------------
// public:		Decompress a GZip file
// --------------------------------------------------------------------
void	CGzip::Gunzip	(	const char *	aDataFile,
							const char *	aGzFile ) {
	gzFile			zstrm	= NULL;
	FILE *			dstrm	= NULL;
	char			buff	[8192];
	int				err;
	int				size;

	try {

		// ------------------------------------------------------------
		// Open (create) the data file
		// ------------------------------------------------------------
		dstrm = ::fopen( aDataFile, "wb" );
		if	( ! dstrm ) {
			CGzip::Zerror( Z_ERRNO, aDataFile );
		}

		// ------------------------------------------------------------
		// Open the gzip file
		// ------------------------------------------------------------
		zstrm = ::gzopen( aGzFile, "rb" );
		if	( ! zstrm ) {
			if	( errno )	CGzip::Zerror( Z_ERRNO, aGzFile );
			else			CGzip::Zerror( Z_MEM_ERROR, aGzFile );
		}

		// ------------------------------------------------------------
		// Loop through the input file
		// ------------------------------------------------------------
		while	( ! ::gzeof( zstrm ) ) {

			size = ::gzread( zstrm, buff, sizeof( buff ) );

			if		( size < 0 ) {
				::gzerror( zstrm, &err );
				CGzip::Zerror( err, aGzFile );
			}
			else if	( size > 0 ) {
				::fwrite( buff, size, 1, dstrm );
				if	( ferror( dstrm ) ) {
					CGzip::Zerror( Z_ERRNO, aDataFile );
				}
			}
		}

		// ------------------------------------------------------------
		// Closing the files
		// ------------------------------------------------------------
		err = ::gzclose( zstrm );
		zstrm = NULL;
		CGzip::Zerror( err, aGzFile );

		err = ::fclose( dstrm );
		dstrm = NULL;
		if	( err ) {
			throw CError( aDataFile, ::strerror( errno ) );
		}
	}

	catch ( ... ) {
		if	( zstrm ) {
			::gzclose( zstrm );
		}

		if	( dstrm ) {
			::fclose( dstrm );
		}

		::unlink( aDataFile );
		throw;
	}
}

// --------------------------------------------------------------------
// private:		Check for zlib errors
// --------------------------------------------------------------------
void	CGzip::Zerror	(	int				aZerrno,
							const char *	aFileName ) {

	if		( aZerrno == Z_ERRNO ) {
		throw CError( aFileName, ::strerror( errno ) );
	}
	else if	( aZerrno != Z_OK ) {
		throw CError( aFileName, zError( aZerrno ) );
	}
}

// --------------------------------------------------------------------
// EOF:	CGzip.cxx
// --------------------------------------------------------------------
