// --------------------------------------------------------------------
// CImportFile.cxx
// Whatis:	Class for importing files to the system
// Authors:	Esko 'Varpu' Ilola	EIL
// History:	EIL	02-JUL-2002		Created	this source
// --------------------------------------------------------------------
#include    "CError.hxx"
#include	"CUnUmod.hxx"
#include	"CGzip.hxx"
#include	"CStorage.hxx"
#include	"CTableUser.hxx"
#include	"CTableAuth.hxx"
#include	"CTablePack.hxx"
#include	"CTableXref.hxx"
#include	"CTablePackFile.hxx"
#include	"CIdentifyFile.hxx"
#include	"CUnUtils.hxx"
#include	"CImportFile.hxx"

// --------------------------------------------------------------------
// public:		Constructor
// --------------------------------------------------------------------
CImportFile::CImportFile ( const char * aUser, bool aSilent ) {
	CMySqlConnect	db( "quest", "", "UTCMS" );
	CMySqlWhere		w;
	CMySqlQuote		q;
	CTableUser		user;
	data_user_tl	list;

	w << "user_logn='" << q.Quote( aUser ) << "'";
	list = user.Select( db, w );
	
	if	( list.size() < 1 ) {
		throw CError( aUser, "User not found" );
	}

	itsUser		= *(list.begin());
	itsSilent	= aSilent;
}

// --------------------------------------------------------------------
// public:		Destructor
// --------------------------------------------------------------------
CImportFile::~CImportFile	() {
}

// --------------------------------------------------------------------
// public:		Import
// --------------------------------------------------------------------
void	CImportFile::Import		(	const char *	aFilePath,
									const char *	aTempDir ) {
	dir_file_tl	list;

	if	( ! itsSilent )	::printf( "Importing: %s\n", aFilePath );

	itsPackList.clear();
	itsTempDir	= aTempDir != NULL ? aTempDir : ".";

	CImportFile::ImportStuf( list, aFilePath );
	if	( list.size() > 0 ) {
		char	pack[1024];

		::my_extract_file( pack, sizeof( pack ), aFilePath );
		::my_strip_suffix( pack );

		CImportFile::ImportPack( pack, list );
	}
}

// --------------------------------------------------------------------
// private:		Import package according to the file list
// --------------------------------------------------------------------
void	CImportFile::ImportPack	(	const char *	aPack,
									dir_file_tl &	aList ) {

	CMySqlConnect	db( "quest", "", "UTCMS" );
	CMySqlWhere		w;
	CMySqlQuote		q;

	CStorage		storage;

	CTableFile		file;
	CTablePack		pack;
	CTableXref		xref;
	CTablePackFile	packfile;

	data_file_t		filedata;
	data_file_tl	filelist;
	data_file_tli	fileloop;

	data_pack_t		packdata;
	data_pack_tl	packlist;
	data_pack_tli	packloop;

	data_packfile_t		packfiledata;
	data_packfile_tl	packfilelist;
	data_packfile_tli	packfileloop;

	dir_file_tli	dirloop;
	char			storename[1024];
	dword_t			version = 0;

	if	( ! itsSilent )	::printf( "\tpack: %s\n", aPack );

	// ----------------------------------------------------------------
	// Loop through all the files inserting them into the system
	// Collect the inserted file records into local list
	// Also inserts/updates authors (if detected and missing) to user
	// ----------------------------------------------------------------
	for	( dirloop = aList.begin(); dirloop != aList.end(); dirloop++ ) {

		try {

			if	( ! itsSilent )	::printf( "\t\tfile: %s\n", (*dirloop).name );

			CIdentifyFile	identify( (*dirloop).name );

			::my_extract_file( storename, sizeof( storename ), (*dirloop).name );
			filedata = storage.Save( storename, 0, (*dirloop).name );

			// ------------------------------------------------------------
			// Should we do the import/export xref thing ?
			// ------------------------------------------------------------
			if	( ! ::strcmp( identify.Typ1(), "UNR" ) ) {
				w = "";
				w << "xref_file=" << filedata.file_idnt;
				if	( xref.Count( db, w ) < 1 ) {
					CImportFile::CreateXref( db, filedata.file_idnt, (*dirloop).name );
				}
			}
	
			// ------------------------------------------------------------
			// File types
			// ------------------------------------------------------------
			filedata.file_type = identify.Type().type_idnt;
			::my_strfit( filedata.file_typ1, sizeof( filedata.file_typ1 ), identify.Typ1() );
			::my_strfit( filedata.file_typ2, sizeof( filedata.file_typ2 ), identify.Typ2() );
			::my_strfit( filedata.file_ctgr, sizeof( filedata.file_ctgr ), identify.Ctgr() );
	
			if	( ! itsSilent )	::printf( 	"\t\t\ttype: %d:%s:%s\n",
											filedata.file_type,
											filedata.file_typ1,
											filedata.file_typ2 );
	
			// ------------------------------------------------------------
			// Author specified ?
			// ------------------------------------------------------------
			if	( filedata.file_auth == 0 ) {
				if	( *(identify.Author()) != 0 ) {
		
					if	( ! itsSilent )	::printf( 	"\t\t\tauth: %s\n",
													identify.Author() );
		
					filedata.file_auth = identify.AuthorId();
				}
			}
	
			// ------------------------------------------------------------
			// Set the owner for the file
			// ------------------------------------------------------------
			if	( filedata.file_user == 0 ) {
				filedata.file_user = itsUser.user_idnt;
			}
	
			file.Update( db, &filedata );
			filelist.push_back( filedata );
		}
		catch ( ... ) {
		}
	}

	// ----------------------------------------------------------------
	// Create the package record
	// ----------------------------------------------------------------
	::memset( &packdata, 0, sizeof( packdata ) );

	packdata.pack_atim = ::time( NULL );
	packdata.pack_mtim = ::time( NULL );
	packdata.pack_ltim = ::time( NULL );
	packdata.pack_user = itsUser.user_idnt;
	::my_strfit( packdata.pack_file, sizeof( packdata.pack_file ) - 5, aPack );
	::strcpy( packdata.pack_name, packdata.pack_file );

	// ----------------------------------------------------------------
	// Check if we already have an identical package around
	// ----------------------------------------------------------------
	w = "";
	w << "pack_file like '" << q.Quote( packdata.pack_file ) << "%'";
	packlist = pack.Select( db, w );
	if	( packlist.size() > 0 ) {
		for	( packloop = packlist.begin(); packloop != packlist.end(); packloop++ ) {
			w = "";
			w << "packfile_pack=" << (*packloop).pack_idnt;
			packfilelist = packfile.Select( db, w );
			if	( packfilelist.size() == filelist.size() ) {
				for	( packfileloop = packfilelist.begin(); packfileloop != packfilelist.end(); packfileloop++ ) {
					for	( fileloop = filelist.begin(); fileloop != filelist.end(); fileloop++ ) {
						if	( (*fileloop).file_idnt == (*packfileloop).packfile_file )	break;
					}
					if	( fileloop == filelist.end() )	break;
				}
				if	( packfileloop == packfilelist.end() ) {
					if	( ! itsSilent )	::printf( "\tpack: %s Already imported - skipped.\n", aPack );
					return;
				}
			}
		}
	}

	// ----------------------------------------------------------------
	// Same user is not allowed to have two packages with identical names
	// therefore we have to check this out.
	// If a similar package already exist, we append a serial number
	// to the package name: pack[1], pack[2] and so on
	// ----------------------------------------------------------------
	do {
		::my_strfit( packdata.pack_name, sizeof( packdata.pack_name ) - 5, aPack ); 
		if	( version ) {
			::sprintf( packdata.pack_name + ::strlen( packdata.pack_name ), "[%d]", version );
		}
		version++;

		w = "";
		w << "pack_name='" << q.Quote( packdata.pack_name ) << "'";
		packlist = pack.Select( db, w );

	}	while	( packlist.size() > 0 );

	if	( ! itsSilent )	::printf( "\tnewp: %s\n", packdata.pack_name );

	// ----------------------------------------------------------------
	// Insert the package itself into the database
	// ----------------------------------------------------------------
	packdata.pack_idnt = pack.NextIdnt( db );
	pack.Insert( db, &packdata );
	itsPackList.push_back( packdata );

	// ----------------------------------------------------------------
	// Create the packfile records
	// ----------------------------------------------------------------
	for	( fileloop = filelist.begin(); fileloop != filelist.end(); fileloop++ ) {
		packfiledata.packfile_file = (*fileloop).file_idnt;
		packfiledata.packfile_pack = packdata.pack_idnt;

		// We MUST check for existing packfile record here
		w = "";
		w << "packfile_file=" << packfiledata.packfile_file << " and "; 
		w << "packfile_pack=" << packfiledata.packfile_pack;
		if	( packfile.Count( db, w ) == 0 ) {
			packfile.Insert( db, &packfiledata );
		}
	}
}

// --------------------------------------------------------------------
// private:		Import a directory
// --------------------------------------------------------------------
void	CImportFile::ImportDir	( 	dir_file_tl	&	aList,
									const char * 	aDir ) {
	dir_file_tl		list = ::my_private_dir( aDir, true );
	dir_file_tli	loop;

	if	( ! itsSilent )	::printf( "Directory: %s\n", aDir );

	for	( loop = list.begin(); loop != list.end(); loop++ ) {
		CImportFile::ImportStuf( aList, (*loop).name );
	}
}

// --------------------------------------------------------------------
// private:		Import an UMOD file
// --------------------------------------------------------------------
void	CImportFile::ImportUmod	( 	CIdentifyFile &	aIdent ) {
	CUnUmod			umod( aIdent.FileName() );
	umod_dir_tlci	loop;
	char			temp[1024];

	::my_make_tempdir( temp, itsTempDir );

	if	( ! itsSilent )	::printf( "Umod: %s\n", aIdent.FileName() );

	try {
		for	( loop = umod.Dir().begin(); loop != umod.Dir().end(); loop++ ) {
			umod.Extract( temp, *loop, true );
		}

		dir_file_tl	list;
		CImportFile::ImportDir( list, temp );
		if	( list.size() > 0 ) {
			CImportFile::ImportPack( aIdent.Name(), list );
		}

		::my_erase_tempdir( temp );
	}

	catch	( ... ) {
		::my_erase_tempdir( temp );
		throw;
	}
}

// --------------------------------------------------------------------
// private:		Import a GZIP file
// --------------------------------------------------------------------
void	CImportFile::ImportGzip	( 	CIdentifyFile &	aIdent ) {
	CGzip			gzip;
	char			temp[1024];
	char			file[1024];

	::my_make_tempdir( temp, itsTempDir );

	if	( ! itsSilent )	::printf( "Gzip: %s\n", aIdent.FileName() );

	try {
		::strcpy( file, temp );
		::strcat( file, "/" );
		::strcat( file, aIdent.Name() );

		gzip.Gunzip( file, aIdent.FileName() );

		dir_file_tl	list;
		CImportFile::ImportDir( list, temp );
		if	( list.size() > 0 ) {
			CImportFile::ImportPack( aIdent.Name(), list );
		}

		::my_erase_tempdir( temp );
	}

	catch	( ... ) {
		::my_erase_tempdir( temp );
		throw;
	}
}

// --------------------------------------------------------------------
// private:		Import a ZIP file
// --------------------------------------------------------------------
void	CImportFile::ImportZip	( 	CIdentifyFile &	aIdent ) {
	char			cmnd[1024];
	char			temp[1024];

	::my_make_tempdir( temp, itsTempDir );

	if	( ! itsSilent )	::printf( "Zip: %s\n", aIdent.FileName() );

	try {

		::sprintf( cmnd, "unzip -ojqq \"%s\" -d %s", aIdent.FileName(), temp );
	    ::system( cmnd );

		dir_file_tl	list;
		CImportFile::ImportDir( list, temp );
		if	( list.size() > 0 ) {
			CImportFile::ImportPack( aIdent.Name(), list );
		}

		::my_erase_tempdir( temp );
	}

	catch	( ... ) {
		::my_erase_tempdir( temp );
		throw;
	}
}

// --------------------------------------------------------------------
// private:		Import a TAR file
// --------------------------------------------------------------------
void	CImportFile::ImportTar	( 	CIdentifyFile &	aIdent ) {
	char			cmnd[1024];
	char			temp[1024];

	::my_make_tempdir( temp, itsTempDir );

	if	( ! itsSilent )	::printf( "Tar: %s\n", aIdent.FileName() );

	try {

		::sprintf( cmnd, "tar xCf %s \"%s\"", temp, aIdent.FileName() );
	    ::system( cmnd );

		dir_file_tl	list;
		CImportFile::ImportDir( list, temp );
		if	( list.size() > 0 ) {
			CImportFile::ImportPack( aIdent.Name(), list );
		}

		::my_erase_tempdir( temp );
	}

	catch	( ... ) {
		::my_erase_tempdir( temp );
		throw;
	}
}

// --------------------------------------------------------------------
// private:		Import something
// --------------------------------------------------------------------
void	CImportFile::ImportStuf	( 	dir_file_tl &	aList,
									const char *	aFile ) {
	struct stat	mystat;
	if	( ::stat( aFile, &mystat ) ) {
		throw CError( aFile, ::strerror( errno ) );
	}

	// ----------------------------------------------------------------
	// Directory - we recurse into it
	// ----------------------------------------------------------------
#ifdef	WIN32
	if		( ( mystat.st_mode & _S_IFDIR ) != 0 ) {
#else
	if		( ( mystat.st_mode & __S_IFMT ) == __S_IFDIR ) {
#endif
		CImportFile::ImportDir( aList, aFile );
	}

	// ----------------------------------------------------------------
	// Ordinary file 
	// ----------------------------------------------------------------
#ifdef	WIN32
	else if	( ( mystat.st_mode & _S_IFREG ) != 0 ) {
#else
	else if	( ( mystat.st_mode & __S_IFMT ) == __S_IFREG ) {
#endif
		CIdentifyFile	ident( aFile );
		const char *	upk = ident.UnpackMethod();

		// ------------------------------------------------------------
		// Unpacking ?
		// ------------------------------------------------------------
		if		( ! ::strcmp( upk, "gzip" ) ) {
			CImportFile::ImportGzip( ident );
		}
		else if	( ! ::strcmp( upk, "umod" ) ) {
			CImportFile::ImportUmod( ident );
		}
		else if	( ! ::strcmp( upk, "ut2mod" ) ) {
			CImportFile::ImportUmod( ident );
		}
		else if	( ! ::strcmp( upk, "zip" ) ) {
			CImportFile::ImportZip( ident );
		}
		else if	( ! ::strcmp( upk, "tar" ) ) {
			CImportFile::ImportTar( ident );
		}
		else if	( ! ::strcmp( upk, "none" ) ) {
			dir_file_t	item;
			::memset( &item, 0, sizeof( item ) );
			::my_strfit( item.name, sizeof( item.name ), aFile );
			aList.push_back( item );
		}
		else if	( ! ::strcmp( upk, "disp" ) ) {
			if	( ! itsSilent )	::printf( "File: %s Disposed\n", aFile );
		}
		else {
			throw CError( "Don't know how to unpack", aFile, upk );
		}
	}
}

// --------------------------------------------------------------------
// private:		Import something
// --------------------------------------------------------------------
void	CImportFile::CreateXref	( 	CMySqlConnect &	aDb,
									dword_t			aFileIdnt,
									const char *	aFile ) {
	try	{
		CUnUtils		util( aFile );
		unr_class_tl	classlist = util.ClassList();
		unr_class_tli	cloop;
		unr_package_tl	importlist= util.ImportList();
		unr_package_tli	ploop;
		CTableXref		xref;
		data_xref_t		xrefdata;
		data_xref_tl	xreflist;

		// ------------------------------------------------------------
		// First we make the export list
		// ------------------------------------------------------------
		for	( cloop = classlist.begin(); cloop != classlist.end(); cloop++ ) {
			::memset( &xrefdata, 0, sizeof( xrefdata ) );
			xrefdata.xref_file = aFileIdnt;
			xrefdata.xref_expo = 1;
			::strcpy( xrefdata.xref_name, (*cloop).name );
			xrefdata.xref_refs = (*cloop).refcount;
			xref.Insert( aDb, &xrefdata );
		}

		// ------------------------------------------------------------
		// And then the import list
		// ------------------------------------------------------------
		for	( ploop = importlist.begin(); ploop != importlist.end(); ploop++ ) {
			::memset( &xrefdata, 0, sizeof( xrefdata ) );
			xrefdata.xref_file = aFileIdnt;
			xrefdata.xref_expo = 0;
			::strcpy( xrefdata.xref_name, (*ploop).name );
			xrefdata.xref_refs = 1;
			xref.Insert( aDb, &xrefdata );
		}
	}

	catch	( ... ) {
	}
}

// --------------------------------------------------------------------
// EOF:	CImportFile.cxx
// --------------------------------------------------------------------
