// --------------------------------------------------------------------
// CStorage.cxx
// Whatis:	Class for manipulating the file	storage
// Authors:	Esko 'Varpu' Ilola	EIL
// History:	EIL	02-JUL-2002		Created	this source
// --------------------------------------------------------------------
#include	"CError.hxx"
#include	"CGzip.hxx"
#include	"CTableXref.hxx"
#include	"CTableDeny.hxx"
#include	"CTablePref.hxx"
#include	"CTablePackFile.hxx"
#include	"CTablePack.hxx"
#include	"CStorage.hxx"

// --------------------------------------------------------------------
// public:		Constructor
// --------------------------------------------------------------------
CStorage::CStorage () {
	itsDb = NULL;
	itsDb =	new	CMySqlConnect( "quest",	"",	"UTCMS"	);
}

// --------------------------------------------------------------------
// public:		Destructor
// --------------------------------------------------------------------
CStorage::~CStorage	() {
	if	( itsDb	)	delete itsDb;
}

// --------------------------------------------------------------------
// public:		Save a file	to the system
// --------------------------------------------------------------------
data_file_t		CStorage::Save	(	const char *	aStorName,
									dword_t			aFlags,
									const char *	aDataFile ) {
	data_file_t		record;
	CTableFile		file;
	CTableDeny		deny;
	CMySqlWhere		w;
	CMySqlQuote		q;
	data_file_tl	list;
	data_file_tli	loop;

	// ----------------------------------------------------------------
	// Denied files get banned right away
	// ----------------------------------------------------------------
	w << "deny_name='" << q.Quote( aStorName ) << "'";
	if	( deny.Count( *itsDb, w ) > 0 ) {
		throw CError( "Denied", aStorName );
	}

	// ----------------------------------------------------------------
	// Get some information about the file we are going to save
	// ----------------------------------------------------------------
	::memset( &record, 0, sizeof( record ) );
	CStorage::FileInfo( record, aDataFile );

	// ----------------------------------------------------------------
	// Fill in file name and suffix (lowercased)
	// ----------------------------------------------------------------
	::my_strfit( record.file_name, sizeof( record.file_name ), aStorName );
	::my_strfix( record.file_name );
	char *	suff = ::strrchr( record.file_name, '.' );
	if	( suff ) {
		*suff = 0;
		::my_strfit( record.file_suff, sizeof( record.file_suff ), suff + 1 );
		::my_strlower( record.file_suff );
	}

	record.file_flag = aFlags;

	// ----------------------------------------------------------------
	// Find out if this file name is already taken
	// ----------------------------------------------------------------
	list = CStorage::Find( record.file_name, record.file_suff );

	// ----------------------------------------------------------------
	// Loop them through for a checksum match
	// If this happens, then the file is already in the system
	// and we just return the correct record
	// Means: The name and the checksum match
	// ----------------------------------------------------------------
	for	( loop = list.begin(); loop != list.end(); loop++ ) {
		if	( ! ::strcmp( record.file_csum, (*loop).file_csum ) ) {
			record = *loop;
			return	record;
		}
	}

	// ----------------------------------------------------------------
	// Do we have a deny write problem here ?
	// ----------------------------------------------------------------
	for	( loop = list.begin(); loop != list.end(); loop++ ) {
		if	( (*loop).file_flag & FLAG_DENY_WRITE ) {
			throw CError( "Access denied", record.file_name );
		}
	}

	// ----------------------------------------------------------------
	// Find out if we have a file with similar checksum
	// If this happens we just make another link to that file
	// We do NOT pack the file again
	// ----------------------------------------------------------------
	w = "";
	w << "file_csum='" << record.file_csum << "'";
	list = file.Select( *itsDb, w );
	if	( list.size() > 0 ) {
		CStorage::SaveParm( record );
		::strcpy( record.file_data, (*(list.begin())).file_data );
		record.file_idnt = file.NextIdnt( *itsDb );
		record.file_psiz = (*(list.begin())).file_psiz;
		file.Insert( *itsDb, &record );
		return record;
	}

	// ----------------------------------------------------------------
	// Set up save parameters and store the file
	// ----------------------------------------------------------------
	CStorage::SaveParm( record );
	file.Insert( *itsDb, &record );
	record.file_psiz = CStorage::Pack( record.file_data, aDataFile );
	file.Update( *itsDb, &record );

	return	record;
}

// --------------------------------------------------------------------
// public:		Load a file	from the system
// --------------------------------------------------------------------
data_file_t		CStorage::Load	(	const char *	aDataFile,
									dword_t			aFileIdnt )	{
	data_file_t		record;
	CTableFile		file;
	CMySqlWhere		w;
	data_file_tl	list;

	// ----------------------------------------------------------------
	// Get the file record
	// ----------------------------------------------------------------
	w << "file_idnt=" << aFileIdnt;
	list = file.Select( *itsDb, w );

	// ----------------------------------------------------------------
	// Not found ?
	// ----------------------------------------------------------------
	if	( list.size() < 1 ) {
		throw CError( "Record not found", w.Where() );
	}
	record = *(list.begin());

	// ----------------------------------------------------------------
	// Do we have a read problem here ?
	// ----------------------------------------------------------------
	if	( record.file_flag & FLAG_DENY_READ ) {
		throw CError( "Access denied", record.file_name );
	}

	// ----------------------------------------------------------------
	// Unpack the file
	// ----------------------------------------------------------------
	CStorage::UnPack( aDataFile, record.file_data );

	// ----------------------------------------------------------------
	// Set file time - the gzip does not take care of this automatically
	// ----------------------------------------------------------------
    struct  utimbuf utb;
    utb.actime  = record.file_ctim;
    utb.modtime = record.file_mtim;
    ::utime( aDataFile, &utb );

	// ----------------------------------------------------------------
	// Finally, update the database
	// ----------------------------------------------------------------
	record.file_lcnt++;
	record.file_ltim = ::time( NULL );
	file.Update( *itsDb, &record );

	return	record;
}

// --------------------------------------------------------------------
// public:		Find file(s) from the system
// --------------------------------------------------------------------
data_file_tl	CStorage::Find	(	const char *	aStorName,
									const char *	aStorSuff )	{
	CTableFile		file;
	CMySqlWhere		w;
	CMySqlQuote		q;

	w << "file_name='" << q.Quote( aStorName ) << "'";
	if	( aStorSuff ) {
		if	( *aStorSuff ) {
			w << " and file_suff='" << q.Quote( aStorSuff ) << "'";
		}
	}

	return	file.Select( *itsDb, w );
}

// --------------------------------------------------------------------
// public:		Delete a file from the system in a certain package
// --------------------------------------------------------------------
void			CStorage::Delete(	dword_t 		aFileIdnt,
									dword_t			aPackIdnt ) {
	CTablePackFile		packfile;
	CTablePack			pack;
	data_packfile_tl	pflist;
	CMySqlWhere			w;

	// ----------------------------------------------------------------
	// Delete the entry from the packfile
	// ----------------------------------------------------------------
	w << "packfile_pack=" << aPackIdnt << " and ";
	w << "packfile_file=" << aFileIdnt;
	packfile.Delete( *itsDb, w );

	// ----------------------------------------------------------------
	// In case there was no more files in this package we delete the
	// package as well
	// ----------------------------------------------------------------
	w = "";
	w << "packfile_pack=" << aPackIdnt;
	if	( packfile.Count( *itsDb, w ) == 0 ) {
		w = "";
		w << "pack_idnt=" << aPackIdnt;
		pack.Delete( *itsDb, w );
	}

	// ----------------------------------------------------------------
	// In case the file is no more referenced in any package, we remove
	// the file as well
	// ----------------------------------------------------------------
	w = "";
	w << "packfile_file=" << aFileIdnt;
	if	( packfile.Count( *itsDb, w ) == 0 ) {
		CStorage::Delete( aFileIdnt );
	}
}

// --------------------------------------------------------------------
// public:		Delete a file from the system
// --------------------------------------------------------------------
void			CStorage::Delete(	dword_t 		aFileIdnt ) {
	data_file_t			record;
	data_file_tl		list;
	CTableFile			file;
	CTablePackFile		packfile;
	CTableXref			xref;
	CMySqlWhere			w;

	// ----------------------------------------------------------------
	// Get the record to delete
	// ----------------------------------------------------------------
	w << "file_idnt=" << aFileIdnt;
	list = file.Select( *itsDb, w );
	if	( list.size() < 1 )	return;
	record = *(list.begin());

	// ----------------------------------------------------------------
	// Deleting a write protected unit not possible
	// ----------------------------------------------------------------
	if	( record.file_flag & FLAG_DENY_DEL ) {
		throw CError( "Access denied", record.file_name );
	}

	// ----------------------------------------------------------------
	// Delete the record
	// ----------------------------------------------------------------
	file.Delete( *itsDb, &record );

	// ----------------------------------------------------------------
	// Remove any entries in the pack_file table
	// ----------------------------------------------------------------
	w = "";
	w << "packfile_file=" << aFileIdnt;
	packfile.Delete( *itsDb, w );

	// ----------------------------------------------------------------
	// Remove any entries in the xref table
	// ----------------------------------------------------------------
	w = "";
	w << "xref_file=" << aFileIdnt;
	xref.Delete( *itsDb, w );

	// ----------------------------------------------------------------
	// Is the data file shared by other entries ?
	// If no other references, we can trash the datafile as well
	// ----------------------------------------------------------------
	w = "";
	w << "file_data='" << record.file_data << "'";
	list = file.Select( *itsDb, w );
	if	( list.size() == 0 ) {
		::unlink( record.file_data );
	}
}

// --------------------------------------------------------------------
// public:		Rename a file in the system
// --------------------------------------------------------------------
data_file_t		CStorage::Rename(	dword_t			aFileIdnt,
									const char *	aStorName,
									const char *	aStorSuff )	{
	data_file_t		record;
	CTableFile		file;
	CTableDeny		deny;
	CMySqlWhere		w;
	CMySqlQuote		q;
	data_file_tl	list;

	// ----------------------------------------------------------------
	// Is the file denied
	// ----------------------------------------------------------------
	w << "deny_name='" << q.Quote( aStorName ) << ".";
	w << q.Quote( aStorSuff ) << "'";
	if	( deny.Count( *itsDb, w ) > 0 ) {
		throw CError( "Denied", aStorName );
	}

	// ----------------------------------------------------------------
	// Get the file record
	// ----------------------------------------------------------------
	w = "";
	w << "file_idnt=" << aFileIdnt;
	list = file.Select( *itsDb, w );

	// ----------------------------------------------------------------
	// Not found ?
	// ----------------------------------------------------------------
	if	( list.size() < 1 ) {
		throw CError( "Record not found", w.Where() );
	}
	record = *(list.begin());

	// ----------------------------------------------------------------
	// Renaming denied ?
	// ----------------------------------------------------------------
	if	( record.file_flag & FLAG_DENY_REN ) {
		throw CError( "Access denied", record.file_name );
	}

	// ----------------------------------------------------------------
	// Change the name on the record
	// ----------------------------------------------------------------
	::my_strfit( record.file_name, sizeof( record.file_name ), aStorName );
	::my_strfix( record.file_name );

	::my_strfit( record.file_suff, sizeof( record.file_suff ), aStorSuff );
	::my_strfix( record.file_suff );

	// ----------------------------------------------------------------
	// Do we already have an entry with the name and the checksum ?
	// ----------------------------------------------------------------
	w = "";
	w << "file_name='" << q.Quote( record.file_name ) << "' and ";
	w << "file_suff='" << q.Quote( record.file_suff ) << "' and ";
	w << "file_csum='" << record.file_csum << "'";
	list = file.Select( *itsDb, w );
	
	if	( list.size() > 0 ) {

		// ------------------------------------------------------------
		// Is this actually the same record
		// ------------------------------------------------------------
		if		( aFileIdnt == (*(list.begin())).file_idnt ) {
			record.file_ltim = ::time( NULL );
			file.Update( *itsDb, &record );
		}
		else {
			record = *(list.begin());

			// ---------------------------------------------------------
			// Take care of the PackFile records as well
			// There can be one or more attached to this file, we must
			// update them all to point to this new record
			// ---------------------------------------------------------
			CTablePackFile		packfile;
			data_packfile_t		pfdata;
			data_packfile_tl	pflist;
			data_packfile_tli	pfloop;

			w = "";
			w << "packfile_file=" << aFileIdnt;
			pflist = packfile.Select( *itsDb, w );

			CStorage::Delete( aFileIdnt );

			for	( pfloop = pflist.begin(); pfloop != pflist.end(); pfloop++ ) {
				pfdata = *pfloop;
				pfdata.packfile_file = record.file_idnt;
				packfile.Insert( *itsDb, &pfdata );
			}

			record.file_ltim = ::time( NULL );
			file.Update( *itsDb, &record );
		}
	}
	else {
		record.file_ltim = ::time( NULL );
		file.Update( *itsDb, &record );
	}

	return record;
}

// --------------------------------------------------------------------
// public:		Set	file flags
// --------------------------------------------------------------------
void			CStorage::Flags	(	dword_t			aFileIdnt,
									dword_t			aFlags	) {
	data_file_t		record;
	CTableFile		file;
	CMySqlWhere		w;
	data_file_tl	list;

	// ----------------------------------------------------------------
	// Get the file record
	// ----------------------------------------------------------------
	w << "file_idnt=" << aFileIdnt;
	list = file.Select( *itsDb, w );

	// ----------------------------------------------------------------
	// Not found ?
	// ----------------------------------------------------------------
	if	( list.size() < 1 ) {
		throw CError( "Record not found", w.Where() );
	}
	record = *(list.begin());

	// ----------------------------------------------------------------
	// Changing flags on fixed records is not permitted
	// ----------------------------------------------------------------
	if	( record.file_flag & FLAG_FIX_FLAG ) {
		throw CError( "Access denied", record.file_name );
	}

	record.file_ltim = ::time( NULL );
	record.file_flag = aFlags;

	file.Update( *itsDb, &record );
}

// --------------------------------------------------------------------
// public:		Get	file flags
// --------------------------------------------------------------------
dword_t			CStorage::Flags	(	dword_t			aFileIdnt )	{
	CTableFile		file;
	CMySqlWhere		w;
	data_file_tl	list;

	// ----------------------------------------------------------------
	// Get the file record
	// ----------------------------------------------------------------
	w << "file_idnt=" << aFileIdnt;
	list = file.Select( *itsDb, w );

	// ----------------------------------------------------------------
	// Not found ?
	// ----------------------------------------------------------------
	if	( list.size() < 1 ) {
		throw CError( "Record not found", w.Where() );
	}

	return	(*(list.begin())).file_flag;
}

// --------------------------------------------------------------------
// private:		Get all sorts of file information
// --------------------------------------------------------------------
void			CStorage::FileInfo(	data_file_t	&	aRecord,
									const char *	aFileName )	{
#ifdef	WIN32
	struct _stat	mystat;
	if	( ::_stat( aFileName, &mystat ) ) {
#else
	struct stat		mystat;
	if	( ::stat( aFileName, &mystat ) ) {
#endif
		throw	CError( aFileName, ::strerror( errno ) );
	}

	// ----------------------------------------------------------------
	// Check that this is an ordinary file
	// ----------------------------------------------------------------
#ifdef	WIN32
	if	( ( mystat.st_mode & _S_IFREG ) == 0 ) {
#else
	if	( ( mystat.st_mode & __S_IFMT ) != __S_IFREG ) {
#endif
		throw	CError( aFileName, "Not a regular file" );
	}

	// ----------------------------------------------------------------
	// Information retrieved from the file
	// ----------------------------------------------------------------
	if	( mystat.st_ctime > ::time( NULL ) )	mystat.st_ctime = ::time( NULL );
	if	( mystat.st_mtime > ::time( NULL ) )	mystat.st_mtime = ::time( NULL );
	if	( mystat.st_ctime > mystat.st_mtime ) {
		aRecord.file_ctim	= mystat.st_mtime;
		aRecord.file_mtim	= mystat.st_mtime;
	}
	else {
		aRecord.file_ctim	= mystat.st_ctime;
		aRecord.file_mtim	= mystat.st_mtime;
	}

	aRecord.file_atim	= ::time( NULL );
	aRecord.file_ltim	= ::time( NULL );
	aRecord.file_size	= mystat.st_size;
	aRecord.file_lcnt	= 0;

	// ----------------------------------------------------------------
	// Set up the rating
	// ----------------------------------------------------------------
	aRecord.file_rate	= 100000;
	aRecord.file_rat0	= 49;
	aRecord.file_rat1	= 0;
	aRecord.file_rat2	= 49;

	// ----------------------------------------------------------------
	// Calculate a checksum for this file
	// ----------------------------------------------------------------
	CStorage::CheckSum( aRecord, aFileName );
}

// --------------------------------------------------------------------
// private:		Calculate checksum
// --------------------------------------------------------------------
void			CStorage::CheckSum(	data_file_t	&	aRecord,
									const char *	aFileName ) {
	FILE *	strm	= NULL;
	dword_t	csum	[10];
	int		i;
	dword_t	rdat;

	try {

		// ------------------------------------------------------------
		// Initialize the checksum with file size
		// ------------------------------------------------------------
		for	( i = 0; i < 10; i++ )	csum[i] = aRecord.file_size;

		// ------------------------------------------------------------
		// Start reading
		// ------------------------------------------------------------
		strm = ::fopen( aFileName, "rb" );
		if	( ! strm ) {
			throw CError( aFileName, ::strerror( errno ) );
		}

		// ------------------------------------------------------------
		// Each byte from the file is added to the checksum array
		// ------------------------------------------------------------
		i = 0;
		while	( ! feof( strm ) ) {
			rdat = 0;
			::fread( &rdat, sizeof( rdat ), 1, strm );
			if	( ferror( strm ) ) {
				throw CError( aFileName, ::strerror( errno ) );
			}
			csum[i] = csum[i] + rdat;
			i++;
			if	( i >= 10 )	i = 0;
		}

		::fclose( strm );	strm = NULL;

		// ------------------------------------------------------------
		// Generate the checksum string which will become 80 characters
		// ------------------------------------------------------------
		for	( i = 0; i < 10; i++ ) {
			CStorage::DwordSum( aRecord.file_csum + (8 * i), csum[i] );
		}
	}

	catch ( ... ) {
		if	( strm )	::fclose( strm );
		throw;
	}
}

// --------------------------------------------------------------------
// private:		DWord hex presentation
// --------------------------------------------------------------------
void			CStorage::DwordSum(	char *			aCsum,
									dword_t			aSum ) {
	CStorage::WordSum( aCsum,     (word_t)( aSum >> 16 ) );
	CStorage::WordSum( aCsum + 4, (word_t)( aSum & 0x0000ffff ) );
	aCsum[8] = 0;
}

// --------------------------------------------------------------------
// private:		Word hex presentation
// --------------------------------------------------------------------
void			CStorage::WordSum(	char *			aCsum,
									word_t			aSum ) {
	CStorage::ByteSum( aCsum,     (byte_t)( aSum >> 8 ) );
	CStorage::ByteSum( aCsum + 2, (byte_t)( aSum & 0x00ff ) );
}

// --------------------------------------------------------------------
// private:		Byte hex presentation
// --------------------------------------------------------------------
void			CStorage::ByteSum(	char *			aCsum,
									byte_t			aSum ) {
	CStorage::NibbSum( aCsum,     (byte_t)( aSum >> 4 ) );
	CStorage::NibbSum( aCsum + 1, (byte_t)( aSum & 0x0f ) );
}

// --------------------------------------------------------------------
// private:		Nibble hex presentation
// --------------------------------------------------------------------
static const char *	__nibbles = "0123456789ABCDEF";
void			CStorage::NibbSum(	char *			aCsum,
									byte_t			aSum ) {
	aCsum[0] = __nibbles[ aSum & 0x0f ];
}

// --------------------------------------------------------------------
// private:		Set parameters for saving
// --------------------------------------------------------------------
void			CStorage::SaveParm(	data_file_t	&	aRecord ) {
	CTableFile		file;
	CTablePref		pref;
	data_pref_tl	list;
	CMySqlWhere		w;
	
	// ----------------------------------------------------------------
	// Get list of the available directories for storing the data
	// ----------------------------------------------------------------
	w << "pref_name >= 'storage000' AND pref_name <= 'storage999'";
	list = pref.Select( *itsDb, w );

	// ----------------------------------------------------------------
	// None available !!!
	// ----------------------------------------------------------------
	if	( list.size() < 1 ) {
		throw CError( "No storage places available" );
	}

	// ----------------------------------------------------------------
	// Select one of these in random if more than one
	// ----------------------------------------------------------------
	::srand( ::time( NULL ) );
	if	( list.size() > 1 ) {
		data_pref_tli	loop;
#ifdef	WIN32
		dword_t			item = (dword_t)( ::rand() % list.size() );
#else
		dword_t			item = (dword_t)( ::random() % list.size() );
#endif

		loop = list.begin();
		while	( item > 0 ) {
			loop++;
			item--;
		}
		::strcpy( aRecord.file_data, (*loop).pref_valu );
	}
	else {
		::strcpy( aRecord.file_data, (*(list.begin())).pref_valu );
	}

	// ----------------------------------------------------------------
	// Fill in the record
	// ----------------------------------------------------------------
	aRecord.file_idnt = file.NextIdnt( *itsDb );
	::sprintf( aRecord.file_data + ::strlen( aRecord.file_data ), "/%d.gz", aRecord.file_idnt );
}

// --------------------------------------------------------------------
// private:		Pack into internal file	(gz)
// --------------------------------------------------------------------
dword_t			CStorage::Pack	(	const char *	aGzFile,
									const char *	aDataFile )	{
	CGzip	gzip;
	return gzip.Gzip( aGzFile, aDataFile );
}

// --------------------------------------------------------------------
// private:		Unpack an internal file
// --------------------------------------------------------------------
void			CStorage::UnPack(	const char *	aDataFile,
									const char *	aGzFile	) {
	CGzip	gzip;
	gzip.Gunzip( aDataFile, aGzFile );
}

// --------------------------------------------------------------------
// EOF:	CStorage.cxx
// --------------------------------------------------------------------
