// --------------------------------------------------------------------
// CWinIniFile.cpp
// Whatis:  Windows style INI file item
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 10-FEB-2003     Created this source
// --------------------------------------------------------------------
#include	"CWinIniFile.hxx"
#include	"CError.hxx"

// --------------------------------------------------------------------
// public:	Constructors
// --------------------------------------------------------------------
CWinIniFile::CWinIniFile	() {
	CWinIniFile::Cleanup();
}

// --------------------------------------------------------------------
CWinIniFile::CWinIniFile			( const CWinIniFile & aC ) {
	CWinIniFile::Cleanup();
	CWinIniFile::Name( aC.Name() );
	CWinIniFile::SectionList( aC.SectionList() );
}

// --------------------------------------------------------------------
CWinIniFile	CWinIniFile::operator =	( const CWinIniFile & aC ) {
	CWinIniFile::Free();
	CWinIniFile::Name( aC.Name() );
	CWinIniFile::SectionList( aC.SectionList() );
	return	*this;
}

// --------------------------------------------------------------------
// public:	Destructor
// --------------------------------------------------------------------
CWinIniFile::~CWinIniFile	() {
	CWinIniFile::Free();
}

// --------------------------------------------------------------------
// public:	Comparison operators
// --------------------------------------------------------------------
bool	CWinIniFile::operator ==	( const CWinIniFile & aC ) {
	return	::strcmp( CWinIniFile::Name(), aC.Name() ) == 0;
}

// --------------------------------------------------------------------
bool	CWinIniFile::operator !=	( const CWinIniFile & aC ) {
	return	::strcmp( CWinIniFile::Name(), aC.Name() ) != 0;
}

// --------------------------------------------------------------------
// public:	Streaming
// --------------------------------------------------------------------
void	CWinIniFile::Write	( FILE * aS ) const {
	CWinIniSection_lci	loop;
	for	( 	loop =  itsSectionList.begin();
			loop != itsSectionList.end();
			loop++ ) {
		(*loop).Write( aS );
	}
}

// --------------------------------------------------------------------
void	CWinIniFile::Write	( const char * aF ) const {
	FILE *	strm	= NULL;

    try {
    	::unlink( aF );
		strm = ::fopen( aF, "w" );
        if	( ! strm ) {
        	throw CError( aF, ::strerror( errno ) );
        }
        CWinIniFile::Write( strm );
        ::fclose( strm );
    }

    catch	( ... ) {
    	if	( strm ) {
        	::fclose( strm );
        }
        throw;
    }
}

// --------------------------------------------------------------------
void	CWinIniFile::Read	( FILE * aS ) {
	CWinIniSection	mysect;
	do {
		mysect.Read( aS );
		if	( (mysect.Name())[0] ) {
			CWinIniFile::Insert( mysect );
		}
	}	while	( (mysect.Name())[0] );
}

// --------------------------------------------------------------------
void	CWinIniFile::Read	( const char * aF ) {
	FILE *	strm	= NULL;

    try {
		strm = ::fopen( aF, "r" );
        if	( ! strm ) {
        	throw CError( aF, ::strerror( errno ) );
        }
        CWinIniFile::Read( strm );
        ::fclose( strm );
    }

    catch	( ... ) {
    	if	( strm ) {
        	::fclose( strm );
        }
        throw;
    }
}

// --------------------------------------------------------------------
// public:	Data access
// --------------------------------------------------------------------
void	CWinIniFile::SectionList	( const CWinIniSection_l & aL ) {
	itsSectionList = aL;
}

// --------------------------------------------------------------------
CWinIniSection	CWinIniFile::Section		( const char * aName ) const {
	CWinIniSection		sect;
	CWinIniSection_lci	loop;
	for	( loop = itsSectionList.begin(); loop != itsSectionList.end(); loop++ ) {
    	if	( (*loop) == aName ) {
        	sect = *loop;
            break;
        }
    }
    return	sect;
}

// --------------------------------------------------------------------
const CWinIniSection *	CWinIniFile::SectionPtr	( const char * aName ) const {
	CWinIniSection		sect;
	CWinIniSection_lci	loop;
	for	( loop = itsSectionList.begin(); loop != itsSectionList.end(); loop++ ) {
    	if	( (*loop) == aName ) {
        	return	&(*loop);
        }
    }
    return	NULL;
}

// --------------------------------------------------------------------
void	CWinIniFile::Name		( const char * aName ) {
	if	( itsName )		delete [] itsName;
	itsName = ::my_private_strdup( aName );
}

// --------------------------------------------------------------------
void	CWinIniFile::Insert		( const CWinIniSection & aI ) {
	CWinIniSection_li	loop;

    // Find similar section
    for	( 	loop  = itsSectionList.begin();
    		loop != itsSectionList.end();
            loop++ ) {
		if	( (*loop) == aI )	break;
    }

    // Found a similar section ?
    if	( loop != itsSectionList.end() ) {
		CWinIniItem_lci	item;
        for	(	item  = aI.ItemList().begin();
				item != aI.ItemList().end();
                item++ ) {
			(*loop).Insert( *item );
        }
    }

    // Not found - append it to the end
    else {
    	itsSectionList.push_back( aI );
    }
}

// --------------------------------------------------------------------
void	CWinIniFile::Insert		( const CWinIniSection_l & aL ) {
	CWinIniSection_lci	loop;
    for	( 	loop  = itsSectionList.begin();
    		loop != itsSectionList.end();
            loop++ ) {
		CWinIniFile::Insert( *loop );
    }
}

// --------------------------------------------------------------------
void	CWinIniFile::Insert		( const char * aSectName, const char * aName, const char * aValue ) {
	CWinIniSection_li	loop;
    CWinIniItem			item;
    item.Name	( aName );
    item.Value	( aValue );
    for	( loop = itsSectionList.begin(); loop != itsSectionList.end(); loop++ ) {
    	if	( (*loop) == aSectName ) {
            (*loop).Insert( item );
        	break;
        }
    }
    if	( loop == itsSectionList.end() ) {
    	CWinIniSection	sec;
        sec.Name( aSectName );
        sec.Insert( item );
        CWinIniFile::Insert( sec );
    }
}

// --------------------------------------------------------------------
void	CWinIniFile::Delete		( const CWinIniSection & aI ) {
	itsSectionList.remove( aI );
}

// --------------------------------------------------------------------
void	CWinIniFile::Delete		( const CWinIniSection_l & aL ) {
	CWinIniSection_lci	loop;
	for	( loop = aL.begin(); loop != aL.end(); loop++ ) {
		itsSectionList.remove( *loop );
	}
}

// --------------------------------------------------------------------
void	CWinIniFile::Delete		( const char * aSectName, const char * aName ) {
	CWinIniSection_li	loop;
    CWinIniItem			item;
    item.Name	( aName );
    for	( loop = itsSectionList.begin(); loop != itsSectionList.end(); loop++ ) {
    	if	( (*loop) == aSectName ) {
			(*loop).Delete( item );
        }
    }
}

// --------------------------------------------------------------------
CWinIniItem_l	CWinIniFile::ItemList( const char * aSectName, const char * aItemName ) const {
	CWinIniItem_l		list;
	CWinIniSection_lci	loop;
    for	( loop = itsSectionList.begin(); loop != itsSectionList.end(); loop++ ) {
    	if	( (*loop) == aSectName ) {
        	CWinIniItem_l	slst = (*loop).ItemList( aItemName );
            CWinIniItem_li	slop;
            for	( slop = slst.begin(); slop != slst.end(); slop++ ) {
            	list.push_back( *slop );
            }
        }
    }
	return	list;
}

// --------------------------------------------------------------------
const char *	CWinIniFile::Value( const char * aSectName, const char * aItemName ) const {
	CWinIniSection_lci	loop;
    for	( loop = itsSectionList.begin(); loop != itsSectionList.end(); loop++ ) {
    	if	( (*loop) == aSectName ) return (*loop).Value( aItemName );
    }
	return	"";
}

// --------------------------------------------------------------------
dword_t			CWinIniFile::IntValue( const char * aSectName, const char * aItemName ) const {
	return	(dword_t)::atol( CWinIniFile::Value( aSectName, aItemName ) );
}

// --------------------------------------------------------------------
bool			CWinIniFile::BoolValue( const char * aSectName, const char * aItemName ) const {
#ifdef	WIN32
	return	::stricmp( CWinIniFile::Value( aSectName, aItemName ), "true" ) == 0;
#else
	return	::my_stricmp( CWinIniFile::Value( aSectName, aItemName ), "true" ) == 0;
#endif
}

// --------------------------------------------------------------------
void	CWinIniFile::Value( const char * aSectName, const char * aName, const char * aValue ) {
	CWinIniSection_li	loop;
    CWinIniItem			item;
    item.Name	( aName );
    item.Value	( aValue );
    for	( loop = itsSectionList.begin(); loop != itsSectionList.end(); loop++ ) {
    	if	( (*loop) == aSectName ) {
			(*loop).Delete( item );
            (*loop).Insert( item );
        	break;
        }
    }
}

// --------------------------------------------------------------------
const char *				CWinIniFile::Name		( void ) const { return itsName ? itsName : ""; }
const CWinIniSection_l &	CWinIniFile::SectionList( void ) const { return itsSectionList; }
void						CWinIniFile::Clear		( void ) { itsSectionList.clear(); }

// --------------------------------------------------------------------
void	CWinIniFile::Free	( void ) {
	if	( itsName )		delete [] itsName;
	CWinIniFile::Cleanup();
}

// --------------------------------------------------------------------
void	CWinIniFile::Cleanup	( void ) {
	itsName = NULL;
	itsSectionList.clear();
}

// --------------------------------------------------------------------
// EOF:	CWinIniFile.cpp
// --------------------------------------------------------------------
