// --------------------------------------------------------------------
// CDeflate.cpp
// Whatis:	Deflate compressor
// Authors:	Esko 'Varpu' Ilola	EIL
//			Jean-loup Gailly
// History:	EIL	22-JUN-2003		Created	this source, based on work of
//								Jean-loup Gailly
// --------------------------------------------------------------------
#include	"CDeflate.hxx"
#include	"CError.hxx"

// --------------------------------------------------------------------
// public:	Constructor
// --------------------------------------------------------------------
CDeflate::CDeflate	() {
	// Wipe everything clean
    ::memset( &itsZS, 0, sizeof( itsZS ) );
    itsHS = false;
    itsDM = false;
}

// --------------------------------------------------------------------
// public:	Destructor
// --------------------------------------------------------------------
CDeflate::~CDeflate	() {
	if	( itsHS ) {
    	if	( itsDM )	::deflateEnd( &itsZS );
        else			::inflateEnd( &itsZS );
    }
    ::memset( &itsZS, 0, sizeof( itsZS ) );
    itsHS = false;
    itsDM = false;
}

// --------------------------------------------------------------------
// public:	Do compression
// --------------------------------------------------------------------
void	CDeflate::Compress	( codec_stream_t & aArgs ) {
   	int	zlresult;

	// Set up our local stream from user args
	itsZS.next_in		= aArgs.next_in;
	itsZS.avail_in      = aArgs.avail_in;
	itsZS.total_in      = aArgs.total_in;
	itsZS.next_out      = aArgs.next_out;
	itsZS.avail_out     = aArgs.avail_out;
	itsZS.total_out     = aArgs.total_out;

	// Is this the first time in ?
    if	( ! itsHS ) {

        // Clear the checksum calculator
	    if	( aArgs.checksum )	aArgs.checksum->Cleanup();

		itsZS.zalloc		= Z_NULL;
		itsZS.zfree         = Z_NULL;
		itsZS.opaque        = Z_NULL;

		itsZS.data_type     = Z_UNKNOWN;

		// In zlib we cannot use the ordinary deflateInit
        // as it will put in zlib headers thus mixing things up
		zlresult = ::deflateInit2(	&itsZS,
        							aArgs.complevel,
                                    Z_DEFLATED,
                                    -MAX_WBITS,
                                    8,
									Z_DEFAULT_STRATEGY );

		if	( zlresult != Z_OK ) {
        	throw CError( "CDeflate::Compress", itsZS.msg );
        }
        itsHS = true;
        itsDM = true;
	}

    // If trying to mix up compress/decompress ???
    if	( itsDM == false ) {
    	throw CError( "CDeflate::Compress", "Illegal use" );
    }

    // Do some checksum calculations as needed
    if	( aArgs.checksum )	aArgs.checksum->Calculate( itsZS.next_in, itsZS.avail_in );

	// Compress a chunk
    zlresult = ::deflate( &itsZS, Z_NO_FLUSH );
	if	( zlresult != Z_OK ) {
       	throw CError( "CDeflate::Compress", itsZS.msg );
    }

	// Bring state back to calling application
	aArgs.next_in		= itsZS.next_in;
	aArgs.avail_in      = itsZS.avail_in;
	aArgs.total_in      = itsZS.total_in;
	aArgs.next_out      = itsZS.next_out;
	aArgs.avail_out     = itsZS.avail_out;
	aArgs.total_out     = itsZS.total_out;
}

// --------------------------------------------------------------------
// public:	Do decompression
// --------------------------------------------------------------------
void	CDeflate::Decompress	( codec_stream_t & aArgs ) {
   	int	zlresult;

	// Set up our local stream from user args
	itsZS.next_in		= aArgs.next_in;
	itsZS.avail_in      = aArgs.avail_in;
	itsZS.total_in      = aArgs.total_in;
	itsZS.next_out      = aArgs.next_out;
	itsZS.avail_out     = aArgs.avail_out;
	itsZS.total_out     = aArgs.total_out;

	// Is this the first time in ?
    if	( ! itsHS ) {

        // Clear the checksum calculator
	    if	( aArgs.checksum )	aArgs.checksum->Cleanup();

		itsZS.zalloc		= Z_NULL;
		itsZS.zfree         = Z_NULL;
		itsZS.opaque        = Z_NULL;

		itsZS.data_type     = Z_UNKNOWN;

		zlresult = ::inflateInit(  &itsZS );

		if	( zlresult != Z_OK ) {
        	throw CError( "CDeflate::Decompress", itsZS.msg );
        }
        itsHS = true;
        itsDM = false;

	}

    // If trying to mix up compress/decompress ???
	if	( itsDM ) {
    	throw CError( "CDeflate::Decompress", "Illegal use" );
    }

	// Compress a chunk
    zlresult = ::deflate( &itsZS, Z_NO_FLUSH );
	if	( zlresult != Z_OK ) {
       	throw CError( "CDeflate::Compress", itsZS.msg );
    }

    // Do some checksum calculations as needed
    if	( aArgs.checksum )	aArgs.checksum->Calculate(	aArgs.next_out,
       													itsZS.total_out - aArgs.total_out );


	// Bring state back to calling application
	aArgs.next_in		= itsZS.next_in;
	aArgs.avail_in      = itsZS.avail_in;
	aArgs.total_in      = itsZS.total_in;
	aArgs.next_out      = itsZS.next_out;
	aArgs.avail_out     = itsZS.avail_out;
	aArgs.total_out     = itsZS.total_out;
}

// --------------------------------------------------------------------
// public:	Flush any pending data
// --------------------------------------------------------------------
bool	CDeflate::End	( codec_stream_t & aArgs ) {
   	int	zlresult;

	// Is there need for doing this ?
    if	( ! itsHS )	return	true;	// We can stop immediately

	// Update our local stream if needed
	itsZS.next_in		= aArgs.next_in;
	itsZS.avail_in      = aArgs.avail_in;
	itsZS.total_in      = aArgs.total_in;
	itsZS.next_out      = aArgs.next_out;
	itsZS.avail_out     = aArgs.avail_out;
	itsZS.total_out     = aArgs.total_out;

    // Were we compressing ?
    if	( itsDM ) {

		// Let's try to finish this !
	    if	( aArgs.checksum )	aArgs.checksum->Calculate( itsZS.next_in, itsZS.avail_in );
	    zlresult = ::deflate( &itsZS, Z_FINISH );

		// It will be either Z_OK or Z_STREAM_END or error
        if	( ( zlresult != Z_OK ) && ( zlresult != Z_STREAM_END ) ) {
	       	throw CError( "CDeflate::End", itsZS.msg );
        }
    }

    // Apparently we were decompressing
    else {

		// Let's try to finish this !
	    zlresult = ::inflate( &itsZS, Z_FINISH );

		// It will be either Z_OK or Z_STREAM_END or error
        if	( ( zlresult != Z_OK ) && ( zlresult != Z_STREAM_END ) ) {
	       	throw CError( "CDeflate::End", itsZS.msg );
        }

		// Calculate the checksum !
	    if	( aArgs.checksum )	aArgs.checksum->Calculate(	aArgs.next_out,
        													itsZS.total_out - aArgs.total_out );
    }

    // Arguments back to caller
	aArgs.next_in		= itsZS.next_in;
	aArgs.avail_in      = itsZS.avail_in;
	aArgs.total_in      = itsZS.total_in;
	aArgs.next_out      = itsZS.next_out;
	aArgs.avail_out     = itsZS.avail_out;
	aArgs.total_out     = itsZS.total_out;

	// Can we get rid ot the stream now ?
    if	( zlresult == Z_STREAM_END ) {
		if	( itsHS ) {
    		if	( itsDM )	::deflateEnd( &itsZS );
        	else			::inflateEnd( &itsZS );
	    }
    	::memset( &itsZS, 0, sizeof( itsZS ) );
	    itsHS = false;
    	itsDM = false;
	}

    return	zlresult == Z_STREAM_END;
}

// --------------------------------------------------------------------
// EOF:	CDeflate.cpp
// --------------------------------------------------------------------
