// --------------------------------------------------------------------
// CZip.cpp
// Whatis:	Class for reading/writing zip files
// Authors:	Esko 'Varpu' Ilola	EIL
// History:	EIL	24-FEB-2003		Created	this source
// --------------------------------------------------------------------
#include	"CZip.hxx"
#include	"CZipStreamFile.hxx"
#include	"CError.hxx"

// --------------------------------------------------------------------
// public:	Constructor and destructor
// --------------------------------------------------------------------
CZip::CZip () {
	CZip::Clear();
}

// --------------------------------------------------------------------
CZip::~CZip () {
	CZip::Free();
}

// --------------------------------------------------------------------
// public:	Copy constructor and assignment operator
// --------------------------------------------------------------------
CZip::CZip				( const CZip & aC ) {
	CZip::Clear();
	itsFileBlocks = aC.FileBlocks();
	itsCentralDir = aC.CentralDir();
}

// --------------------------------------------------------------------
CZip & CZip::operator =	( const CZip & aC ) {
	CZip::Free();
	itsFileBlocks = aC.FileBlocks();
	itsCentralDir = aC.CentralDir();
	return *this;
}

// --------------------------------------------------------------------
// public:	Reading ZIP
// --------------------------------------------------------------------
void	CZip::Read	( const char * aZipFile ) {
	CZipStreamFile	myfile( aZipFile );
	myfile.OpenRO();
	CZip::Read( myfile );
}

// --------------------------------------------------------------------
void	CZip::Read	( IZipStream & aReader ) {
	CZip::Free();
	CZipHeaderSignature	signature;

	signature.Read( aReader );
	while	( signature == zip_signature_localfile ) {
   		CZipFileBlock		myblock;

		myblock.Read( signature, aReader );

		itsFileBlocks.push_back( myblock );

		if	( aReader.Eof() )	break;
		else					signature.Read( aReader );
	}

	if	( signature == zip_signature_centralfile ) {
		itsCentralDir.Read( signature, aReader );
	}
	else {
		throw	CError( aReader.Name(), "Missing central directory" );
	}
}

// --------------------------------------------------------------------
// public:	Writing ZIP
// --------------------------------------------------------------------
void	CZip::Write	( const char * aZipFile, CZipNotification * aNotify ) {
	CZipStreamFile	myfile( aZipFile );
	myfile.Create();
	CZip::Write( myfile, aNotify );
}

// --------------------------------------------------------------------
void	CZip::Write	( IZipStream & aWriter, CZipNotification * aNotify ) {
	CZipFileBlock_li	loop;

    // If we have this notify - fill it in
    if	( aNotify != NULL ) {
    	dword_t	totalsize;
		aNotify->Task			( "Saving to disk" );
		aNotify->FileName		( "" );
        totalsize = 0;
	    for	( loop = itsFileBlocks.begin(); loop != itsFileBlocks.end(); loop++ ) {
			totalsize = totalsize + (*loop).LocalFileHeader().UncompressedSize().Value();
	    }
        aNotify->TotalBytes		( totalsize );
        aNotify->FileBytes		( 0 );
        aNotify->TotalFiles		( itsFileBlocks.size() );
		aNotify->TotalBytesDone	( 0 );
		aNotify->FileBytesDone	( 0 );
        aNotify->Aborting		( false );
    }
    for	( loop = itsFileBlocks.begin(); loop != itsFileBlocks.end(); loop++ ) {
	    if	( aNotify ) {
        	if	( aNotify->Aborting() )	break;
	        aNotify->FileBytes( (*loop).LocalFileHeader().UncompressedSize().Value() );
	        aNotify->FileBytesDone( 0 );
        }

        // Are we compressing these on-the fly ?
        if	( ( (*loop).FileData().CompressedFileSize() == 0 ) &&
        	  ( (*loop).FileData().CompressedData().Size() == 0 ) ) {
			(*loop).CompressWrite( aWriter, aNotify );
			itsCentralDir.Update( itsFileBlocks );
        }
        else {
			(*loop).Write( aWriter );
        }

	    if	( aNotify ) {
        	if	( aNotify->Aborting() )	break;
			aNotify->FilesDone( aNotify->FilesDone() + 1 );
        }
	}
	itsCentralDir.Write( itsFileBlocks, aWriter );
}

// --------------------------------------------------------------------
// public:	Manipulating directory
// --------------------------------------------------------------------
void	CZip::Cleanup	( void ) {
	CZip::Free();
}

// --------------------------------------------------------------------
void	CZip::Free		( void ) {
	CZip::Clear();
}

// --------------------------------------------------------------------
void	CZip::Clear		( void ) {
	CZipFileBlock_li	loop;

    for	( loop = itsFileBlocks.begin(); loop != itsFileBlocks.end(); loop++ ) {
		(*loop).Cleanup();
	}
	itsFileBlocks.clear();
	itsCentralDir.Cleanup();
}

// --------------------------------------------------------------------
void	CZip::AddDirectoryEntry	(	const CGz &		aGz,
									const char *	aDirectoryName,
									time_t			aFileTime ) {
	// First, remove the entry - just in case
	CZip::DelDirectoryEntry( aDirectoryName );

    // Add to the list of local entries
    CZipFileBlock	myblock( aGz, aDirectoryName, aFileTime );
	itsFileBlocks.push_back( myblock );

    // Same for the central directory
	itsCentralDir.AddDirectoryEntry( aGz, aDirectoryName, aFileTime );

    // Last but not least - update records
	itsCentralDir.Update( itsFileBlocks );
}

// --------------------------------------------------------------------
void	CZip::AddDirectoryEntry	( const CZipFlags &	aFlags,
								  const char *		aLocalFileName,
								  const char *		aDirectoryName ) {
	// First, remove the entry - just in case
	CZip::DelDirectoryEntry( aDirectoryName );

    // Add to the list of local entries
    CZipFileBlock	myblock(  aFlags, aLocalFileName, aDirectoryName );
	itsFileBlocks.push_back( myblock );

    // Same for the central directory
	itsCentralDir.AddDirectoryEntry( aFlags, aLocalFileName, aDirectoryName );

    // Last but not least - update records
	itsCentralDir.Update( itsFileBlocks );
}

// --------------------------------------------------------------------
void	CZip::AddDirectoryEntry	( const CZipFlags &	aFlags,
								  const char *		aDataBlock,
                                  dword_t			aDataLen,
								  const char *		aDirectoryName ) {
	// First, remove the entry - just in case
	CZip::DelDirectoryEntry( aDirectoryName );

    // Add to the list of local entries
    CZipFileBlock	myblock(  aFlags, aDataBlock, aDataLen, aDirectoryName );
	itsFileBlocks.push_back( myblock );

    // Same for the central directory
	itsCentralDir.AddDirectoryEntry( aFlags, myblock, aDirectoryName );

    // Last but not least - update records
	itsCentralDir.Update( itsFileBlocks );
}

// --------------------------------------------------------------------
void	CZip::AddDirectoryEntry	( const CZipFlags &	aFlags,
								  IZipStream *		aStream,
                                  dword_t			aSize,
								  const char *		aDirectoryName ) {

	// First, remove the entry - just in case
	CZip::DelDirectoryEntry( aDirectoryName );

    // Add to the list of local entries
    CZipFileBlock	myblock(  aFlags, aStream, aSize, aDirectoryName );
	itsFileBlocks.push_back( myblock );

    // Same for the central directory
	itsCentralDir.AddDirectoryEntry( aFlags, myblock, aDirectoryName );

    // Last but not least - update records
	itsCentralDir.Update( itsFileBlocks );
}

// --------------------------------------------------------------------
void	CZip::DelDirectoryEntry	( const char *		aFilter ) {
	// First, remove the entries from local file entries
	CZipFileBlock_li	loop;
    loop = itsFileBlocks.begin();
    while	( loop != itsFileBlocks.end() ) {
    	if	( (*loop).Filter( aFilter ) )	loop = itsFileBlocks.erase( loop );
        else								loop++;
    }

    // And after this from the central directory
    itsCentralDir.DelDirectoryEntry( aFilter );

    // Last but not least - update records
	itsCentralDir.Update( itsFileBlocks );
}

// --------------------------------------------------------------------
// public:	Manipulating comments
// --------------------------------------------------------------------
void	CZip::AddZipComment		( const char * aComment ) {
	itsCentralDir.AddZipComment( aComment );

    // Last but not least - update records
	itsCentralDir.Update( itsFileBlocks );
}

// --------------------------------------------------------------------
void	CZip::AddFileComment	( const char * aFilter,
								  const char * aComment ) {
	itsCentralDir.AddFileComment( aFilter, aComment );

    // Last but not least - update records
	itsCentralDir.Update( itsFileBlocks );
}

// --------------------------------------------------------------------
// public:	Decompress data in memory into local files
// --------------------------------------------------------------------
void	CZip::Decompress		( const CZipFlags &	aFlags,
								  const char * 		aTargetPath,
								  const char * 		aFilter ) {
	CZipFileBlock_li	loop;

    for	( loop = itsFileBlocks.begin(); loop != itsFileBlocks.end(); loop++ ) {
		if	( (*loop).Filter( aFilter ) ) {
        	(*loop).Decompress( aFlags, aTargetPath );
        }
    }
}

// --------------------------------------------------------------------
// public:	Compress data from local files into memory
// --------------------------------------------------------------------
void	CZip::Compress			( CZipNotification * aNotify ) {
	dword_t				totalsize;
	CZipFileBlock_li	loop;

    // If we have this notify - fill it in
    if	( aNotify != NULL ) {
		aNotify->Task( "" );
		aNotify->FileName( "" );
        totalsize = 0;
	    for	( loop = itsFileBlocks.begin(); loop != itsFileBlocks.end(); loop++ ) {
			totalsize = totalsize + (*loop).LocalFileHeader().UncompressedSize().Value();
	    }
        aNotify->TotalBytes( totalsize );
        aNotify->FileBytes( 0 );
        aNotify->TotalFiles( itsFileBlocks.size() );
		aNotify->TotalBytesDone( 0 );
		aNotify->FileBytesDone( 0 );
        aNotify->Aborting( false );
    }

	// Do the compression
    for	( loop = itsFileBlocks.begin(); loop != itsFileBlocks.end(); loop++ ) {
        if	( aNotify ) {
	        aNotify->FileBytes( (*loop).LocalFileHeader().UncompressedSize().Value() );
	        aNotify->FileBytesDone( 0 );
		}

       	(*loop).Compress( aNotify );

		// Check for an abortion request
        if	( aNotify ) {
        	if	( aNotify->Aborting() )	break;
	        aNotify->FilesDone( aNotify->FilesDone() + 1 );
        }
    }

    // Last but not least - update records
	itsCentralDir.Update( itsFileBlocks );
}

// --------------------------------------------------------------------
const CZipFileBlock_l &	CZip::FileBlocks		( void ) const {
	return itsFileBlocks;
}

// --------------------------------------------------------------------
const CZipCentralDir &	CZip::CentralDir		( void ) const	{
	return itsCentralDir;
}

// --------------------------------------------------------------------
dword_t	CZip::ZippedSize	( void ) const {
	CZipFileBlock_lci	loop;
    dword_t				zize = 0;

    for	( loop = itsFileBlocks.begin(); loop != itsFileBlocks.end(); loop++ ) {
    	zize = zize + (*loop).ZippedSize();
    }
	return	zize + itsCentralDir.ZippedSize();
}

// --------------------------------------------------------------------
// EOF:	CZip.cpp
// --------------------------------------------------------------------
