// --------------------------------------------------------------------
// CZipCentralDir.cpp
// Whatis:	Central directory of a zip file
// Authors:	Esko 'Varpu' Ilola	EIL
// History:	EIL	02-MAR-2003		Created	this source
// --------------------------------------------------------------------
#include	"CZipCentralDir.hxx"
#include	"CError.hxx"

// --------------------------------------------------------------------
// public:	Constructor and destructor
// --------------------------------------------------------------------
CZipCentralDir::CZipCentralDir () {
	CZipCentralDir::Clear();
}

// --------------------------------------------------------------------
CZipCentralDir::~CZipCentralDir () {
	CZipCentralDir::Free();
}

// --------------------------------------------------------------------
// public:	Copy constructor and assignment operator
// --------------------------------------------------------------------
CZipCentralDir::CZipCentralDir				( const CZipCentralDir & aC ) {
	CZipCentralDir::Clear();
	itsFileHeaders		= aC.FileHeaders();
	itsDigitalSignature	= aC.DigitalSignature();
	itsEOCD				= aC.EOCD();
}

// --------------------------------------------------------------------
CZipCentralDir & CZipCentralDir::operator =	( const CZipCentralDir & aC ) {
	CZipCentralDir::Free();
	itsFileHeaders		= aC.FileHeaders();
	itsDigitalSignature	= aC.DigitalSignature();
	itsEOCD				= aC.EOCD();
	return *this;
}

// --------------------------------------------------------------------
// public:	Interfaces
// --------------------------------------------------------------------
void	CZipCentralDir::Read 	(	const CZipHeaderSignature &	aSig,
                                    IZipStream &				aReader ) {

	CZipCentralDir::Free();

    CZipHeaderSignature	signature;
   	CZipFileHeader		header;

	// Central directory itself
	signature = aSig;
	while	( signature == zip_signature_centralfile ) {

		header.Cleanup();
        header.Read( signature, aReader );
        itsFileHeaders.push_back( header );

		signature.Cleanup();
		signature.Read( aReader );
    }

	// Digital signature - this might have been omitted by some zippers
	if	( signature == zip_signature_digital ) {
		itsDigitalSignature.Read( signature, aReader );

		signature.Cleanup();
		signature.Read( aReader );
    }

	// End of central directory marker
	if	( signature != zip_signature_eocd ) {
    	throw CError( aReader.Name(), "Missing end of central directory" );
    }
    else {
		itsEOCD.Read( signature, aReader );
    }
}

// --------------------------------------------------------------------
void	CZipCentralDir::Write				( CZipFileBlock_l & aBlocks, IZipStream & aWriter ) {
	CZipFileHeader_li	floop;
	CZipFileBlock_li	bloop;

    for	(	floop = itsFileHeaders.begin(), bloop = aBlocks.begin();
    		floop != itsFileHeaders.end();
            floop++, bloop++ ) {
    	(*floop).Write( *bloop, aWriter );
    }
	itsDigitalSignature.Write( aWriter );
    itsEOCD.Write( aWriter );
}

// --------------------------------------------------------------------
void	CZipCentralDir::Free	( void ) {
	CZipCentralDir::Clear();
}

// --------------------------------------------------------------------
void	CZipCentralDir::Clear	( void ) {
	CZipFileHeader_li	floop;
    for	(	floop = itsFileHeaders.begin();
    		floop != itsFileHeaders.end();
            floop++ ) {
    	(*floop).Cleanup();
    }
	itsFileHeaders.clear();
    itsDigitalSignature.Cleanup();
	itsEOCD.Cleanup();
}

// --------------------------------------------------------------------
const CZipFileHeader_l &		CZipCentralDir::FileHeaders		( void ) const { return itsFileHeaders; }
const CZipDigitalSignature &	CZipCentralDir::DigitalSignature( void ) const { return itsDigitalSignature; }
const CZipEOCD &				CZipCentralDir::EOCD			( void ) const { return itsEOCD; }
void							CZipCentralDir::Cleanup			( void ) { CZipCentralDir::Free(); }

// --------------------------------------------------------------------
void	CZipCentralDir::AddZipComment		( const char * aComment ) {
	itsEOCD.AddZipComment( aComment );
}

// --------------------------------------------------------------------
void	CZipCentralDir::AddFileComment		( const char * aFilter, const char * aComment ) {
	CZipFileHeader_li	floop;
	for	( 	floop =  itsFileHeaders.begin();
            floop != itsFileHeaders.end();
            floop++ ) {
		if	( (*floop).Filter( aFilter ) ) {
        	(*floop).AddFileComment( aComment );
        }
    }
}

// --------------------------------------------------------------------
void	CZipCentralDir::DelDirectoryEntry	( const char * aFilter ) {
	CZipFileHeader_li	floop;

	floop = itsFileHeaders.begin();
    while	( floop != itsFileHeaders.end() ) {
		if	( (*floop).Filter( aFilter ) ) {
			(*floop).Cleanup();
            itsFileHeaders.erase( floop );
        	floop = itsFileHeaders.begin();
        }
        else {
        	floop++;
        }
    }
}

// --------------------------------------------------------------------
void	CZipCentralDir::AddDirectoryEntry	( const CGz & aGz, const char * aDirName, time_t aTime ) {
	// Remove the entry just in case
	CZipCentralDir::DelDirectoryEntry( aDirName );

	// Add it to my list
    CZipFileHeader	myheader( aGz, aDirName, aTime );
    itsFileHeaders.push_back( myheader );
}

// --------------------------------------------------------------------
void	CZipCentralDir::AddDirectoryEntry	( const CZipFlags & aFg, const CZipFileBlock & aFb, const char * aDirName ) {
	// Remove the entry just in case
	CZipCentralDir::DelDirectoryEntry( aDirName );

	// Add it to my list
    CZipFileHeader	myheader( aFg, aFb, aDirName );
    itsFileHeaders.push_back( myheader );
}

// --------------------------------------------------------------------
void	CZipCentralDir::AddDirectoryEntry	( const CZipFlags & aFg, const char * aLocalFile, const char * aDirName ) {
	// Remove the entry just in case
	CZipCentralDir::DelDirectoryEntry( aDirName );

	// Add it to my list
    CZipFileHeader	myheader( aFg, aLocalFile, aDirName );
    itsFileHeaders.push_back( myheader );
}

// --------------------------------------------------------------------
dword_t	CZipCentralDir::ZippedSize	( void ) const {
	CZipFileHeader_lci	floop;
    dword_t				zize	= 0;
    for	(	floop = itsFileHeaders.begin();
    		floop != itsFileHeaders.end();
            floop++ ) {
    	zize = zize + (*floop).ZippedSize();
    }
	zize = zize + itsDigitalSignature.ZippedSize();
    zize = zize + itsEOCD.ZippedSize();
    return zize;
}

// --------------------------------------------------------------------
// public:	Update contents of the central directory
// --------------------------------------------------------------------
void	CZipCentralDir::Update	( const CZipFileBlock_l & aFbl ) {
	dword_t				fboffset	= 0;
    CZipFileBlock_lci	fbloop;
	CZipFileHeader_li	cdloop;

    for	(	fbloop =  aFbl.begin();
    		fbloop != aFbl.end();
            fbloop++ ) {
		// Find the entry in the central directory
	    for	(	cdloop = itsFileHeaders.begin();
    			cdloop != itsFileHeaders.end();
        	    cdloop++ ) {
			if	( (*fbloop).LocalFileHeader().FileName() == (const char *)(*cdloop).FileName().Data() ) {

				// Update the header offset
				(*cdloop).LocalHeaderOffset( fboffset );

				// If available, update the CRC32 and the compressed size as well
                if	( (*fbloop).LocalFileHeader().CompressedSize().Value() > 0 ) {
					(*cdloop).CompressedSize( (*fbloop).LocalFileHeader().CompressedSize().Value() );
                }

                if	( (*fbloop).LocalFileHeader().Crc32().Crc32() != 0 ) {
					(*cdloop).Crc32( (*fbloop).LocalFileHeader().Crc32().Crc32() );
                }

            	break;
            }
    	}
        fboffset = fboffset + (*fbloop).ZippedSize();
    }

	// Update the EOCD
	itsEOCD.DiskNumber					( 0 );
	itsEOCD.CentralDirDiskNumber		( 0 );
	itsEOCD.CentralDirDiskEntryCount	( (word_t)itsFileHeaders.size() );
	itsEOCD.CentralDirTotalEntryCount	( (word_t)itsFileHeaders.size() );
	itsEOCD.CentralDirOffset			( fboffset );

	// Calculate central directory size
	fboffset = 0;
    for	(	cdloop = itsFileHeaders.begin();
   			cdloop != itsFileHeaders.end();
       	    cdloop++ ) {
    	fboffset = fboffset + (*cdloop).ZippedSize();
	}

	itsEOCD.CentralDirSize				( fboffset );
}

// --------------------------------------------------------------------
// EOF:	CZipCentralDir.cpp
// --------------------------------------------------------------------
