// --------------------------------------------------------------------
// CZipStreamTar.cpp
// Whatis:	Some flags
// Authors:	Esko 'Varpu' Ilola	EIL
// History:	EIL	24-FEB-2003		Created	this source
// --------------------------------------------------------------------
#include	"CZipStreamTar.hxx"
#include	"CError.hxx"

// --------------------------------------------------------------------
// Definition of the TAR header record
// --------------------------------------------------------------------
typedef	struct {
    char    name	[100];
    char    mode	[8];
    char    uid		[8];
    char    gid		[8];
    char    size	[12];
    char    mtime	[12];
    char    chksum	[8];
    char    linkflag[1];
    char    linkname[100];
    char    magic	[8];
    char    uname	[32];
    char    gname	[32];
    char    devmajor[8];
	char    devminor[8];
}	tar_header_t;

// --------------------------------------------------------------------
// public:	Constructors
// --------------------------------------------------------------------
CTarFile::CTarFile	( 	const char *	aLocalFile,
    					const char *	aTarName ) {
	CTarFile::Cleanup();
    itsLocalFile	= ::my_private_strdup( aLocalFile );
    itsTarName	 	= ::my_private_strdup( aTarName );
}

// --------------------------------------------------------------------
CTarFile::CTarFile	( 	const byte_t *	aDataBlock,
    					dword_t			aDataSize,
            	        const char *	aTarName ) {
	CTarFile::Cleanup();
    itsDataBlock	= aDataBlock;
    itsDataSize		= aDataSize;
    itsTarName	 	= ::my_private_strdup( aTarName );
}

// --------------------------------------------------------------------
// public:	Destructor
// --------------------------------------------------------------------
CTarFile::~CTarFile	() {
	CTarFile::Free();
}

// --------------------------------------------------------------------
// public:	Copy constructor and assignment operator
// --------------------------------------------------------------------
CTarFile::CTarFile				( const CTarFile & aC ) {
	CTarFile::Cleanup();
	itsLocalFile	= aC.LocalFile() ? ::my_private_strdup( aC.LocalFile() ) : NULL;
    itsTarName		= ::my_private_strdup( aC.TarName() );
    itsDataBlock	= aC.DataBlock();
    itsDataSize		= aC.DataSize();
}

// --------------------------------------------------------------------
CTarFile & CTarFile::operator =	( const CTarFile & aC ) {
	CTarFile::Free();
	itsLocalFile	= aC.LocalFile() ? ::my_private_strdup( aC.LocalFile() ) : NULL;
    itsTarName		= ::my_private_strdup( aC.TarName() );
    itsDataBlock	= aC.DataBlock();
    itsDataSize		= aC.DataSize();
	return *this;
}

// --------------------------------------------------------------------
// public:	Accessing instance data
// --------------------------------------------------------------------
const char *		CTarFile::LocalFile	( void ) const { return itsLocalFile ; }
const char *		CTarFile::TarName   ( void ) const { return itsTarName ; }
const byte_t *		CTarFile::DataBlock ( void ) const { return itsDataBlock ; }
dword_t				CTarFile::DataSize  ( void ) const { return itsDataSize ; }

// --------------------------------------------------------------------
// private:	Private helpers
// --------------------------------------------------------------------
void	CTarFile::Free ( void ) {
	if	( itsTarName )		delete [] itsTarName;
    if	( itsLocalFile )	delete [] itsLocalFile;
	CTarFile::Cleanup();
}

// --------------------------------------------------------------------
void	CTarFile::Cleanup ( void ) {
    itsLocalFile	= NULL;
    itsTarName		= NULL;
    itsDataBlock	= NULL;
    itsDataSize		= 0L;
}

// --------------------------------------------------------------------
// public:	Constructor
// --------------------------------------------------------------------
CZipStreamTar::CZipStreamTar(	const CTarFile_l & aList ) {
	CZipStreamTar::Cleanup();
	itsTarFileList = aList;
    itsTarFileLoop = itsTarFileList.begin();
	itsDataOffs	   = sizeof( itsDataBuf );
}

// --------------------------------------------------------------------
// public:	Destructor
// --------------------------------------------------------------------
CZipStreamTar::~CZipStreamTar( ) {
	CZipStreamTar::Free();
}

// --------------------------------------------------------------------
// public:	Virtual Interface
// --------------------------------------------------------------------
bool	CZipStreamTar::CanSeek		( void ) {
	return	false;
}

// --------------------------------------------------------------------
bool	CZipStreamTar::CanRead		( void ) {
	return	true;
}

// --------------------------------------------------------------------
bool	CZipStreamTar::CanWrite	( void ) {
	return	false;
}

// --------------------------------------------------------------------
void	CZipStreamTar::Seek		( int4_t aOffs, zip_stream_seek_t aType ) {
	throw CError( "Unable to seek tar stream" );
}

// --------------------------------------------------------------------
int4_t	CZipStreamTar::Tell		( void ) {
	throw CError( "Unable to seek tar stream" );
}

// --------------------------------------------------------------------
bool	CZipStreamTar::Eof			( void ) {
    if	( itsDataOffs < sizeof( itsDataBuf ) )		return false;
    if	( itsDataLeft > 0 )							return false;
    if	( itsStream ) {
    	if	( ! feof( itsStream ) )					return false;
    }
	if  ( itsTarFileLoop != itsTarFileList.end() )	return false;
	return true;
}

// --------------------------------------------------------------------
dword_t	CZipStreamTar::Read		( void * aData, dword_t aLen ) {
	dword_t	mysize	= 0;

    while	( ( mysize < aLen ) && ( ! CZipStreamTar::Eof() ) ) {
    	if		( itsDataOffs < sizeof( itsDataBuf ) ) {
        	((char *)aData)[mysize++] = itsDataBuf[itsDataOffs++];
        }
        else {
			CZipStreamTar::Refill();
        }
	}
	return	mysize;
}

// --------------------------------------------------------------------
void	CZipStreamTar::Write		( const void * aData, dword_t aLen ) {
	throw CError( "Unable to write tar stream" );
}

// --------------------------------------------------------------------
dword_t	CZipStreamTar::Size		( void ) {
	throw CError( "Can't get size" );
}

// --------------------------------------------------------------------
const char *	CZipStreamTar::Name( void ) {
	return	itsTarName;
}

// --------------------------------------------------------------------
void			CZipStreamTar::FillNote	( CZipNotification * aNote ) {
	if	( aNote ) {
		aNote->FileName		( itsTarName ? itsTarName : "" );
        aNote->FilesDone	( itsCounter );
        aNote->FileBytes	( itsDataSize );
        aNote->FileBytesDone( itsDataDone );
    }
}

// --------------------------------------------------------------------
// private:	Private helpers
// --------------------------------------------------------------------
void	CZipStreamTar::Refill	( void ) {
	// Can we use existing stream ?
    if	( itsStream ) {
    	if	( ! feof( itsStream ) ) {
        	::memset( itsDataBuf, 0, sizeof( itsDataBuf ) );
            ::fread(  itsDataBuf, 1, sizeof( itsDataBuf ), itsStream );
			itsDataDone += sizeof( itsDataBuf );
            itsDataOffs = 0;
            return;
        }
    }

    // Can we use existing data block ?
    if		( itsDataLeft > sizeof( itsDataBuf ) ) {
    	::memcpy( itsDataBuf, itsDataPtr, sizeof( itsDataBuf ) );
        itsDataLeft -= sizeof( itsDataBuf );
        itsDataPtr	+= sizeof( itsDataBuf );
		itsDataDone += sizeof( itsDataBuf );
        itsDataOffs = 0;
        return;
    }
    else if	( itsDataLeft > 0 ) {
       	::memset( itsDataBuf, 0, sizeof( itsDataBuf ) );
    	::memcpy( itsDataBuf, itsDataPtr, itsDataLeft );
		itsDataDone += itsDataLeft;
        itsDataLeft = 0;
        itsDataPtr	= NULL;
        itsDataOffs = 0;
        return;
    }

	// Clean it up !
    if	( itsStream )	::fclose( itsStream );
    itsStream = NULL;

    // Do we have entries left in the list ?
	if  ( itsTarFileLoop != itsTarFileList.end() ) {
        itsTarName = (*itsTarFileLoop).TarName();
		if	( (*itsTarFileLoop).LocalFile() ) {
			CZipStreamTar::PrepareHeader( (*itsTarFileLoop).LocalFile() );
        }
        else {
			CZipStreamTar::PrepareHeader( 	(*itsTarFileLoop).DataBlock(),
            								(*itsTarFileLoop).DataSize() );
        }
		itsCounter++;
        itsTarFileLoop++;
    }
}

// --------------------------------------------------------------------
void	CZipStreamTar::PrepareHeader	( const char * aFileName ) {

    struct stat	mystat;

	if	( ::strlen( itsTarName ) > 99 ) {
    	throw CError( itsTarName, "Too long" );
    }
    if	( ::stat( aFileName, &mystat ) != 0 ) {
    	throw CError( aFileName, ::strerror( errno ) );
    }

	CZipStreamTar::MakeHeader( mystat.st_size, mystat.st_mtime );

    itsStream = ::fopen( aFileName, "rb" );
    if	( ! itsStream ) {
    	throw CError( aFileName, ::strerror( errno ) );
    }
    itsDataSize		= mystat.st_size;
    itsDataDone		= 0;
	itsDataPtr		= NULL;
	itsDataLeft		= 0L;
    itsDataOffs = 0;
}

// --------------------------------------------------------------------
void	CZipStreamTar::PrepareHeader	( const byte_t * aData, dword_t aBl ) {
	if	( ::strlen( itsTarName ) > 99 ) {
    	throw CError( itsTarName, "Too long" );
    }

	CZipStreamTar::MakeHeader( aBl, ::time( NULL ) );

    itsDataSize		= aBl;
    itsDataDone		= 0;
	itsStream		= NULL;
	itsDataPtr		= aData;
	itsDataLeft		= aBl;
    itsDataOffs		= 0;
}

// --------------------------------------------------------------------
void	CZipStreamTar::MakeHeader		( dword_t aSize, time_t aTime ) {
	tar_header_t *	th;

    th = (tar_header_t *)itsDataBuf;

    // Clear the entire data buffer
    ::memset( itsDataBuf, 0, sizeof( itsDataBuf ) );

    // Put my file name in the buffer - a simple strcpy is fine here
    ::strcpy( th->name, itsTarName );

	// Put some fixed values - these can't be determined in MsDos
    ::strcpy( th->mode, "0000666" );
    ::strcpy( th->uid,  "0000000" );
    ::strcpy( th->gid,  "0000000" );

    // Put in the size and the time
    ::sprintf( th->size,  "000000000000%o", aSize );
    while	( ::strlen( th->size ) > 11 ) ::strcpy( th->size, th->size + 1 );
    ::sprintf( th->mtime, "000000000000%o", aTime );
    while	( ::strlen( th->mtime ) > 11 ) ::strcpy( th->mtime, th->mtime + 1 );

	// Prepare for checksum calculation and calculate it
    ::strcpy( th->chksum, "        " );
    ::sprintf( th->chksum, "00000000%o", CZipStreamTar::Checksum() );
    while	( ::strlen( th->chksum ) > 7 ) ::strcpy( th->chksum, th->chksum + 1 );
}

// --------------------------------------------------------------------
dword_t		CZipStreamTar::Checksum	( void ) {
	dword_t	sum = 0;
    for	( int i = 0; i < sizeof( itsDataBuf ); i++ ) {
    	sum = sum + ((byte_t)itsDataBuf[i]);
    }
    return sum;
}

// --------------------------------------------------------------------
void	CZipStreamTar::Free	( void ) {
	if	( itsStream )	::fclose( itsStream );
    CZipStreamTar::Cleanup();
}

// --------------------------------------------------------------------
void	CZipStreamTar::Cleanup	( void ) {
	::memset( itsDataBuf, 0, sizeof( itsDataBuf ) );
	itsDataOffs		= 0L;
	itsStream		= NULL;
	itsDataPtr		= NULL;
	itsDataLeft		= 0L;
	itsTarName		= NULL;
    itsTarFileList.clear();
	itsCounter		= 0;
	itsDataSize		= 0;
	itsDataDone		= 0;
}

// --------------------------------------------------------------------
// EOF:	CZipStreamTar.cpp
// --------------------------------------------------------------------
