// --------------------------------------------------------------------
// FurPop.cxx
// Whatis:  A process that serves FurPush connections
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 02-FEB-2003     Created this source
// --------------------------------------------------------------------
#include    "CError.hxx"
#include    "CTcpSocket.hxx"
#include	"CCpuMutex.hxx"
#include    "CImportFile.hxx"
#include	"CTableFile.hxx"
#include	"CTablePackFile.hxx"
#include	"CFileScore.hxx"

//---------------------------------------------------------------------------
// Defines for teh service
//---------------------------------------------------------------------------
#define	FURPUSH_PORT		(unsigned short)31075
#define	INVALID_CSUM		0xffffffff
#define	MAX_THREAD_COUNT	32

//---------------------------------------------------------------------------
// local:	Class used by the thread
//---------------------------------------------------------------------------
class	CUploadData {

	//-----------------------------------------------------------------------
	public:		// Constructor and destructor
	//-----------------------------------------------------------------------
	CUploadData		( CTcpSocket * aSock );
	~CUploadData 	( );
	void			Process			(	void );


	//-----------------------------------------------------------------------
	private:	// Copy constructor and assignment operator disabled
	//-----------------------------------------------------------------------
	CUploadData				( const CUploadData & );
	CUploadData operator =	( const CUploadData & );

	//-----------------------------------------------------------------------
	private:	// Private helpers
	//-----------------------------------------------------------------------
	dword_t			CalcChecksum	( 	dword_t 		aLen );
	char			ReceiveByte		( 	void );
	void			ReceiveString	(	char *			aB,
										int				aL,
										char			aTc );
	void			GetFileDetails	(	void );
	void			ReceiveFileData	(	void );
	void			SetPackAuthor	(	dword_t			aPackId );
	void			ProcessFile		(	void );

	//-----------------------------------------------------------------------
	private:	// Instance data
	//-----------------------------------------------------------------------
	char			itsFileName		[1024];
	CTcpSocket *	itsSock;
	dword_t			itsClSize;
	time_t			itsClTime;
};

//---------------------------------------------------------------------------
// local:	Common data for threads
//---------------------------------------------------------------------------
static	CCpuMutex	__data_mutex;
static	CCpuMutex	__proc_mutex;
static	int			__number_of_threads	= 0;
static	char *		__proclist[MAX_THREAD_COUNT] = { NULL };

//---------------------------------------------------------------------------
// local:	Constructor for the thread data
//---------------------------------------------------------------------------
CUploadData::CUploadData ( CTcpSocket * aSock ) {
	itsSock		= aSock;
	itsClSize	= 0;
	itsClTime	= 0;
	::memset( itsFileName, 0, sizeof( itsFileName ) );

	// Set us on a free proclist slot
	__data_mutex.Lock();
	for	( int i = 0; i < MAX_THREAD_COUNT; i++ ) {
		if	( __proclist[i] == NULL ) {
			__proclist[i] = itsFileName;
			break;
		}
	}
	__number_of_threads++;
	__data_mutex.Unlock();
}

//---------------------------------------------------------------------------
// local:	Destructor for the thread data
//---------------------------------------------------------------------------
CUploadData::~CUploadData ( ) {
	if	( itsSock ) delete itsSock;

	// Remove us from the proclist slot
	__data_mutex.Lock();
	for	( int i = 0; i < MAX_THREAD_COUNT; i++ ) {
		if	( __proclist[i] == itsFileName ) {
			__proclist[i] = NULL;
		}
	}
	__number_of_threads--;
	__data_mutex.Unlock();
}

//---------------------------------------------------------------------------
// local:	Calculate checksum for file.
//---------------------------------------------------------------------------
dword_t		CUploadData::CalcChecksum	( dword_t	aLen ) {
	dword_t	csum = 0;
    dword_t	offs = 0;
	int		rchr = 0;
	int		i;
	FILE *	strm = ::fopen( itsFileName, "rb" );
	byte_t	buff	[2048];

	if	( ! strm )	throw	CError( itsFileName, ::strerror( errno ) );

	offs = 0;
	while	( offs < aLen ) {
		if	( aLen - offs < 2048 ) {
	       	rchr = fgetc( strm );
    	    csum = csum + (dword_t)rchr;
			offs++;
		}
		else {
			::fread( buff, 1, 2048, strm );
			for	( i = 0; i < 2048; i++ ) {
	    	    csum = csum + (dword_t)buff[i];
			}
			offs = offs + 2048;
		}
	}


	::fclose( strm );

	return	csum;
}


// --------------------------------------------------------------------
// local:	Receive one byte
// --------------------------------------------------------------------
char	CUploadData::ReceiveByte	( void ) {
	int		byte = 0;

	do	{
		byte = itsSock->Read();
		if	( byte == SOCKET_READ_ERROR )		throw 1;
		if	( byte == SOCKET_READ_DISCONNECT )	throw 1;
	}	while	( byte == SOCKET_READ_IDLE );

	return	(char)byte;
}

// --------------------------------------------------------------------
// local:	Receive string up to terminating char (disposed)
// --------------------------------------------------------------------
void	CUploadData::ReceiveString	( 	char *			aB,
										int				aL,
										char			aTc ) {
	char	chr;
	int		off	= 0;
	do	{
		chr = CUploadData::ReceiveByte();
		if	( chr != aTc ) {
			aB[off++] = chr;
			if	( off > aL + 2 )	throw 1;
		}
		else {
			aB[off] = 0;
		}
	}	while	( chr != aTc );
}


// --------------------------------------------------------------------
// Receiving the negotiation package
// The format of the package is as follows:
// filename,filesize,y,m,d,H,M,S<cr>(0x0d)
// --------------------------------------------------------------------
void	CUploadData::GetFileDetails	( void ) {
	char	fname	[1024];
	char	fsiz	[10];
	char	year	[6];
	char	month	[6];
	char	day		[6];
	char	hour	[6];
	char	min		[6];
	char	sec		[6];
	int		i;
	char *	fpt;
	struct	stat	mystat;
	struct	tm		mytm;
	time_t	tnow;


	CUploadData::ReceiveString( fname, 	1024,	',' );
	CUploadData::ReceiveString( fsiz,	10, 	',' );
	CUploadData::ReceiveString( year,	6, 		',' );
	CUploadData::ReceiveString( month,	6, 		',' );
	CUploadData::ReceiveString( day,	6, 		',' );
	CUploadData::ReceiveString( hour,	6, 		',' );
	CUploadData::ReceiveString( min,	6, 		',' );
	CUploadData::ReceiveString( sec,	6, 		'\x0d' );

	// The files reside in the upload directory - permanently bound
	// to /usr/upload
	::strcpy( itsFileName, "/usr/upload/" );
	::strcat( itsFileName, fname );

	tnow = ::time( NULL );
	::memcpy( &mytm, ::gmtime( &tnow ), sizeof( mytm ) );

	mytm.tm_year	= ::atoi( year ) - 1900;
	mytm.tm_mon		= ::atoi( month ) - 1;
	mytm.tm_mday	= ::atoi( day );
	mytm.tm_hour	= ::atoi( hour );
	mytm.tm_min		= ::atoi( min );
	mytm.tm_sec		= ::atoi( sec );

	itsClTime = ::mktime( &mytm );
	itsClSize = (dword_t)::atol( fsiz );


	// Check for valid file name
	if	( fname[0] == '.' ) {
		itsSock->Write( "XInvalid characters in file name\x0d" );
		itsSock->Flush();
		throw 1;
	}

	for	( fpt = fname; *fpt; fpt++ ) {
		if	( ::strchr( "\"$%&/=?\\+@<>|", *fpt ) ) {
			itsSock->Write( "XInvalid characters in file name\x0d" );
			itsSock->Flush();
			throw 1;
		}
	}

	// Check for a zip
	fpt = ::strstr( itsFileName, ".zip" );
	if	( ! fpt ) {
		itsSock->Write( "XInvalid file name (not .zip)\x0d" );
		itsSock->Flush();
		throw 1;
	}
	if	( fpt[4] != 0 ) {
		itsSock->Write( "XInvalid file name (not .zip)\x0d" );
		itsSock->Flush();
		throw 1;
	}

	// Already processing this file ?
	__data_mutex.Lock();
	for	( i = 0; i < MAX_THREAD_COUNT; i++ ) {
		if	( __proclist[i] == NULL )					continue;
		if	( __proclist[i] == itsFileName )			continue;
		if	( ::strcmp( __proclist[i], itsFileName ) )	continue;
		itsSock->Write( "XAlready processing the file\x0d" );
		itsSock->Flush();
		__data_mutex.Unlock();
		throw 1;
	}
	__data_mutex.Unlock();

	if	( ! ::stat( itsFileName, &mystat ) ) {
		if	( ::time( NULL ) - mystat.st_mtime > 3600 * 24 ) {
			::unlink( itsFileName );
		}
		else {
			itsSock->Write( "F" );
			itsSock->Write( (dword_t)(mystat.st_size) );
			itsSock->Write( "," );
			itsSock->Write( CUploadData::CalcChecksum( (dword_t)(mystat.st_size) ) );
		}
	}
	itsSock->Write( "\x0d" );
	itsSock->Flush();
}


// --------------------------------------------------------------------
// local:	Receive the file
// --------------------------------------------------------------------
void	CUploadData::ReceiveFileData	( void ) {
	dword_t		woffset	= 0;
	FILE *		strm;

	// Try to open the file in read/write mode
	strm = ::fopen( itsFileName, "r+" );
	if	( ! strm ) {
		strm = ::fopen( itsFileName, "w" );
		if	( ! strm ) {
			throw	1;
		}
	}
	else {
		::fseek( strm, 0L, SEEK_END );
		woffset = ::ftell( strm );
	}

	try {
		// Read until we meet the client data amount
		while	( woffset < itsClSize ) {
			::fputc( CUploadData::ReceiveByte(), strm );
			woffset++;
		}
		::fclose( strm );
	}

	catch	( ... ) {
		::fclose( strm );
		throw;
	}

	itsSock->Write( CUploadData::CalcChecksum( itsClSize ) );
	itsSock->Write( "\x0d" );
	itsSock->Flush();

	// Client should now answer '1' or '0' (keep/delete)
	if	( CUploadData::ReceiveByte() != '1' ) {
		::unlink( itsFileName );
		throw	1;
	}
}

// --------------------------------------------------------------------
// local:	Set up author information on a package
// --------------------------------------------------------------------
void	CUploadData::SetPackAuthor	( dword_t	aPackId ) {
	CMySqlConnect		db( "quest", "", "UTCMS" );
	CMySqlWhere			w;
	CTableFile			file;
	CTableType			type;
	CTablePack			pack;
	CTablePackFile		packfile;
	data_packfile_tl	pflist;
	data_packfile_tli	pfloop;
	data_file_tl		flist;
	data_file_tl		list;
	data_file_tli		loop;
	data_file_t			data;
	data_type_tl		tlist;
	data_type_t			tdata;
	dword_t				authorid	= 0;
	dword_t				authorprio	= 0;
	
	w << "packfile_pack=" << aPackId;
	pflist = packfile.Select( db, w );

	// Now, get all files in this package
	for	( pfloop = pflist.begin(); pfloop != pflist.end(); pfloop++ ) {
		w = "";
		w << "file_idnt=" << (*pfloop).packfile_file;
		flist = file.Select( db, w );
		if	( flist.size() > 0 ) {
			list.push_back( *(flist.begin()) );

			CFileScore	myscore( (*pfloop).packfile_file );
			myscore.Update();
		}
	}

	// Search a file entry that has an author and the file is used in this pack only
	// Also, check the file type priority
	for	( loop = list.begin(); loop != list.end(); loop++ ) {
		if	( (*loop).file_auth > 0 ) {
			w = "";
			w << "packfile_file=" << (*loop).file_idnt;
			if	( packfile.Count( db, w ) == 1 ) {
				w = "";
				w << "type_idnt=" << (*loop).file_type;
				tlist = type.Select( db, w );

				if	( tlist.size() > 0 ) {
					tdata = *(tlist.begin());
					if	( tdata.type_apri >= authorprio ) {
						authorid	= (*loop).file_auth;
						authorprio	= tdata.type_apri;
					}
				}
			}
		}
	}

	// Did we find any ?
	if	( authorid > 0 ) {
		for	( loop = list.begin(); loop != list.end(); loop++ ) {
			if	( (*loop).file_auth == 0 ) {
				data = *loop;
				data.file_auth = authorid;
				file.Update( db, &data );
			}
		}
		CFileScore	myscore( 0 );
		myscore.AuthorScore( authorid );
		myscore.Update();
	}
}

// --------------------------------------------------------------------
// local:	Process the file
// --------------------------------------------------------------------
void	CUploadData::ProcessFile	( void ) {
	try {
		__proc_mutex.Lock();

		CImportFile	import( "utcms", true );
		import.Import( itsFileName );
	
		data_pack_tl	list = import.PackList();
		data_pack_tli	loop;
	
		// Set up author information on each package
		for	( loop = list.begin(); loop != list.end(); loop++ ) {
			CUploadData::SetPackAuthor( (*loop).pack_idnt );
		}
		::unlink( itsFileName );

		__proc_mutex.Unlock();
	}

	catch	( ... ) {
		__proc_mutex.Unlock();
	}
}

// --------------------------------------------------------------------
// local:	The process
// --------------------------------------------------------------------
void	CUploadData::Process	( void ) {
	CUploadData::GetFileDetails	();
	CUploadData::ReceiveFileData();
	delete itsSock;
	itsSock = NULL;
	CUploadData::ProcessFile();
}

// --------------------------------------------------------------------
// local:	Process the client
// --------------------------------------------------------------------
static	void *	__client_thread ( void * aArg ) {
	CUploadData *	data = (CUploadData *)aArg;

	try {
		data->Process();
		delete	data;
	}

	catch ( ... ) {
		delete	data;
	}

	// Finally before exiting release the slot and delete the data
	return	NULL;
}

// --------------------------------------------------------------------
// local:	The process
// --------------------------------------------------------------------
static	void	__process	( void ) {
	CTcpSocket			mysocket( (socket_type_t)socket_type_stream );
	CTcpSocket *		client = NULL;
	pthread_t			mythread;

	mysocket.Bind( FURPUSH_PORT );
	mysocket.Listen();
	::chdir( "/usr/upload" );

	do	{
		try	{
			client = mysocket.Accept();
			if	( client ) {
				if	( __number_of_threads < MAX_THREAD_COUNT ) {
					CUploadData *	mydata = new CUploadData( client );
					mythread = 0;
					::pthread_create( &mythread, NULL, __client_thread, (void *)mydata );
					::pthread_detach( mythread );
				}
				else {
					delete client;
				}
			}
			else {
				SLEEP( 1000 );
			}
		}
		catch	( ... ) {
		}
	}	while	( true );
}

// --------------------------------------------------------------------
// public:  Program entrypoint
// --------------------------------------------------------------------
extern  int     main( 	int aAc, char ** aAv ) {
	::daemon( 1, 0 );
	__process();
    return  0;
}

// --------------------------------------------------------------------
// EOF: Motd.cxx
// --------------------------------------------------------------------
