// --------------------------------------------------------------------
// Storage.cxx
// Whatis:  Manipulates the file storage @ UTCMS
// Authors: Esko 'Varpu' Ilola  EIL
// History: EIL 02-JUN-2002     Created this source
// --------------------------------------------------------------------
#include    "CError.hxx"
#include    "CStorage.hxx"
#include    "CIdentifyFile.hxx"
#include    "CAuthor.hxx"
#include    "CTableAuth.hxx"
#include    "CTableFile.hxx"
#include    "CTablePack.hxx"
#include    "CTablePackFile.hxx"
#include    "CTableType.hxx"
#include	"CMySqlConnect.hxx"
#include	"CMySqlWhere.hxx"
#include	"CFileScore.hxx"

// --------------------------------------------------------------------
// Usage:
// --------------------------------------------------------------------
static const char *	__usage[] = {
	"This is a program to manage the storage",
	"Usage:    Storage command arguments ...",
	"command:  write <file> <name>  Saves <file> into storage with <name>",
	"          read  <ident> <file> Retrieves <ident> to file from storage",
	"          info  <file>         Lists information of <file>",
	"          del   <ident>        Removes <idnt> from system",
	"          ren   <ident> <name> Renames to <name>",
	"          era   <wild>         Erases files matching the <wild>",
	"          erpk  <pakid>        Erases all files of a package",
	"          pack  <name>         Show package id(s)",
	"          cleanonunr           Erases all packages that have no UNR files",
	"          fixtypes             Fix file typing after rule changes",
	"          fixpsiz              Fix packed sizes",
	"          fixauth              Fix author names",
	"          fixscore             Fix the scores",
	"          fixpacknames         Fix package names",
	"          test                 Diverse testing",
	NULL
};

// --------------------------------------------------------------------
// local:	Help
// --------------------------------------------------------------------
static	void	__help( void ) {
	const char ** h = __usage;
	while	( *h )	::printf( "%s\n", *(h++) );
}

// --------------------------------------------------------------------
// local:	Set up author information on a package
// --------------------------------------------------------------------
static	void	__set_pack_author	( dword_t	aPackId ) {
	CMySqlConnect		db( "quest", "", "UTCMS" );
	CMySqlWhere			w;
	CTablePackFile		packfile;
	CTableFile			file;
	CTableType			type;
	data_packfile_tl	pflist;
	data_packfile_tli	pfloop;
	data_file_tl		flist;
	data_file_tl		list;
	data_file_tli		loop;
	data_file_t			data;
	data_type_tl		tlist;
	data_type_t			tdata;
	dword_t				authorid	= 0;
	dword_t				authorprio	= 0;
	bool				firstone	= true;
	

	w << "packfile_pack=" << aPackId;
	pflist = packfile.Select( db, w );

	// Now, get all files in this package
	for	( pfloop = pflist.begin(); pfloop != pflist.end(); pfloop++ ) {
		w = "";
		w << "file_idnt=" << (*pfloop).packfile_file;
		flist = file.Select( db, w );
		if	( flist.size() > 0 ) {
			list.push_back( *(flist.begin()) );
		}
	}

	// Search a file entry that has an author and the file is used in this pack only
	// Also, check the file type priority
	for	( loop = list.begin(); loop != list.end(); loop++ ) {
		if	( (*loop).file_auth > 0 ) {
			w = "";
			w << "packfile_file=" << (*loop).file_idnt;
			if	( packfile.Count( db, w ) == 1 ) {
				w = "";
				w << "type_idnt=" << (*loop).file_type;
				tlist = type.Select( db, w );

				if	( tlist.size() > 0 ) {
					tdata = *(tlist.begin());
					if	( tdata.type_apri >= authorprio ) {
						authorid	= (*loop).file_auth;
						authorprio	= tdata.type_apri;
					}
				}
			}
		}
	}

	// Did we find any ?
	if	( authorid > 0 ) {
		for	( loop = list.begin(); loop != list.end(); loop++ ) {
			if	( (*loop).file_auth == 0 ) {
				if	( firstone )	::printf( "\n\t%s.%s\n", (*loop).file_name, (*loop).file_suff );
				else				::printf( "\t%s.%s\n", (*loop).file_name, (*loop).file_suff );
				data = *loop;
				data.file_auth = authorid;
				file.Update( db, &data );
			}
		}
	}
}

// --------------------------------------------------------------------
// local:	Information from info
// --------------------------------------------------------------------
static	void	__finfo ( data_file_t & aInfo ) {
	CMySqlConnect	db( "quest", "", "UTCMS" );
	CMySqlWhere		w;
	CTableType		type;
	CTableAuth		auth;
	data_type_tl	typelist;
	data_auth_tl	authlist;
	const char *	typeexpr = "Unknown";
	const char *	authexpr = "Unknown";


	w << "type_idnt=" << aInfo.file_type;
	typelist = type.Select( db, w );

	if	( typelist.size() > 0 ) {
		typeexpr = (*(typelist.begin())).type_name;
	}

	w = "";
	w << "auth_idnt=" << aInfo.file_auth;
	authlist = auth.Select( db, w );
	if	( authlist.size() > 0 ) {
		authexpr = (*(authlist.begin())).auth_name;
	}

	::printf( "Name       : %s\n", aInfo.file_name );
	::printf( "Suffix     : %s\n", aInfo.file_suff );
	::printf( "Identifier : %d\n", aInfo.file_idnt );
	::printf( "Author     : %d [%s]\n", aInfo.file_auth, authexpr );
	::printf( "Type       : %d [%s]\n", aInfo.file_type, typeexpr );
	::printf( "Type 1     : %s\n", aInfo.file_typ1 );
	::printf( "Type 2     : %s\n", aInfo.file_typ2 );
	::printf( "Version    : %d\n", aInfo.file_vers );
	::printf( "Checksum   : %s\n", aInfo.file_csum );
	::printf( "Created    : %s\n", ::my_time_string( aInfo.file_ctim ) );
	::printf( "Modified   : %s\n", ::my_time_string( aInfo.file_mtim ) );
	::printf( "Arrived    : %s\n", ::my_time_string( aInfo.file_atim ) );
	::printf( "Accessed   : %s\n", ::my_time_string( aInfo.file_ltim ) );
	::printf( "Size       : %d\n", aInfo.file_size );
	::printf( "Packed size: %d\n", aInfo.file_psiz );
	::printf( "Load count : %d\n", aInfo.file_lcnt );
	::printf( "Flags      : %d\n", aInfo.file_flag );
	::printf( "Storage    : %s\n", aInfo.file_data );
}

// --------------------------------------------------------------------
// local:	Write
// --------------------------------------------------------------------
static	void	__write	( int aAc, char ** aAv ) {
	CStorage	storage;
	if	( aAc < 4 )	throw CError( aAv[1], "Not enough arguments" );
	data_file_t	info = storage.Save( aAv[3], 0, aAv[2] );
	::printf( "----------------------------------------\n" );
	__finfo( info );
	::printf( "----------------------------------------\n" );
}

// --------------------------------------------------------------------
// local:	Read
// --------------------------------------------------------------------
static	void	__read	( int aAc, char ** aAv ) {
	CStorage	storage;
	if	( aAc < 4 )	throw CError( aAv[1], "Not enough arguments" );
	data_file_t	info = storage.Load( aAv[3], ::atol( aAv[2] ) );
	::printf( "----------------------------------------\n" );
	__finfo( info );
	::printf( "----------------------------------------\n" );
}

// --------------------------------------------------------------------
// local:	Info
// --------------------------------------------------------------------
static	void	__info	( const char * aIdnt ) {
	CStorage		storage;
	data_file_tl	list;
	data_file_tli	loop;

	char	file	[101];

	::my_strfit( file, sizeof( file ), aIdnt );
	::my_strfix( file );

	char *	suff = ::strrchr( file, '.' );
	if	( suff ) {
		*suff = 0;
		suff++;
	}
	else {
		suff = "";
	}

	list = storage.Find( file, suff );

	for	( loop = list.begin(); loop != list.end(); loop++ ) {
		::printf( "----------------------------------------\n" );
		__finfo( *loop );
	}
	::printf( "----------------------------------------\n" );
}

// --------------------------------------------------------------------
// local:	Delete
// --------------------------------------------------------------------
static	void	__del	( const char * aIdnt ) {
	CStorage	storage;
	storage.Delete( ::atol( aIdnt ) );
}

// --------------------------------------------------------------------
// local:	Erase
// --------------------------------------------------------------------
static	void	__era	( const char * aName ) {
	CStorage		storage;
	CMySqlConnect	db( "quest", "", "UTCMS" );
	CMySqlWhere		w;
	CTableFile		file;
	data_file_tl	list;
	data_file_tli	loop;


	w << "file_name like '" << aName << "'";
	list = file.Select( db, w );

	::printf( "Erasing %d files (%s)\n", (int)list.size(), aName );
	for	( loop = list.begin(); loop != list.end(); loop++ ) {

		::printf( "Erasing: %s\n", (*loop).file_name );
		storage.Delete( (*loop).file_idnt );
	}
}

// --------------------------------------------------------------------
// local:	Erase a package
// --------------------------------------------------------------------
static	void	__erpk	( dword_t aPack ) {
	CStorage			storage;
	CMySqlConnect		db( "quest", "", "UTCMS" );
	CMySqlWhere			w;
	CMySqlQuote			q;
	CTableFile			file;
	CTablePack			pack;
	CTablePackFile		packfile;
	data_packfile_tl	packfilelist;
	data_packfile_tli	packfileloop;
	data_file_tl		list;

	w = "";
	w << "packfile_pack=" << aPack;
	packfilelist = packfile.Select( db, w );

	for	( packfileloop = packfilelist.begin(); packfileloop != packfilelist.end(); packfileloop++ ) {
		w = "";
		w << "packfile_file=" << (*packfileloop).packfile_file;
		dword_t	count = packfile.Count( db, w );
		w = "";
		w << "file_idnt=" << (*packfileloop).packfile_file;
		if		( count == 1 ) {
			list = file.Select( db, w );
			if	( list.size() > 0 ) {
				::printf( 	"Erasing file: %d - %s.%s\n",
							aPack,
							(*(list.begin())).file_name,
							(*(list.begin())).file_suff );
				
				storage.Delete( (*(list.begin())).file_idnt );
			}
		}
		else if	( count > 1 ) {
			list = file.Select( db, w );
			if	( list.size() > 0 ) {
				::printf( 	"Erasing reference: %d - %s.%s\n",
							aPack,
							(*(list.begin())).file_name,
							(*(list.begin())).file_suff );
				w = "";
				w << "packfile_pack=" << aPack << " and ";
				w << "packfile_file=" << (*(list.begin())).file_idnt;
				packfile.Delete( db, w );
			}
		}
	}
	w = "";
	w << "pack_idnt='" << aPack << "'";
	pack.Delete( db, w );
}

// --------------------------------------------------------------------
// local:	Rename
// --------------------------------------------------------------------
static	void	__ren	( int aAc, char ** aAv ) {
	CStorage	storage;
	if	( aAc < 4 )	throw CError( aAv[1], "Not enough arguments" );
	else {
		char	file	[101];

		::my_strfit( file, sizeof( file ), aAv[3] );
		::my_strfix( file );

		char *	suff = ::strrchr( file, '.' );
		if	( suff ) {
			*suff = 0;
			suff++;
		}
		else {
			suff = "";
		}

		data_file_t	info = storage.Rename( ::atol( aAv[2] ), file, suff );
		::printf( "----------------------------------------\n" );
		__finfo( info );
		::printf( "----------------------------------------\n" );
	}
}

// --------------------------------------------------------------------
// local:	Removes packages that have no UNR files in them
// --------------------------------------------------------------------
static	void	__cleannonu	( void ) {
	CStorage		storage;
	CMySqlConnect	db( "quest", "", "UTCMS" );
	CMySqlWhere		w;
	CMySqlQuote		q;
	CTablePack		pack;
	CTablePackFile	packfile;
	CTableFile		file;

	data_pack_tl	packlist;
	data_pack_tli	packloop;

	w << "pack_idnt > 0";
	packlist = pack.Select( db, w );
	for	( packloop = packlist.begin(); packloop != packlist.end(); packloop++ ) {
		data_packfile_tl	packfilelist;
		data_packfile_tli	packfileloop;

		w = "";
		w << "packfile_pack=" << (*packloop).pack_idnt;
		packfilelist = packfile.Select( db, w );
		for	( packfileloop = packfilelist.begin(); packfileloop != packfilelist.end(); packfileloop++ ) {
			w = "";
			w << "file_idnt=" << (*packfileloop).packfile_file << " and (";
			w << "file_typ1='UNR' or ";
			w << "file_typ1='3DF' or ";
			w << "file_typ1='IMG' or ";
			w << "file_typ1='SND' or ";
			w << "file_typ1='MUS'";
			w << ")";
			if	( file.Count( db, w ) > 0 )	break;
		}
		if	( packfileloop == packfilelist.end() ) {
			__erpk( (*packloop).pack_idnt );
		}
	}
}

// --------------------------------------------------------------------
// local:	Tell the number of pack whose name is nnnn
// --------------------------------------------------------------------
static	void	__pack(	const char * aName ) {
	CMySqlConnect		db( "quest", "", "UTCMS" );
	CMySqlWhere			w;
	CTablePack			pack;
	data_pack_tl		packlist;
	data_pack_tli		packloop;

	w << "pack_name like '%" << aName << "%'";
	packlist = pack.Select( db, w );
	
	for	( packloop = packlist.begin(); packloop != packlist.end(); packloop++ ) {
		::printf( "%d %s\n", (*packloop).pack_idnt, (*packloop).pack_name );
	}
}

// --------------------------------------------------------------------
// local:	Make identification
// --------------------------------------------------------------------
static	void	__identifyit( 	data_file_t &	aData,
								const char * 	aName ) {
	try {
		CStorage		storage;
	
		storage.Load( aName, aData.file_idnt );
	
		CIdentifyFile	identify( aName );
	
		aData.file_type = identify.Type().type_idnt;
		::strcpy( aData.file_typ1, identify.Typ1() );
		::strcpy( aData.file_typ2, identify.Typ2() );
		::strcpy( aData.file_ctgr, identify.Ctgr() );
	}
	catch	( ... ) {
		::printf( "***WARNING*** Possible malicious file, file_idnt = %d\n", aData.file_idnt );
	}
}

// --------------------------------------------------------------------
// local:	Fix the file types when rules have changed - takes a long time
// --------------------------------------------------------------------
static	void	__fixtypes	( void ) {
	CMySqlConnect	db( "quest", "", "UTCMS" );
	CMySqlWhere		w;
	CTableFile		file;
	data_file_tl	list;
	data_file_t		data;
	data_file_t		info;
	dword_t			idnt;
	char			temp[1024];
	char			name[1024];

	// ----------------------------------------------------------------
	// Create teh temporary directory
	// ----------------------------------------------------------------
	::my_make_tempdir( temp, "." );

	// ----------------------------------------------------------------
	// We have to go through the files gracefully - one by one
	// because this is one HUGE flock !
	// ----------------------------------------------------------------
	for	( idnt = file.NextIdnt( db ); idnt > 0; idnt-- ) {

		// ------------------------------------------------------------
		// Get teh file
		// ------------------------------------------------------------
		w = "";
		w << "file_idnt=" << idnt;
		list = file.Select( db, w );
		if	( list.size() < 1 )	continue;

		// ------------------------------------------------------------
		// Get the actual file data
		// ------------------------------------------------------------
		data = *(list.begin());
		
		// ------------------------------------------------------------
		// Output some info
		// ------------------------------------------------------------
		::printf( 	"\r%8d - %24s.%-4s - ",
					idnt,
					data.file_name,
					data.file_suff );

		// ------------------------------------------------------------
		// Set up target file name
		// ------------------------------------------------------------
		::sprintf(	name, "%s/%s.%s",
					temp,
					data.file_name,
					data.file_suff );
		::printf( "%s\n", name );

		// ------------------------------------------------------------
		// Extract teh file
		// ------------------------------------------------------------
		::memcpy( &info, &data, sizeof( info ) );
		__identifyit( info, name );

		// ------------------------------------------------------------
		// Have the types changed somehow ?
		// ------------------------------------------------------------
		if	( ::memcmp( &info, &data, sizeof( info ) ) ) {
			::printf( "Updating:\n" );
			if	( info.file_type != data.file_type )			::printf( "\ttype %d -> %d\n", data.file_type, info.file_type );
			if	( ::strcmp( info.file_typ1, data.file_typ1 ) )	::printf( "\ttyp1 %s -> %s\n", data.file_typ1, info.file_typ1 );
			if	( ::strcmp( info.file_typ2, data.file_typ2 ) )	::printf( "\ttyp2 %s -> %s\n", data.file_typ2, info.file_typ2 );
			if	( ::strcmp( info.file_ctgr, data.file_ctgr ) )	::printf( "\tctgr %s -> %s\n", data.file_ctgr, info.file_ctgr );

			file.Update( db, &info );
		}

		// ------------------------------------------------------------
		// Remove teh file
		// ------------------------------------------------------------
		::unlink( name );
	}


	// ----------------------------------------------------------------
	// Clean up the mess
	// ----------------------------------------------------------------
	::my_erase_tempdir( temp );
}

// --------------------------------------------------------------------
// local:	Fix packed sizes of the files
// --------------------------------------------------------------------
static	void	__fixpsiz	( void ) {
	CMySqlConnect	db( "quest", "", "UTCMS" );
	CMySqlWhere		w;
	CTableFile		file;
	data_file_tl	list;
	data_file_t		data;
	struct stat		mystat;
	dword_t			idnt;

	// ----------------------------------------------------------------
	// We have to go through the files gracefully - one by one
	// because this is one HUGE flock !
	// ----------------------------------------------------------------
	for	( idnt = 0; idnt < file.NextIdnt( db ); idnt++ ) {

		// ------------------------------------------------------------
		// Progress indication
		// ------------------------------------------------------------
		::printf( "\r%d   ", idnt );

		// ------------------------------------------------------------
		// Get teh file
		// ------------------------------------------------------------
		w = "";
		w << "file_idnt=" << idnt;
		list = file.Select( db, w );
		if	( list.size() < 1 )	continue;

		// ------------------------------------------------------------
		// Get the actual file data
		// ------------------------------------------------------------
		data = *(list.begin());
		
		// ------------------------------------------------------------
		// Make a stat to the gzip
		// ------------------------------------------------------------
		if	( ::stat( data.file_data, &mystat ) ) {
			::printf( "Can't access %s\n", data.file_data );
			continue;
		}

		// Does the stored value differ from the actual value ?
		if	( mystat.st_size == data.file_psiz ) {
			continue;
		}

		// Report about the fix
		::printf( "%s %d -> %d\n", data.file_data, data.file_psiz, (dword_t)mystat.st_size );

		// Make teh fix
		data.file_psiz = (dword_t)mystat.st_size;
		file.Update( db, &data );

	}

}

// --------------------------------------------------------------------
// local:	Fix scores on all files
// --------------------------------------------------------------------
static	void	__fixscore	( void ) {
	CMySqlConnect	db( "quest", "", "UTCMS" );
	CMySqlWhere		w;
	CTableAuth		auth;
	CTableFile		file;
	data_auth_tl	alist;
	data_auth_t		adata;
	dword_t			aidnt;
	data_file_tl	flist;
	dword_t			fidnt;

	// Loop through all the files and try to fix their scores
	::printf( "Updating scores for files\n" );
	for	( fidnt = 1; fidnt < file.NextIdnt( db ); fidnt++ ) {
		w = "";
		w << "file_idnt=" << fidnt;
		flist = file.Select( db, w );
		if	( flist.size() > 0 ) {
			::printf( "\r%d  ", fidnt );

			CFileScore	myscore( 0 );

			myscore.FileScore( (*(flist.begin())).file_idnt );
			myscore.Update();
		}
	}
	::printf( "\n" );

	// Loop through all the authors and try to fix their scores
	::printf( "Updating scores for authors\n" );
	for	( aidnt = 1; aidnt < auth.NextIdnt( db ); aidnt++ ) {
		w = "";
		w << "auth_idnt=" << aidnt;
		alist = auth.Select( db, w );
		if	( alist.size() > 0 ) {
			::printf( "%d ", aidnt );

			CFileScore	myscore( 0 );

			myscore.AuthorScore( (*(alist.begin())).auth_idnt );
			myscore.Update();

			adata = myscore.AuthData();

			::printf( 	"%d %d %d %d %d\n",
						adata.auth_rate,
						adata.auth_rat0,
						adata.auth_rat1,
						adata.auth_rat2,
						adata.auth_dlct );
		}
	}
}

// --------------------------------------------------------------------
// local:	Fix author names on all packages
// --------------------------------------------------------------------
static	void	__fixauth	( void ) {
	CMySqlConnect	db( "quest", "", "UTCMS" );
	CMySqlWhere		w;
	CTableAuth		auth;
	CTableFile		file;
	CTablePack		pack;
	data_auth_tl	alist;
	data_file_tl	flist;
	data_file_t		fdata;
	data_pack_tl	plist;
	dword_t			fidnt;
	dword_t			pidnt;


	// Clean references on files
	::printf( "Clearing references on file table\n" );
	for	( fidnt = 0; fidnt < file.NextIdnt( db ); fidnt++ ) {
		w = "";
		w << "file_idnt=" << fidnt;
		flist = file.Select( db, w );
		if	( flist.size() > 0 ) {
			::printf( "\r%d", fidnt );

			fdata = *(flist.begin());
			fdata.file_auth = 0;
			file.Update( db, &fdata );
		}
	}


	// Remove all author records themselves
	::printf( "\nRemoving all author entries\n" );
	auth.Delete( db );


	// Loop through all the files and try to fix their authors
	::printf( "Updating authors for files\n" );
	for	( fidnt = 0; fidnt < file.NextIdnt( db ); fidnt++ ) {
		w = "";
		w << "file_idnt=" << fidnt;
		flist = file.Select( db, w );
		if	( flist.size() > 0 ) {
			fdata = *(flist.begin());
			if	( fdata.file_auth == 0 ) {

				try {
					CAuthor	myauthor( fdata.file_idnt, true );
					if	( myauthor.LongName() ) {
						::printf( "%d ", fdata.file_idnt );
						::printf( " %s.%s - ", fdata.file_name, fdata.file_suff );
						::printf( " %s\n", myauthor.LongName() );
	
						fdata.file_auth = myauthor.AuthorId();
						file.Update( db, &fdata );
					}
				}
				catch ( ... ) {
					::printf( "***WARNING*** Possible malicious file, file_idnt = %d\n", fdata.file_idnt );
				}
			}
		}
	}

	// Loop through all packages and fix their authors as well
	::printf( "Updating authors for packages\n" );
	for	( pidnt = 0; pidnt < pack.NextIdnt( db ); pidnt++ ) {
		w = "";
		w << "pack_idnt=" << pidnt;
		plist = pack.Select( db, w );
		if	( plist.size() > 0 ) {
			::printf( 	"%d %s                                          \r",
						pidnt,
						(*(plist.begin())).pack_file );
			__set_pack_author( pidnt );
		}
	}

	::printf( "\n" );

	// Will also fix their scores
	__fixscore();

}

// --------------------------------------------------------------------
// local:	Fix package names - detect duplicates and rename them
// --------------------------------------------------------------------
static	void	__fixpacknames	( void ) {
	CMySqlConnect	db( "quest", "", "UTCMS" );
	CMySqlWhere		w;
	CMySqlQuote		q;
	CTablePack		pack;
	data_pack_tli	ploop;
	data_pack_tl	plist;
	data_pack_t		pdata;
	data_pack_t		npdata;
	dword_t			version;
	dword_t			pidnt;


	// Loop through all the packs and try to fix their names
	::printf( "Updating names for packages\n" );
	for	( pidnt = 0; pidnt < pack.NextIdnt( db ); pidnt++ ) {

		w = "";
		w << "pack_idnt=" << pidnt;
		plist = pack.Select( db, w );
		if	( plist.size() < 1 )	continue;
		pdata = *(plist.begin());

		::printf( "\r%d", pidnt );

		// Get all packages that share this particular name
		w = "";
		w << "pack_name='" << q.Quote( pdata.pack_name ) << "' order by pack_atim";
		plist = pack.Select( db, w );
		if	( plist.size() < 2 )	continue;

		// Now we have a list with one or more duplicate in it
		// Rename the packages to be pname[n]
		ploop	= plist.begin();

		for	( ploop = plist.begin(); ploop != plist.end(); ploop++ ) {
			pdata = *ploop;

			// Find a non-existent name for the package
			version = 0;
			do	{
				version++;
				w = "";
				w << "pack_name='" << q.Quote( pdata.pack_name ) << "[" << version << "]'";
			}	while	( pack.Count( db, w ) > 0 );

			npdata = pdata;
			::sprintf( npdata.pack_name, "%s[%d]", pdata.pack_name, version );

			pack.Update( db, &npdata );

			::printf( "\r%d renaming %s -> %s\n", pdata.pack_idnt, pdata.pack_name, npdata.pack_name );

		}
	}
}

// --------------------------------------------------------------------
// local:	Do some testing
// --------------------------------------------------------------------
static	void	__do_testing	( void ) {
	

}

// --------------------------------------------------------------------
// public:  Program entrypoint
// --------------------------------------------------------------------
extern  int     main( 	int aAc, char ** aAv ) {
	int	result	= 0;

	try	{
		if		( aAc < 2 )									__help();
		else if	( ! ::strcmp( aAv[1], "cleanonunr" ) )		__cleannonu();
		else if	( ! ::strcmp( aAv[1], "fixtypes" ) )		__fixtypes();
		else if	( ! ::strcmp( aAv[1], "fixpsiz" ) )			__fixpsiz();
		else if	( ! ::strcmp( aAv[1], "fixauth" ) )			__fixauth();
		else if	( ! ::strcmp( aAv[1], "fixscore" ) )		__fixscore();
		else if	( ! ::strcmp( aAv[1], "fixpacknames" ) )	__fixpacknames();
		else if	( ! ::strcmp( aAv[1], "test" ) )			__do_testing();


		else if	( aAc < 3 )								throw CError( aAv[1], "Too few arguments" );
		else if	( ! ::strcmp( aAv[1], "info" ) )		__info	( aAv[2] );
		else if	( ! ::strcmp( aAv[1], "pack" ) )		__pack	( aAv[2] );
		else if	( ! ::strcmp( aAv[1], "del" ) )			__del	( aAv[2] );
		else if	( ! ::strcmp( aAv[1], "era" ) )			__era	( aAv[2] );
		else if	( ! ::strcmp( aAv[1], "erpk" ) ) {
			for	( int arg = 2; arg < aAc; arg++ ) {
				__erpk	( ::atol( aAv[arg] ) );
			}
		}


		else if	( aAc < 4 )								throw CError( aAv[1], "Too few arguments" );
		else if	( ! ::strcmp( aAv[1], "write" ) )		__write( aAc, aAv );
		else if	( ! ::strcmp( aAv[1], "read" ) )		__read( aAc, aAv );
		else if	( ! ::strcmp( aAv[1], "ren" ) )			__ren( aAc, aAv );
		else											throw CError( aAv[1], "Unknown command" );
    }

    catch   ( CError e ) {
		::printf( "Storage: ERROR: %s\n", e.Error() );
        result = -1;
    }

    catch   ( ... ) {
		::printf( "Storage: ERROR: Abort\n" );
        result = -1;
    }

    return  result;
}

// --------------------------------------------------------------------
// EOF: Storage.cxx
// --------------------------------------------------------------------
